//
// File: lib/services/primaryMenuService.js
// Description: This service file contains the logic for interacting with the
// 'primarymenu' table using the Prisma Client for all CRUD operations.
//
import prisma from '@/lib/prismaconfig'; // Assuming you have a central Prisma client configuration.

/**
 * Fetches all primary menu items.
 * @returns {Promise<Array<Object>>} A promise that resolves with an array of menu items.
 */
export const getPrimaryMenuItems = async () => {
  console.log('Fetching all primary menu items from the database');
  return await prisma.primarymenu.findMany();
};

/**
 * Fetches a single primary menu item by its ID.
 * @param {string | number} id The ID of the menu item.
 * @returns {Promise<Object | null>} A promise that resolves with the menu item or null if not found.
 */
export const getPrimaryMenuItemById = async (id) => {
  return await prisma.primarymenu.findUnique({
    where: { pm_pid: Number(id) },
  });
};

/**
 * Creates a new primary menu item.
 * @param {Object} data The data for the new menu item.
 * @returns {Promise<Object>} A promise that resolves with the created menu item.
 */
export const createPrimaryMenuItem = async (data) => {
  return await prisma.primarymenu.create({ data });
};

/**
 * Updates an existing primary menu item.
 * @param {string | number} id The ID of the menu item to update.
 * @param {Object} data The data to update the menu item with.
 * @returns {Promise<Object>} A promise that resolves with the updated menu item.
 */
export const updatePrimaryMenuItem = async (id, data) => {
  return await prisma.primarymenu.update({
    where: { pm_pid: Number(id) },
    data,
  });
};

/**
 * Deletes a primary menu item.
 * @param {string | number} id The ID of the menu item to delete.
 * @returns {Promise<void>} A promise that resolves when the deletion is complete.
 */
export const deletePrimaryMenuItem = async (id) => {
  await prisma.primarymenu.delete({
    where: { pm_pid: Number(id) },
  });
};
