// services/chapterService.js
import prisma from '@/lib/prismaconfig';



export const getAllTasks = async () => {
  return await prisma.acs_task.findMany({
    where: {
      task_isdelete: "active", // 👈 filter out deleted chapter
    },
  });
};

export const getAllStatus = async () => {
  return await prisma.acs_status.findMany({
    where: {
      status_isdelete: "active", // 👈 filter out deleted chapter
      status_pid: {
        in: [1, 2],
      },
    },
  });
};

export const getAllEntryStatus = async () => {
  return await prisma.acs_status.findMany({
    where: {
      status_isdelete: "active", // 👈 filter out deleted chapter
      status_pid: {
        in: [1, 2, 6],
      },
    },
  });
};

export const getAllMDStatus = async () => {
  return await prisma.acs_status.findMany({
    where: {
      status_isdelete: "active", // 👈 filter out deleted chapter
      status_pid: {
        in: [3, 4, 5],
      },
    },
  });
};

export const getAllRCERStatus = async () => {
  return await prisma.acs_status.findMany({
    where: {
      status_isdelete: "active", // 👈 filter out deleted chapter
      status_pid: {
        in: [3, 4, 5],
      },
    },
  });
};

export const getAllRCBTRStatus = async () => {
  return await prisma.acs_status.findMany({
    where: {
      status_isdelete: "active", // 👈 filter out deleted chapter
      status_pid: {
        in: [1, 4],
      },
    },
  });
};

export const getAllEmails = async () => {
  return await prisma.acs_email.findMany({
    where: {
      email_isdelete: "active", // 👈 filter out deleted chapter
    },
  });
};
export const getAllQCEmails = async () => {
  return await prisma.acs_users.findMany({
    where: {
      user_is_active: "active", // 👈 filter out deleted chapter
      user_role: 5, // 👈 filter out deleted chapter
    },
  });
};
// Create new entry
export const createEntry = async (data) => {
  const entry = await prisma.acs_entry.create({
    data,
  });
  // console.log(entry);
  return entry;
};
export const createStage = async (data) => {
  return await prisma.acs_stage_movements.create({
    data,
  });
};
// Create pending new entry
export const createPendingEntry = async (data) => {
  const pending = await prisma.acs_pending_resume.create({
    data,
  });
  return pending;
};

export const updateResumeEntry = async (pr_pid, data) => {
  return await prisma.acs_pending_resume.update({
    where: { pr_pid: Number(pr_pid) },
    data,
  });
};

// Get entry by ID
export const getPendingEntryById = async (pr_pid) => {
  return await prisma.acs_pending_resume.findUnique({
    where: { pr_pid: Number(pr_pid) },
  });
};

// Get entry by ID
export const getEntryById = async (entry_pid) => {
  return await prisma.acs_entry.findUnique({
    where: { entry_pid: Number(entry_pid) },
  });
};

export const getEntryPendingStatus = async () => {
  try {
    // 1. Get the current date and format it as 'YYYY-MM-DD'
    // This is done in JavaScript, not in the raw SQL query.
    const today = new Date();
    const formattedDate = today.toISOString().split('T')[0];

    // 2. Use a parameterized query with `$queryRaw`
    // Pass the formattedDate variable as a parameter using the `${}` syntax.
    const entries = await prisma.$queryRaw`
      SELECT
        l.entry_pid,
        l.entry_status,
        l.entry_date,
        U.user_empid
      FROM acs_entry AS l
      INNER JOIN acs_users AS U
        ON U.user_pid = l.entry_createdBy
      WHERE l.entry_date = ${formattedDate}
        AND l.entry_status IS NULL;
    `;

    // Return an object with a boolean indicating presence of data and the data itself
    return {
      hasData: entries.length > 0,
      data: entries
    };
  } catch (error) {
    console.error("Error fetching entries:", error);
    throw error;
  }
};



// Get entry by ID
export const getEmailById = async (email_pid) => {
  return await prisma.acs_email.findUnique({
    where: { email_pid: Number(email_pid) },
  });
};

export const getQCEmailById = async (qcuserPID) => {
  return await prisma.acs_users.findUnique({
    where: { user_pid: Number(qcuserPID) },
  });
};
// Update entry
export const updateEntry = async (entry_pid, data) => {
  const entry = await prisma.acs_entry.update({
    where: { entry_pid: Number(entry_pid) },
    data,
  });
  return entry;
};



// export const getAllCompletedEntrys = async (userId) => {
//   return await prisma.acs_entry.findMany({
//     include: {
//       book: {
//         select: {
//           book_title: true,
//         },
//       },
//       chapter: {
//         select: {
//           chap_title: true,
//         },
//       },
//       task: {
//         select: {
//           task_name: true,
//         },
//       },
//       status: {
//         select: {
//           status_name: true,
//         },
//       },
//       pending:{
//          select: {
//           status_name: true,
//         },
//       },
//     },
//     where: {
//       entry_isdelete: "active",
//       entry_createdBy : userId,
//       entry_status: {
//         not: null,
//       },
//     },
//   });
// };

export const getAllCompletedEntrys = async (userId) => {
  // Fetch entries
  const entries = await prisma.acs_entry.findMany({
    include: {
      user: {
        select: {
          user_name: true,
        },
      },
      book: {
        select: {
          book_title: true,
        },
      },
      chapter: {
        select: {
          chap_title: true,
        },
      },
      task: {
        select: {
          task_name: true,
        },
      },
      status: {
        select: {
          status_name: true,
        },
      },
      qcUser: {
        select: {
          user_email: true,
        },
      },
      pending: {
        select: {
          pr_duration: true, // Assuming duration is stored as "HH:MM:SS"
        },
      },
    },
    where: {
      entry_isdelete: "active",
      entry_createdBy: userId,
      entry_status: { not: null },
    },
    orderBy: {
      entry_pid: 'desc', // Assumes latest record by creation date
    },
  });

  // Helper: convert "HH:MM:SS" → seconds
  const timeToSeconds = (timeStr) => {
    if (!timeStr) return 0;
    const [h, m, s] = timeStr.split(":").map(Number);
    return h * 3600 + m * 60 + s;
  };

  // Helper: convert seconds → "HH:MM:SS"
  const secondsToTime = (totalSeconds) => {
    const hours = Math.floor(totalSeconds / 3600);
    const minutes = Math.floor((totalSeconds % 3600) / 60);
    const seconds = totalSeconds % 60;
    return `${String(hours).padStart(2, "0")}:${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}`;
  };

  // Compute total pending duration for each entry
  const result = entries.map((entry) => {
    const totalPendingSeconds = entry.pending.reduce((acc, p) => acc + timeToSeconds(p.pr_duration), 0);
    return {
      ...entry,
      total_pending_duration: secondsToTime(totalPendingSeconds),
    };
  });

  return result;
};

export const getAllUserCompletedEntrys = async (userId) => {
  // Fetch entries
  const entries = await prisma.acs_entry.findMany({
    include: {
      user: {
        select: {
          user_name: true,
        },
      },
      book: {
        select: {
          book_title: true,
        },
      },
      chapter: {
        select: {
          chap_title: true,
        },
      },
      task: {
        select: {
          task_name: true,
        },
      },
      status: {
        select: {
          status_name: true,
        },
      },
      // email: {
      //   select: {
      //     email_name: true,
      //   },
      // },
      qcUser: {
        select: {
          user_email: true,
        },
      },
      pending: {
        select: {
          pr_duration: true, // Assuming duration is stored as "HH:MM:SS"
        },
      },
    },
    where: {
      entry_isdelete: "active",
      entry_status: { not: null },
      NOT: { entry_createdBy: userId },
    },
     orderBy: {
      entry_pid: 'desc', // Assumes latest record by creation date
    },
  });

  // Helper: convert "HH:MM:SS" → seconds
  const timeToSeconds = (timeStr) => {
    if (!timeStr) return 0;
    const [h, m, s] = timeStr.split(":").map(Number);
    return h * 3600 + m * 60 + s;
  };

  // Helper: convert seconds → "HH:MM:SS"
  const secondsToTime = (totalSeconds) => {
    const hours = Math.floor(totalSeconds / 3600);
    const minutes = Math.floor((totalSeconds % 3600) / 60);
    const seconds = totalSeconds % 60;
    return `${String(hours).padStart(2, "0")}:${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}`;
  };

  // Compute total pending duration for each entry
  const result = entries.map((entry) => {
    const totalPendingSeconds = entry.pending.reduce((acc, p) => acc + timeToSeconds(p.pr_duration), 0);
    return {
      ...entry,
      total_pending_duration: secondsToTime(totalPendingSeconds),
    };
  });

  return result;
};


export const getAllPendingEntrys = async (userId) => {
  const entries = await prisma.acs_entry.findMany({
    include: {
      user: {
        select: {
          user_name: true,
        },
      },
      book: {
        select: {
          book_title: true,
        },
      },
      chapter: {
        select: {
          chap_title: true,
        },
      },
      task: {
        select: {
          task_name: true,
        },
      },
      status: {
        select: {
          status_name: true,
        },
      },
    },
    where: {
      entry_isdelete: "active",
      entry_status: null,
      entry_createdBy: userId
    },
     orderBy: {
      entry_pid: 'desc', // Assumes latest record by creation date
    },
  });

  // Count only pending_resume with null/empty pr_resumeTime
  const entriesWithResumeCount = await Promise.all(
    entries.map(async (entry) => {
      const count = await prisma.acs_pending_resume.count({
        where: {
          pr_entry_pid: entry.entry_pid,
          pr_isdelete: "active", // Optional: exclude deleted
          OR: [
            { pr_resumeTime: null },
            { pr_resumeTime: "" }, // optional if empty string is used
          ],
        },
      });

      return {
        ...entry,
        pending_resume_count: count, // always a number (0 if none)
      };
    })
  );

  return entriesWithResumeCount;
};

export const getAllUserPendingEntrys = async (userId) => {
  const entries = await prisma.acs_entry.findMany({
    include: {
      user: { 
        select: { 
          user_name: true 
        } 
      },
      book: { 
        select: { 
          book_title: true
         } 
      },
      chapter: { 
        select: { 
          chap_title: true 
        } 
      },
      task: {
         select: { 
          task_name: true 
        }
       },
      status: { 
        select: { 
          status_name: true 
        } 
      },
    },
    where: {
      entry_isdelete: "active",
      entry_status: null,
      NOT: { entry_createdBy: userId },
    },
    orderBy: {
      entry_pid: "desc",
    },
  });

  // Filter only entries having at least one pending_resume with null/empty pr_resumeTime
  const filteredEntries = [];

  for (const entry of entries) {
    const pendingCount = await prisma.acs_pending_resume.count({
      where: {
        pr_entry_pid: entry.entry_pid,
        pr_isdelete: "active",
        OR: [
          { pr_resumeTime: null },
          { pr_resumeTime: "" },
        ],
      },
    });

    if (pendingCount > 0) {
      filteredEntries.push({
        ...entry,
        pending_resume_count: pendingCount,
      });
    }
  }

  return filteredEntries;
};

export const getAllUserInprogressEntrys = async (userId) => {
  const entries = await prisma.acs_entry.findMany({
    include: {
      user: { 
        select: { 
          user_name: true 
        } 
      },
      book: { 
        select: { 
          book_title: true
         } 
      },
      chapter: { 
        select: { 
          chap_title: true 
        } 
      },
      task: {
         select: { 
          task_name: true 
        }
       },
      status: { 
        select: { 
          status_name: true 
        } 
      },
    },
    where: {
      entry_isdelete: "active",
      entry_status: null,
      NOT: { entry_createdBy: userId },
    },
    orderBy: {
      entry_pid: "desc",
    },
  });

  // Filter only entries having at least one pending_resume with null/empty pr_resumeTime
  const filteredEntries = [];

  for (const entry of entries) {
    const pendingCount = await prisma.acs_pending_resume.count({
      where: {
        pr_entry_pid: entry.entry_pid,
        pr_isdelete: "active",
        OR: [
          { pr_resumeTime: null },
          { pr_resumeTime: "" },
        ],
      },
    });

    if (pendingCount == 0) {
      filteredEntries.push({
        ...entry,
        pending_resume_count: pendingCount,
      });
    }
  }

  return filteredEntries;
};

export const getPendingById = async (entry_pid) => {
  return await prisma.acs_pending_resume.findFirst({
    where: {
      pr_entry_pid: Number(entry_pid),
      pr_resumeDate: null,
      pr_resumeTime: null
    },
    orderBy: {
      pr_pid: 'desc'
    },
  });
};

export const fetchPendingEntries = async (entry_pid) => {
  return await prisma.acs_pending_resume.findMany({
    where: {
      pr_entry_pid: Number(entry_pid),
      pr_isdelete: "active", // 👈 filter out deleted chapter
    },
  });
};



export const getPendingDurationById = async (entry_pid) => {
  // Fetch all pending resume entries for the given entry ID
  const pendingData = await prisma.acs_pending_resume.findMany({
    where: { pr_entry_pid: Number(entry_pid) },
  });

  let totalSeconds = 0;

  // Sum durations in seconds
  pendingData.forEach((item) => {
    if (!item.pr_duration) return;
    const [hours, minutes, seconds] = item.pr_duration.split(":").map(Number);
    totalSeconds += hours * 3600 + minutes * 60 + seconds;
  });

  // Convert totalSeconds back to HH:MM:SS
  const hours = Math.floor(totalSeconds / 3600);
  const minutes = Math.floor((totalSeconds % 3600) / 60);
  const seconds = totalSeconds % 60;

  return (
    String(hours).padStart(2, "0") + ":" + String(minutes).padStart(2, "0") + ":" + String(seconds).padStart(2, "0")
  );
};


export const getBookEntryId = async (book_pid) => {
  return await prisma.acs_chapters.count({
    where: { chap_bookTitle: Number(book_pid) },
  });
};

export const getChapterEntryId = async (chapter_pid) => {
  return await prisma.acs_entry.count({
    where: { entry_chapterId: Number(chapter_pid) },
  });
};

export const getAllEditEntrys = async () => {
  // Fetch entries
  const entries = await prisma.acs_entry.findMany({
    include: {
      user: {
        select: {
          user_name: true,
        },
      },
      book: {
        select: {
          book_title: true,
        },
      },
      chapter: {
        select: {
          chap_title: true,
        },
      },
      task: {
        select: {
          task_name: true,
        },
      },
      status: {
        select: {
          status_name: true,
        },
      },
      // email: {
      //   select: {
      //     email_name: true,
      //   },
      // },
      qcUser: {
        select: {
          user_email: true,
        },
      },
      pending: true,
    },
    where: {
      entry_isdelete: "active",
    },
     orderBy: {
      entry_pid: 'desc', // Assumes latest record by creation date
    },
  });

  // Helper: convert "HH:MM:SS" → seconds
  const timeToSeconds = (timeStr) => {
    if (!timeStr) return 0;
    const [h, m, s] = timeStr.split(":").map(Number);
    return h * 3600 + m * 60 + s;
  };

  // Helper: convert seconds → "HH:MM:SS"
  const secondsToTime = (totalSeconds) => {
    const hours = Math.floor(totalSeconds / 3600);
    const minutes = Math.floor((totalSeconds % 3600) / 60);
    const seconds = totalSeconds % 60;
    return `${String(hours).padStart(2, "0")}:${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}`;
  };

  // Compute total pending duration for each entry
  const result = entries.map((entry) => {
    const totalPendingSeconds = entry.pending.reduce((acc, p) => acc + timeToSeconds(p.pr_duration), 0);
    return {
      ...entry,
      total_pending_duration: secondsToTime(totalPendingSeconds),
    };
  });

  return result;
};

/* export const bookReportDetails = async (payload) => {
  const { book_pid, start_date, end_date } = payload;
  // Base SQL
  let query = `
    SELECT
      b.book_title,
      t.task_name,
      TIME_FORMAT(SEC_TO_TIME(SUM(TIME_TO_SEC(e.entry_duration))), '%H:%i:%S') AS totalTime,
      COUNT(e.entry_taskId) AS task_count,
      TIME_FORMAT(
        SEC_TO_TIME(SUM(TIME_TO_SEC(e.entry_duration)) / COUNT(e.entry_taskId)),
        '%H:%i:%S'
      ) AS avg_time_per_task
    FROM acs_entry AS e
    LEFT JOIN acs_books AS b ON b.book_pid = e.entry_bookId
    LEFT JOIN acs_task AS t ON t.task_pid = e.entry_taskId
    WHERE e.entry_isdelete = 'Active'
  `;

  // 🧠 Build dynamic conditions safely
  const conditions = [];

  if (book_pid) {
    conditions.push(`e.entry_bookId = ${book_pid}`);
  }

  if (start_date && end_date) {
    conditions.push(`DATE(e.entry_end_date) BETWEEN '${start_date}' AND '${end_date}'`);
  } else if (start_date) {
    conditions.push(`DATE(e.entry_end_date) = '${start_date}'`);
  } else if (end_date) {
    conditions.push(`DATE(e.entry_end_date) = '${end_date}'`);
  }

  // If no filters selected, use today by default
  if (conditions.length === 0) {
    conditions.push(`DATE(e.entry_end_date) = CURDATE()`);
  }

  // Add all conditions dynamically
  query += ` AND ${conditions.join(' AND ')}`;

  // Grouping
  query += `
    GROUP BY e.entry_bookId, e.entry_taskId
  `;

  // Execute the query
  return await prisma.$queryRawUnsafe(query);
}; */


export const bookReportDetails = async (payload) => {
  const { book_pid, start_date, end_date } = payload;
 
  // Build dynamic where conditions
  const where = { entry_isdelete: 'active' };
 
  if (book_pid) {
    where.entry_bookId = Number(book_pid);
  }
 
  if (start_date && end_date) {
    where.entry_end_date = {
      gte: start_date,
      lte: end_date,
    };
  } else {
     return [];
  }
 
  // Fetch entries
  const entries = await prisma.acs_entry.findMany({
    where,
    select: {
      entry_bookId: true,
      entry_taskId: true,
      entry_duration: true, // string field
    },
  });
 
   // Helper function to convert time string (HH:MM:SS) into seconds
  const convertTimeToSeconds = (time) => {
    const [hours, minutes, seconds] = time.split(':').map(Number);
    return (hours * 3600) + (minutes * 60) + (seconds || 0);
  };
 
  // Group by entry_createdBy and entry_taskId
  const grouped = {};
 
  for (const entry of entries) {
    const key = `${entry.entry_bookId}-${entry.entry_taskId}`;
    if (!grouped[key]) {
      grouped[key] = {
        entry_bookId: entry.entry_bookId,
        entry_taskId: entry.entry_taskId,
        total_duration: 0,
        count: 0,
      };
    }
 
    const durationInSeconds = convertTimeToSeconds(entry.entry_duration);
    grouped[key].total_duration += durationInSeconds || 0; // Ensure valid number
    grouped[key].count += 1;
  }
 
  const groupedArray = Object.values(grouped);
 
  // Fetch user and task names
  const bookIds = [...new Set(groupedArray.map((e) => e.entry_bookId))];
  const taskIds = [...new Set(groupedArray.map((e) => e.entry_taskId))];
 
  const books = await prisma.acs_books.findMany({
    where: { book_pid: { in: bookIds } },
    select: { book_pid: true, book_title: true },
  });
 
  const tasks = await prisma.acs_task.findMany({
    where: { task_pid: { in: taskIds } },
    select: { task_pid: true, task_name: true },
  });
 
  const booksMap = Object.fromEntries(books.map((u) => [u.book_pid, u.book_title]));
  const taskMap = Object.fromEntries(tasks.map((t) => [t.task_pid, t.task_name]));

  // Format results
  const formatTime = (seconds) => {
    const h = Math.floor(seconds / 3600).toString().padStart(2, '0');
    const m = Math.floor((seconds % 3600) / 60).toString().padStart(2, '0');
    const s = Math.floor(seconds % 60).toString().padStart(2, '0');
    return `${h}:${m}:${s}`;
  };
 
  const result = groupedArray.map((g) => {
    return {
      task_name: taskMap[g.entry_taskId] || 'Unknown Task',
      book_title: booksMap[g.entry_bookId] || 'Unknown User',
      totalTime: formatTime(g.total_duration),
      task_count: g.count,
      avg_time_per_task: formatTime(g.total_duration / (g.count || 1)),
    };
  });
  return result;
}; 

export const bookallReportDetails = async (payload) => {
  const { book_pid, start_date, end_date } = payload;

  // Build dynamic where conditions
  const where = { entry_isdelete: 'active' };

  if (book_pid) {
    where.entry_bookId = Number(book_pid);
  }

  if (start_date && end_date) {
    where.entry_end_date = {
      gte: start_date,
      lte: end_date,
    };
  } else {
     return [];
  }

  // Fetch entries with book and task details directly
  const entries = await prisma.acs_entry.findMany({
    where,
    select: {
      entry_duration: true,
      entry_bookId: true,
      entry_taskId: true,
      book: {
        select: { book_title: true },
      },
      task: {
        select: { task_name: true },
      },
    },
  });

  // Directly format result per record
  const result = entries.map((entry) => ({
    book_title: entry.book?.book_title || 'Unknown Book',
    task_name: entry.task?.task_name || 'Unknown Task',
    entry_duration: entry.entry_duration || '00:00:00',
    task_count: '1',
  }));

  return result;
};














