// services/ceReassignService.js
import prisma from "../prismaconfig.js";
import { sendEmailCopyEditReAssign } from "./emailService.js";

export async function processCeReassignments() {
  const today = new Date().toISOString().split("T")[0];
  console.log(`🕒 Running CE reassignment job for: ${today}`);

  try {
    // Step 1: Fetch due entries
    const dueEntries = await prisma.acs_ce_editing_details.findMany({
      where: {
        ce_due_date: today,
        ce_status: 1,
      },
      select: {
        ce_pid: true,
        ce_chap_id: true,
        ce_editor_id: true,
        chapter: {
          select: { chap_title: true },
        },
        user: {
          select: { user_name: true, user_email: true },
        },
      },
    });

    if (!dueEntries || dueEntries.length === 0) {
      console.log("✅ No CE records due today.");
      return;
    }

    const timestamp = new Date().toLocaleString("en-IN", {
      timeZone: "Asia/Kolkata",
    });

    // Step 2: Loop through each record
    for (const entry of dueEntries) {
      const { ce_pid, chapter, user } = entry;
      const chap_title = chapter?.chap_title || "Untitled Chapter";
      const copy_editor_name = user?.user_name || "Editor";
      const editor_email =
        user?.user_email || process.env.AUTH_USER_EMAIL || "";

      console.log(
        `🔄 Processing CE reassignment for ${copy_editor_name} (${editor_email}) | Chapter: "${chap_title}"`
      );

      // Skip if no email found
      if (!editor_email) {
        console.warn(
          `⚠️ No valid email found for ${copy_editor_name}, skipping this entry.`
        );
        continue;
      }

      // Step 3: Prepare mail details
      const subject = `${chap_title} has been reassigned`;
      const message = `
        <div style="font-family: Calibri, sans-serif; font-size: 14px; color: black;">
          Dear ${copy_editor_name},<br><br>
          Because you have not accepted this file for copyedit, it will now be assigned to another copyeditor.<br><br>
          No further action is required on your part.<br><br>
          Best Regards,<br>
          Ebooks Copyedit Team
        </div>
      `;

      const ccList = ['anibillgates@pdmrindia.com'];
      const bccList = [];
      const attachments = [];

      try {
        // Step 4: Send mail
        const mailResponse = await sendEmailCopyEditReAssign(
          editor_email,
          subject,
          message,
          ccList,
          bccList,
          attachments
        );

        if (mailResponse.success) {
          // Step 5: Update DB after successful mail
          await prisma.acs_ce_editing_details.update({
            where: { ce_pid },
            data: {
              ce_status: 4,
              ce_timeout_date: timestamp,
            },
          });

          console.log(
            `📨 Email sent successfully & CE updated for ${copy_editor_name} (Chapter: "${chap_title}")`
          );
        } else {
          console.error(
            `❌ Failed to send email to ${copy_editor_name}: ${mailResponse.error || "Unknown error"}`
          );
        }
      } catch (error) {
        console.error(
          `🚨 Unexpected error while sending mail to ${copy_editor_name}:`,
          error.message
        );
      }
    }

    console.log("✅ CE reassignment processing completed.");
  } catch (error) {
    console.error("🚨 Fatal error during CE reassignment process:", error);
  }
}
