// app/users/create/actions.js
'use server';

import { createUser, updateUser, getUserByEmpid, getUserByEmail } from '@/lib/services/userService';
import { genPassword } from "@/lib/utils/authUtils";
import fs from "fs";
import path from "path";

export async function createUserAction(formData) {
  try {
    const user_empid = formData.get("user_empid");
    const user_name = formData.get("user_name");
    const user_email = formData.get("user_email");
    const user_pid = parseInt(formData.get("user_pid"));
    const user_role = parseInt(formData.get("user_role"));
    const user_password = formData.get("user_password");
    const user_con_pwd = formData.get("user_con_pwd");

    if (!user_empid || !user_name || !user_email || !user_role || !user_password || !user_con_pwd) {
      return {
        success: false,
        message: "Missing required fields",
      };
    }

    if (user_password !== user_con_pwd) {
      return {
        success: false,
        message: "Passwords do not match",
      };
    }

    const existingEmpid = await getUserByEmpid(user_empid);
    if (existingEmpid) {
      return {
        success: false,
        message: "Employee ID already exists",
      };
    }

    const existingUser = await getUserByEmail(user_email);
    if (existingUser) {
      return {
        success: false,
        message: "Email ID already exists",
      };
    }
    let profilePicturePath = null;
    const file = formData.get("user_profile");

    if (file && file.name) {
      const arrayBuffer = await file.arrayBuffer();
      const buffer = Buffer.from(arrayBuffer);

      // get extension from uploaded file
      const ext = path.extname(file.name);
      const newFilename = `${user_empid}${ext}`;
      const newPath = path.join(process.cwd(), "public", "emp_profile", newFilename);

      // save file
      fs.writeFileSync(newPath, buffer);

      // only save filename in DB
      profilePicturePath = newFilename;
    }
    const user_hash_password = await genPassword(user_password);
    if (!user_hash_password) {
      return {
        success: false,
        message: "Password hashing failed",
      };
    }
    const userData = {
      user_empid,
      user_name,
      user_email,
      user_hash_password,
      user_role,
      user_createdBy : user_pid,
      user_profile: profilePicturePath, // just "1885.jpg"
    };
    await createUser(userData);
    return {
      success: true,
      message: "User Created Successfully",
      userData: userData,
    };
  } catch (error) {
    console.error("Error creating user details:", error);
    return {
      success: false,
      message: "Server error during user details"
    };
  }
}

export async function updateUserAction(userId, formData) {
  try {
    const user_empid = formData.get("user_empid");
    const user_name = formData.get("user_name");
    const user_email = formData.get("user_email");
    const user_role = parseInt(formData.get("user_role"));

    if (!user_empid || !user_name || !user_email || !user_role) {
      return {
        success: false,
        message: "Missing required fields",
      };
    }

    // Check for existing employee ID, excluding the current user
    const existingEmpid = await getUserByEmpid(user_empid);
    if (existingEmpid && existingEmpid.user_pid !== Number(userId)) {
      return {
        success: false,
        message: "Employee ID already exists",
      };
    }

    // Check for existing email, excluding the current user
    const existingUser = await getUserByEmail(user_email);
    if (existingUser && existingUser.user_pid !== Number(userId)) {
      return {
        success: false,
        message: "Email ID already exists",
      };
    }
    const userData = {
      user_empid,
      user_name,
      user_email,
      user_role,
    };
    let profilePicturePath = null;
    const file = formData.get("user_profile");

    if (file && file.name) {
      const arrayBuffer = await file.arrayBuffer();
      const buffer = Buffer.from(arrayBuffer);

      // get extension from uploaded file
      const ext = path.extname(file.name);
      const newFilename = `${user_empid}${ext}`;
      const newPath = path.join(process.cwd(), "public", "emp_profile", newFilename);

      // save file
      fs.writeFileSync(newPath, buffer);

      // only save filename in DB
      profilePicturePath = newFilename;
      userData.user_profile = profilePicturePath;
    }
    await updateUser(userId, userData);
    return {
      success: true,
      message: "User Updated Successfully",
      userData: userData,
    };
  } catch (error) {
    console.error("Error creating user details:", error);
    return {
      success: false,
      message: "Server error during user details"
    };
  }
}


export async function updateDeleteAction(userId, payload) {
   try {
    const books = await updateUser(userId, payload);
    return {
      success: true,
      message: "User deleted Successfully",
      books
    };
  } catch (error) {
    console.error("Error updateing user details:", error);
    return {
      success: false,
      message: "Server error during user details"
    };
  }
}