import { writeFile, mkdir } from 'fs/promises';
import { join } from 'path';
import { NextResponse } from 'next/server';

import { 
  updateEntry, 
  getEntryById, 
  createStage,
  getQCEmailById,
  getPendingDurationById,
} from '@/lib/services/entryService';

import {
  getChapterById,
  updateChapter,
} from '@/lib/services/chapterService';

import { sendEmail } from '@/lib/services/emailService';

import moment from 'moment';

export const maxDuration = 600;

const timeToSeconds = (timeStr) => {
  const [h, m, s] = timeStr.split(':').map(Number);
  return h * 3600 + m * 60 + s;
};

const secondsToTime = (totalSeconds) => {
  const hours = Math.floor(totalSeconds / 3600);
  const minutes = Math.floor((totalSeconds % 3600) / 60);
  const seconds = totalSeconds % 60;
  return `${String(hours).padStart(2, '0')}:${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`;
};

export async function POST(request) {
  try {
    console.log('📨 API Route: File upload started');

    const formData = await request.formData();
    const entryId = formData.get('entryId');
    const entry_chapterId = formData.get('entry_chapterId');
    const entry_comments = formData.get('entry_comments');
    const entry_wordcount = formData.get('entry_wordcount');
    const entry_contact_email = formData.get('entry_contact_email');
    const entry_status = formData.get('entry_status');
    const file = formData.get('entry_upload_file');

    if (!entryId) {
      return NextResponse.json(
        { success: false, message: 'Entry ID is required' },
        { status: 400 }
      );
    }

    const entry_data = await getEntryById(entryId);

    if (!entry_data) {
      return NextResponse.json(
        { success: false, message: 'Entry ID not found' },
        { status: 404 }
      );
    }

    // Parse start and end datetime
    const startDateTime = new Date(`${entry_data.entry_date}T${entry_data.entry_start_time}`);
    const currentTime = new Date();
    const endTime = currentTime.toTimeString().split(" ")[0];
    const endDate = currentTime.toISOString().split("T")[0];
    const endDateTime = new Date(`${endDate}T${endTime}`);

    const durationMs = endDateTime - startDateTime;

    if (durationMs < 0) {
      return { success: false, message: "End time cannot be before start time" };
    }

    // Format duration as HH:MM:SS
    const totalSeconds = Math.floor(durationMs / 1000);
    const hours = Math.floor(totalSeconds / 3600);
    const minutes = Math.floor((totalSeconds % 3600) / 60);
    const seconds = totalSeconds % 60;

    let duration = `${hours.toString().padStart(2, "0")}:${minutes
      .toString()
      .padStart(2, "0")}:${seconds.toString().padStart(2, "0")}`;

    const pending_data = await getPendingDurationById(entryId);
    //  console.log(pending_data);
    //  console.log(duration);
    if (pending_data) {
      const durationSeconds = timeToSeconds(duration);
      const pendingSeconds = timeToSeconds(pending_data);
      const remainingSeconds = Math.max(0, durationSeconds - pendingSeconds);
      duration = secondsToTime(remainingSeconds);
    }


    // ✅ CORRECT: Prepare data with proper string values
    const updatedData = {
      entry_comments,
      entry_wordcount: Number(entry_wordcount),
      entry_email: Number(entry_contact_email),
      entry_status: Number(entry_status),
      entry_end_time: endTime,     // ✅ String: "HH:MM:SS"
      entry_end_date: endDate,     // ✅ String: "YYYY-MM-DD"
      entry_duration: duration,   // ✅ String: "HH:MM:SS"
    };

    console.log('📝 Data to update:', {
      entry_end_time: endTime,
      entry_end_date: endDate,
      entry_duration: duration,
    });
    // return

    // Handle file upload
    let uploadedFileName = null;
    if (file && file.size > 0) {
      console.log(`📁 Uploading file: ${file.name} (${(file.size / 1024 / 1024).toFixed(2)} MB)`);

      try {
        const arrayBuffer = await file.arrayBuffer();
        const buffer = Buffer.from(arrayBuffer);

        const MAX_FILE_SIZE = 1 * 1024 * 1024 * 1024;
        if (buffer.length > MAX_FILE_SIZE) {
          return NextResponse.json(
            {
              success: false,
              message: `File size exceeds maximum limit of ${MAX_FILE_SIZE / 1024 / 1024 / 1024} GB`,
            },
            { status: 400 }
          );
        }

        const timestamp = Date.now();
        const newFilename = `${timestamp}_${file.name}`;
        const uploadDir = join(process.cwd(), 'public', 'First_check');

        try {
          await mkdir(uploadDir, { recursive: true });
        } catch (err) {
          console.error('❌ Error creating directory:', err);
        }

        const newPath = join(uploadDir, newFilename);
        await writeFile(newPath, buffer);
        uploadedFileName = newFilename;

        console.log(`✅ File saved: ${newFilename} (${(buffer.length / 1024 / 1024).toFixed(2)} MB)`);
      } catch (fileError) {
        console.error('❌ File upload error:', fileError);
        return NextResponse.json(
          {
            success: false,
            message: `File upload failed: ${fileError.message}`,
          },
          { status: 500 }
        );
      }
    }

    if (uploadedFileName) {
      updatedData.entry_file = uploadedFileName;
    }

    await updateEntry(entryId, updatedData);

    const updateDatawordcount = {
      chap_wordcount: Number(entry_wordcount),
    };

    if (entry_contact_email) {
      const updatedDatacontact_email = {
        chap_contact_qc_user: Number(entry_contact_email),
      };
      await updateChapter(entry_chapterId, updatedDatacontact_email);
    }

    await updateChapter(entry_chapterId, updateDatawordcount);

    const entry_data_res = await getEntryById(entryId);
    const chapterDetails = await getChapterById(entry_data_res.entry_chapterId);

    let from_stage = '';
    let to_stage = '';

    if (entry_data_res.entry_taskId == 1 && entry_status == 1) {
      from_stage = 'FC';
      to_stage = 'FCQC';
    } else if (entry_data_res.entry_taskId == 1 && entry_status == 2) {
      from_stage = 'FC';
      to_stage = 'FCR';
    } else if (entry_data_res.entry_taskId == 2 && entry_status == 1) {
      from_stage = 'FCR';
      to_stage = 'FCQC';
    } else if (entry_data_res.entry_taskId == 2 && entry_status == 2) {
      from_stage = 'FCR';
      to_stage = 'FCR';
    } else if (entry_data_res.entry_taskId == 3 && entry_status == 6) {
      from_stage = 'RCR1';
      to_stage = 'RCR1';
    } else if (entry_data_res.entry_taskId == 4 && entry_status == 6) {
      from_stage = 'RCR2';
      to_stage = 'RCR2';
    } else if (entry_data_res.entry_taskId == 5 && entry_status == 6) {
      from_stage = 'RCR3';
      to_stage = 'RCR3';
    } else if (
      entry_data_res.entry_taskId == 3 &&
      chapterDetails.chap_copyEditing === 'yes' &&
      entry_status == 1
    ) {
      from_stage = 'RCR1';
      to_stage = 'CE';
    } else if (entry_data_res.entry_taskId == 3 && entry_status == 1) {
      from_stage = 'RCR1';
      to_stage = 'ER';
    } else if (entry_data_res.entry_taskId == 4 && entry_status == 1) {
      from_stage = 'RCR2';
      to_stage = 'ER';
    } else if (entry_data_res.entry_taskId == 5 && entry_status == 1) {
      from_stage = 'RCR3';
      to_stage = 'ER';
    }

    const now = new Date();
    const stageData = {
      stage_bookId: Number(entry_data_res.entry_bookId),
      stage_chapId: Number(entry_data_res.entry_chapterId),
      stage_entryId: Number(entryId),
      stage_notes: entry_comments,
      stage_uploadFile: uploadedFileName,
      stage_status: Number(entry_status),
    //   stage_dispachDate: now.toISOString().split('T'),
    stage_dispachDate: now.toISOString().split("T")[0],
      stage_from: from_stage,
      stage_to: to_stage,
      stage_createdBy: entry_data_res.entry_createdBy,
    };


    await createStage(stageData);

    const chapterData = {
      chap_currentStage: to_stage,
      chap_copyEditingStatus: 1,
    };

    await updateChapter(entry_data_res.entry_chapterId, chapterData);

    if ((entry_data_res.entry_taskId == 1 || entry_data_res.entry_taskId == 2) && entry_status == 1) {
      try {
        const email_data = await getQCEmailById(entry_contact_email);
        if (email_data) {
          const to = email_data.user_email;
          const subject = `New Chapter ${chapterDetails?.chap_title} Moved for QC Review`;
          const formattedDate = moment().format('DD MMMM YYYY');

          const text = `Hi ${email_data?.user_name},\n\nNew chapter ${chapterDetails?.chap_title} has been moved for QC Review.\n\nBest Regards,\n\nACS Team\n\nDate: ${formattedDate}`;

          const cc = [];
          const bcc = process.env.QC_BCCEMAILS
            ? process.env.QC_BCCEMAILS.split(',').map((email) => email.trim())
            : [];

          await sendEmail(to, subject, text, cc, bcc);
        }
      } catch (error) {
        console.error('⚠️ Warning: Email sending failed:', error);
      }
    }

    console.log('✅ API Route: Success');
    return NextResponse.json({
      success: true,
      message: 'Entry updated successfully',
      file: uploadedFileName,
    });
  } catch (error) {
    console.error('❌ API Error:', error);
    return NextResponse.json(
      {
        success: false,
        message: `Server error: ${error.message}`,
      },
      { status: 500 }
    );
  }
}
