"use client";

import { useEffect, useMemo, useState, Suspense } from "react";
import dynamic from "next/dynamic";
import { useRouter } from "next/navigation";
import Swal from "sweetalert2";
import { toast } from "react-toastify";
import { getUserColumns } from "./columns";
import { updateDeleteAction } from "@/lib/actions/userActions";
import Select from "react-select";
import selectStyles from "@/app/components/selectStyles";
import BooksListShimmer from "@/app/components/BookListShimmer";



// ✅ Lazy load CustomDataTable
const CustomDataTable = dynamic(() => import("../../dataTable/customDataTable"), {
  loading: () => <BooksListShimmer rows={6} columns={5} />, // shimmer placeholder while table loads
  ssr: false,
});

export default function BooksList({ initialUsers = [], roles = [] }) {
  const [data, setData] = useState(initialUsers || []);
  const [filteredData, setFilteredData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedRole, setselectedRole] = useState(null);
  const router = useRouter();

  const roleOptions =
    roles?.map((u) => ({
      value: String(u.role_pid),
      label: u.role_name.charAt(0).toUpperCase() + u.role_name.slice(1),
    })) || [];

  // 🕐 Simulate data fetch / re-render
  useEffect(() => {
    setLoading(true);
    const timer = setTimeout(() => {
      setData(initialUsers || []);
      setFilteredData(initialUsers || []);
      setLoading(false);
    }, 800); // delay for smooth shimmer view
    return () => clearTimeout(timer);
  }, [initialUsers]);

  // 🧩 Filter by role
  useEffect(() => {
    const filtered = data.filter((item) => {
      const matchesRole = selectedRole
        ? Number(item.user_role) === Number(selectedRole.value)
        : true;
      return matchesRole;
    });
    setFilteredData(filtered);
  }, [selectedRole, data]);

  const handleDelete = async (userId, userName) => {
    const result = await Swal.fire({
      title: "Are you sure?",
      text: `Do you really want to delete the user named "${userName}"?`,
      icon: "warning",
      showCancelButton: true,
      confirmButtonColor: "#d33",
      cancelButtonColor: "#3085d6",
      confirmButtonText: "Yes, delete it!",
    });
    if (!result.isConfirmed) return;

    try {
      const payload = { user_is_active: "inactive" };
      const res = await updateDeleteAction(userId, payload);
      if (res?.success) {
        setData((prev) => prev.filter((u) => u.user_pid !== userId));
        toast.success("User has been deleted.");
      } else {
        toast.error(res?.message || "Failed to delete user.");
      }
    } catch (error) {
      toast.error(error.message || "Error deleting user");
    }
  };

  const columns = useMemo(() => getUserColumns(router, handleDelete), [router]);

  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg space-y-4">
      {/* Header */}
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">User Lists</h2>
            <p className="text-gray-500 mt-1 text-sm">
              Below is a list of all users.
            </p>
          </div>
          <button
            onClick={() => router.push("/users/create")}
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
          >
            + New User
          </button>
        </div>
      </header>

      

      {/* ✅ Table or Loader */}
      {loading ? (
        <BooksListShimmer rows={6} columns={5} />
      ) : (
        <Suspense fallback={<BooksListShimmer rows={6} columns={5} />}>
          {/* Filters */}
      <div className="bg-purple-50 border border-purple-200 rounded-lg p-2 shadow-sm">
        <div className="flex flex-wrap gap-4 items-center justify-start">
          {/* Role Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Role
            </label>
            <Select
              instanceId="role-filter"
              options={roleOptions}
              value={selectedRole}
              onChange={setselectedRole}
              placeholder="Select Role"
              isClearable
              className="w-33 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>

          {/* Clear Filters */}
          <div className="pt-5">
            <button
              onClick={() => {
                setselectedRole(null);
                setFilteredData(initialUsers);
              }}
              className="text-sm px-4 py-2 bg-purple-100 hover:bg-purple-200 text-purple-700 rounded transition-all duration-200 cursor-pointer"
            >
              Clear Filters
            </button>
          </div>
        </div>
      </div>
          <div className="animate-fadeIn transition-opacity duration-500">
            <CustomDataTable
              columns={columns}
              data={filteredData}
              progressPending={false}
              defaultSort={{ key: "Working Title", direction: "ascending" }}
            />
          </div>
        </Suspense>
      )}
    </div>
  );
}
