"use client";

import { useState, useRef } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
  FiUploadCloud,
  FiFileText,
  FiAlertCircle,
  FiCheckCircle,
  FiDownload,
  FiLoader,
  FiRefreshCw,
} from "react-icons/fi";
import { Printer, Monitor, Mail } from "lucide-react";

const STATUS = {
  READY: "Ready to upload",
  UPLOADING: "Uploading file...",
  COMPRESSING: "Compressing images...",
  COMPLETE: "Compression complete!",
  FAILED: "Compression failed",
};

export default function UploadForm() {
  const [file, setFile] = useState(null);
  const [status, setStatus] = useState(STATUS.READY);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");
  const [downloadUrl, setDownloadUrl] = useState("");
  const [sizeStats, setSizeStats] = useState(null);
  const [dpi, setDpi] = useState("96");
  const [progress, setProgress] = useState(0);
  const dropRef = useRef(null);

  const compressionOptions = [
    {
      value: "220",
      label: "High Quality",
      desc: "",
      icon: <Printer className="w-6 h-6 text-indigo-600" />,
    },
    {
      value: "150",
      label: "Medium Quality",
      desc: "",
      icon: <Monitor className="w-6 h-6 text-teal-600" />,
    },
    {
      value: "96",
      label: "Low Quality",
      desc: "",
      icon: <Mail className="w-6 h-6 text-rose-600" />,
    },
  ];

  // ---------------- FILE VALIDATION ----------------
  const validateFile = (file) => {
    if (!file) return "Please select a file.";
    const validTypes = [
      "application/msword",
      "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
    ];
    if (!validTypes.includes(file.type))
      return "Only .doc or .docx files are allowed.";
    if (file.size > 1 * 1024 * 1024 * 1024)
      return "File size exceeds 1 GB limit.";
    return null;
  };

  // ---------------- DRAG & DROP ----------------
  const handleDrop = (e) => {
    e.preventDefault();
    if (downloadUrl) return; // disable when done
    dropRef.current.classList.remove("border-blue-400", "bg-blue-50");
    const droppedFile = e.dataTransfer.files[0];
    const validationError = validateFile(droppedFile);
    if (validationError) {
      setError(validationError);
      return;
    }
    setFile(droppedFile);
    setError("");
  };

  const handleDragOver = (e) => {
    e.preventDefault();
    if (downloadUrl) return;
    dropRef.current.classList.add("border-blue-400", "bg-blue-50");
  };

  const handleDragLeave = (e) => {
    e.preventDefault();
    dropRef.current.classList.remove("border-blue-400", "bg-blue-50");
  };

  // ---------------- SUBMIT ----------------
  const handleSubmit = async (e) => {
    e.preventDefault();
    setError("");
    setDownloadUrl("");
    setSizeStats(null);
    setProgress(0);

    const validationError = validateFile(file);
    if (validationError) {
      setError(validationError);
      return;
    }

    setLoading(true);
    setStatus(STATUS.UPLOADING);

    // Simulated progress bar animation (smooth and realistic)
    const interval = setInterval(() => {
      setProgress((prev) => (prev < 95 ? prev + 5 : prev));
    }, 500);

    try {
      const formData = new FormData();
      formData.append("file", file);
      formData.append("dpi", dpi);

      setStatus(STATUS.COMPRESSING);
      const response = await fetch("/settings/file_compression", {
        method: "POST",
        body: formData,
      });

      const result = await response.json();
      clearInterval(interval);
      setLoading(false);
      setProgress(100);

      if (result.success) {
        setStatus(STATUS.COMPLETE);
        setDownloadUrl(result.downloadUrl);
        setSizeStats({
          original: result.originalSize,
          compressed: result.compressedSize,
        });
      } else {
        setStatus(STATUS.FAILED);
        setError(result.message || "Compression failed.");
      }
    } catch (err) {
      clearInterval(interval);
      setLoading(false);
      setStatus(STATUS.FAILED);
      setError("Network or server error.");
    }
  };

  const formatBytes = (bytes) => {
    if (bytes === 0) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
  };

  const fileInputRef = useRef(null);


  // Centralized reset function
const resetUI = () => {
  setDownloadUrl("");
  setFile(null);
  setStatus(STATUS.READY);
  setSizeStats(null);
  setProgress(0);
  setError("");
  setDpi("96"); // reset DPI selection if needed

  // Clear actual file input
  if (fileInputRef.current) {
    fileInputRef.current.value = null;
  }
};


  return (
    <motion.div
      initial={{ opacity: 0, scale: 0.95 }}
      animate={{ opacity: 1, scale: 1 }}
      className="bg-white rounded-2xl shadow-2xl w-full max-w-6xl overflow-hidden border border-gray-200 mx-auto grid grid-cols-1 lg:grid-cols-2 min-h-[600px]"
    >
      {/* LEFT SIDE */}
      <div className="p-8 md:p-10 bg-white flex flex-col justify-start">
        <h1 className="text-3xl font-extrabold text-blue-800 mb-8 border-b pb-4">
          ⚙️ DOCX Optimization
        </h1>

        <form onSubmit={handleSubmit} className="w-full">
          {/* Compression Options */}
          <div className="mb-8">
            <h2 className="text-xl font-bold text-gray-800 mb-4">
              Select Compression Level Size
            </h2>
            <div className="grid gap-3 sm:grid-cols-3">
              {compressionOptions.map((opt) => (
                <motion.label
                  key={opt.value}
                  whileHover={{ scale: 1.03 }}
                  whileTap={{ scale: 0.98 }}
                  className={`relative cursor-pointer rounded-xl p-4 border transition-all duration-300 ${
                    dpi === opt.value
                      ? "border-indigo-500 bg-indigo-50 shadow-md"
                      : "border-gray-200 hover:bg-gray-50"
                  }`}
                >
                  <input
                    type="radio"
                    name="dpi"
                    value={opt.value}
                    checked={dpi === opt.value}
                    onChange={(e) => setDpi(e.target.value)}
                    className="hidden"
                  />
                  <div className="flex flex-col items-center gap-2">
                    {opt.icon}
                    <p className="font-semibold text-gray-800 text-base text-center">
                      {opt.label}
                    </p>
                    <p className="text-xs text-gray-500 text-center">
                      {opt.desc}
                    </p>
                  </div>
                </motion.label>
              ))}
            </div>
          </div>

          {/* Drag & Drop */}
          <div
            ref={dropRef}
            onDrop={handleDrop}
            onDragOver={handleDragOver}
            onDragLeave={handleDragLeave}
            className={`border-2 border-dashed rounded-xl p-8 text-center mb-6 transition-all duration-300 ${
              downloadUrl
                ? "opacity-50 pointer-events-none"
                : file
                ? "border-green-400 bg-green-50"
                : "border-gray-300 hover:border-blue-400 hover:bg-blue-50"
            }`}
          >
            <FiUploadCloud size={40} className="mx-auto text-blue-500 mb-3" />
            <p className="text-gray-500 mb-3 font-medium">
              Drag & drop your **.doc/.docx** file here
            </p>
            <input
              ref={fileInputRef}  // 👈 attach ref here
              type="file"
              accept=".doc,.docx"
              disabled={!!downloadUrl}
              onChange={(e) => {
                const selectedFile = e.target.files[0];
                const validationError = validateFile(selectedFile);
                if (validationError) {
                  setError(validationError);
                  setFile(null);
                  return;
                }
                setFile(selectedFile);
                setError("");
                setStatus(STATUS.READY);
              }}
              className="block w-full border border-gray-300 p-2 rounded-md text-sm cursor-pointer file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
            />
            {file && (
              <motion.div
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                className="flex items-center justify-between mt-4 p-3 bg-white rounded-lg shadow-inner text-sm"
              >
                <div className="flex items-center gap-2 text-gray-700">
                  <FiFileText className="text-indigo-500" />
                  <span className="font-semibold truncate max-w-[200px]">
                    {file.name}
                  </span>
                </div>
                <span className="text-gray-500 font-medium">
                  {formatBytes(file.size)}
                </span>
              </motion.div>
            )}
          </div>

          {/* Error */}
          


          {/* Submit */}
          <button
            type="submit"
            disabled={loading || !file || downloadUrl}
            className={`w-full font-bold py-3 px-4 rounded-lg flex justify-center items-center text-white transition-all duration-300 ${
              loading || !file || downloadUrl
                ? "bg-gray-400 cursor-not-allowed"
                : "bg-blue-600 hover:bg-blue-700 shadow-xl hover:shadow-2xl transform hover:-translate-y-0.5"
            }`}
          >
            {loading ? (
              <>
                <FiLoader className="mr-2 animate-spin" /> Processing...
              </>
            ) : (
              "Start Compression"
            )}
          </button>
        </form>
      </div>

      {/* RIGHT SIDE */}
      <div className="flex flex-col p-8 md:p-10 bg-gradient-to-br from-indigo-700 to-blue-800 text-white justify-between min-h-[600px]">
        {/* STATUS */}
        <div className="mb-10">
          <h2 className="text-3xl font-bold mb-2">Compression Status</h2>
          <p className="text-indigo-200">
            Real-time update on your file processing.
          </p>

          <motion.div
            key={status}
            initial={{ opacity: 0, y: 10 }}
            animate={{ opacity: 1, y: 0 }}
            className={`mt-6 p-4 rounded-xl font-semibold text-lg flex items-center gap-3 ${
              status === STATUS.COMPLETE
                ? "bg-green-600"
                : status === STATUS.FAILED
                ? "bg-red-600"
                : "bg-blue-600"
            }`}
          >
            {status === STATUS.COMPLETE && <FiCheckCircle size={24} />}
            {status === STATUS.FAILED && <FiAlertCircle size={24} />}
            {loading && <FiLoader size={24} className="animate-spin" />}
            {!loading &&
              status !== STATUS.COMPLETE &&
              status !== STATUS.FAILED && <FiUploadCloud size={24} />}
            {status}
          </motion.div>

          {/* Progress Bar */}
          <AnimatePresence>
            {(loading || progress > 0) && (
              <motion.div
                initial={{ width: "0%" }}
                animate={{ width: `${progress}%` }}
                exit={{ opacity: 0 }}
                transition={{ duration: 0.3 }}
                className="h-3 bg-blue-400 rounded-full mt-4 shadow-lg"
              />
            )}
          </AnimatePresence>
        </div>

        {/* RESULT */}
        <div className="flex-grow flex items-center justify-center">
          <AnimatePresence mode="wait">
            {downloadUrl ? (
              <motion.div
                key="download-success"
                initial={{ opacity: 0, scale: 0.8 }}
                animate={{ opacity: 1, scale: 1 }}
                exit={{ opacity: 0, scale: 0.8 }}
                transition={{ duration: 0.5 }}
                className="text-center border-2 border-dashed border-green-300 rounded-2xl p-8 bg-white text-gray-800 w-full max-w-sm"
              >
                <FiCheckCircle
                  size={60}
                  className="mx-auto text-green-500 mb-4"
                />
                <h2 className="text-2xl font-bold mb-2">
                  Compression Complete!
                </h2>
                <p className="text-gray-600 mb-4">Download your optimized file</p>

                {sizeStats && (
                  <div className="bg-green-50 border border-green-200 rounded-xl p-4 text-sm mb-6">
                    <p className="mb-1">
                      Original Size: <strong>{sizeStats.original}</strong>
                    </p>
                    <p>
                      Compressed Size: <strong>{sizeStats.compressed}</strong>
                    </p>
                  </div>
                )}

                <div className="flex flex-col gap-3">

<motion.a
  href={downloadUrl}
  download
  onClick={async () => {
    if (downloadUrl) {
      const fileName = downloadUrl.split("/").pop();

      // Call server to delete file after download
      await fetch("/views/settings/file_compression/upload/actions", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ filename: fileName }),
      });

      // Reset UI
      resetUI();
    }
  }}
  className="inline-flex justify-center items-center gap-2 bg-green-600 hover:bg-green-700 text-white px-6 py-3 rounded-lg font-semibold shadow-lg transition-all duration-300 w-full"
  whileHover={{ scale: 1.05 }}
  whileTap={{ scale: 0.95 }}
>
  <FiDownload size={20} /> Download File
</motion.a>
<motion.button
  onClick={async () => {
    if (downloadUrl) {
      const fileName = downloadUrl.split("/").pop();

      // Delete file before starting new optimization
      await fetch("/views/settings/file_compression/upload/actions", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ filename: fileName }),
      });
    }

    // Reset UI
    resetUI();
  }}
  className="inline-flex justify-center items-center gap-2 bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-lg font-semibold shadow-lg transition-all duration-300 w-full"
  whileHover={{ scale: 1.05 }}
  whileTap={{ scale: 0.95 }}
>
  <FiRefreshCw size={20} /> Start New Optimization
</motion.button>

</div>

              </motion.div>
            ) : (
              <motion.div
                key="placeholder-initial"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                exit={{ opacity: 0 }}
                className="text-center border-2 border-dashed border-indigo-400 rounded-2xl p-10 text-indigo-200"
              >
                <FiDownload size={50} className="mx-auto mb-4" />
                <p className="text-lg">Optimized file link will appear here.</p>
              </motion.div>
            )}
          </AnimatePresence>
        </div>
      </div>
    </motion.div>
  );
}
