"use client";

import { useMemo, useState, useRef } from "react";
import Select from "react-select";
import { createBookReportAction,allBookReportList } from "@/lib/actions/entryActions"; // your API/service function
import * as XLSX from "xlsx";
import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";
import Papa from "papaparse";
import Swal from "sweetalert2";
import ChartDataLabels from 'chartjs-plugin-datalabels';

// Import for chart
import { Bar, Line } from "react-chartjs-2";
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  BarElement,
  PointElement,
  LineElement,
  Title,
  Tooltip,
  Legend,
} from 'chart.js';

ChartJS.register(
  CategoryScale,
  LinearScale,
  BarElement,
  PointElement,
  LineElement,
  Title,
  Tooltip,
  Legend,
  ChartDataLabels
);

export default function BooksReport({ initialReport = [], bookList = [], role }) {
  const [filteredGrouped, setFilteredGrouped] = useState([]);
  const [loading, setLoading] = useState(false);

  // filters
  const [startDate, setStartDate] = useState("");
  const [endDate, setEndDate] = useState("");
  const [selectedBook, setSelectedBook] = useState(null);
  const [selectcsvreport, setFilterCsvGrouped] = useState(null);
  const [reportTitle, setReportTitle] = useState("Book Wise Reports");
  const [chatMessages, setChatMessages] = useState([]); 
  const [selectedOption, setSelectedOption] = useState(null);
  const [selectedLineOption, setSelectedLineOption] = useState(null);

  const chartRef = useRef(null);
  const lineRef = useRef(null);

  const selectStyles = { menuPortal: (base) => ({ ...base, zIndex: 9999 }) };

  // react-select book options
  const bookOptions = useMemo(
    () =>
      (bookList || []).map((b) => ({
        value: String(b.book_pid),
        label: b.book_title?.charAt(0).toUpperCase() + b.book_title?.slice(1),
      })),
    [bookList]
  );

  // Options for download bar, line Chart formats
  const options = [
    { value: "png", label: "Download PNG" },
    { value: "jpg", label: "Download JPG" },
  ];

  // Handle bar chart selection
  const handleImageChange = (option) => {
    setSelectedOption(option);
    handleDownloadChartImage(option.value);
  };

  // Handle line chart selection
  const handleLineImageChange = (option) => {
    setSelectedLineOption(option);
    handleDownloadLineImage(option.value);
  };

  const formatTimeDisplay = (t) => {
    if (!t && t !== 0) return "-";
    if (typeof t === "string") return t.split(".")[0];
    if (t instanceof Date) return t.toTimeString().split(" ")[0];
    return String(t);
  };

  // Improved Pi Chat generator
  const generatePiResponseOnChange = (data) => {
    if (!data || data.length === 0) return "ℹ️ No data available for the selected filters.";

    let totalBooks = data.length;
    let totalTasks = 0;
    let grandTotal = 0;

    data.forEach((g) => {
      totalTasks += g.tasks.length;
      grandTotal += g.total || 0;
    });

    let msg = `📊 **Book Report Update**\n
      • Total Books: ${totalBooks}
      • Total Tasks: ${totalTasks}
      • Grand Total Count: ${grandTotal}
    `;

    data.forEach((g) => {
      msg += `📚 **${g.book_title}**\n`;
      g.tasks.forEach((t) => {
        msg += ` • ${t.task_name}: ${t.count}\n`;
      });
      msg += ` ⭐ Total for book: ${g.total}\n\n`;
    });

    return msg.trim();
  };

  // 🔹 Collect all unique task names across all books
  const allTaskNames = Array.from(
    new Set(filteredGrouped.flatMap((group) => group.tasks.map((t) => t.task_name)))
  );
  
  const totalBars = filteredGrouped.length * allTaskNames.length;
  const dynamicThickness = totalBars > 30 ? 10 : totalBars > 20 ? 20 : 30; 

  // 🔹 Create one dataset per task_name
  const datasets = allTaskNames.map((taskName, idx) => {
    const colors = [
      "#8C3FE1",
      "#E557AE",
      "#4CAF50",
      "#FF9800",
      "#2196F3",
      "#FF5722",
      "#9C27B0",
      "#009688",
      "#FFC107",
      "#795548",
    ];
    const avgTimeData = filteredGrouped.map((group) => {
        const task = group.tasks.find((t) => t.task_name === taskName);
        return task ? task.avg_time : ""; // formatter
    });
    return {
      label: taskName,
      data: filteredGrouped.map((group) => {
        const task = group.tasks.find((t) => t.task_name === taskName);
        return task ? task.count : 0; // 0 if that task doesn’t exist in this book
      }),
      avgTime: avgTimeData,
      backgroundColor: colors[idx % colors.length],
      borderRadius: 6,
      barThickness: dynamicThickness,   //✅ dynamic size
      maxBarThickness: dynamicThickness,
    };
  });

  // 🔹 Prepare chart data
  const barData = {
    labels: filteredGrouped.map((group) => group.book_title),
    datasets: datasets,
  };

  // 🔹 Chart options
  const barOptions = {
    responsive: true,
    maintainAspectRatio: false,
    devicePixelRatio: 3,
    plugins: {
      legend: {
        position: "bottom",
        labels: {
          font: {
            size: 12,
            weight: "bold",
            weight: "normal",
            family: "Helvetica, sans-serif", // ✅ Safe font for PDF rendering
          },
        },
      },
      title: {
        display: true,
        text: "Graphical Representation Book Wise Report",
        font: { 
          size: 16, 
          weight: "bold",
          family: "Helvetica, sans-serif", // ✅ Normal readable font
        },
        padding: {
          bottom: 30, // space below the title
        },
      },
      datalabels: {
        display: true,
        color: "#000", // black text
        anchor: "end",
        align: "top",
        offset: 4,
        font: { 
          size: 11, 
          weight: "bold",
          family: "Helvetica, sans-serif", // ✅ No pixelation
        },
        formatter: (value) => (value > 0 ? value : ""), // Show only non-zero values
      },
      tooltip: {
        bodyFont: {
          family: "Helvetica, sans-serif",
        },
        titleFont: {
          family: "Helvetica, sans-serif",
        },
        /* callbacks: {
          label: (context) =>
            `${context.dataset.label}: ${context.parsed.y}`,
        }, */
        callbacks: {
          label: function(context) {
            let lines = [];

            lines.push(`${context.dataset.label || ''}: ${context.parsed.y}`);

            const avgTimeArray = context.dataset.avgTime;
            const avgTime = avgTimeArray ? avgTimeArray[context.dataIndex] : null;

            if (avgTime !== null) {
                lines.push(`Avg Time: ${avgTime}`);
            }
            return lines;
          },
        },
      },
    },
    scales: {
      x: {
        title: {
          display: true,
          text: "Task Name",
          font: { 
            size: 12, 
            weight: "bold",
            family: "Helvetica, sans-serif", 
          },
        },
        ticks: {
          font: { 
            size: 12, 
            weight: "bold",
            family: "Helvetica, sans-serif",
          },
        },
        categoryPercentage: 0.5, // lower = more space between groups
        barPercentage: 0.8, 
      },
      y: {
        beginAtZero: true,
        title: {
          display: true,
          text: "Task Count",
          font: { 
            size: 12, 
            weight: "bold",
            family: "Helvetica, sans-serif", 
          },
        },
        ticks: { 
          stepSize: 1,
          font: { 
            size: 12, 
            weight: "bold",
            family: "Helvetica, sans-serif",
          },
        },
      },
    },
  };

  //Line Chart Option 
  const lineOptions = {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
      legend: {
        position: "bottom",
        labels: {
          font: {
            size: 12,
            weight: "bold",
            family: "Helvetica, sans-serif",
          },
        },
      },
      title: {
        display: true,
        text: "Graphical Representation Book Wise Report",
        font: { 
          size: 16, 
          weight: "bold",
          family: "Helvetica, sans-serif",
        },
        padding: { bottom: 30 },
      },
      datalabels: {
        display: true,
        color: "#000",
        anchor: "end",
        align: "top",
        offset: 4,
        font: { size: 11, weight: "bold", family: "Helvetica, sans-serif" },
        formatter: (value) => (value > 0 ? value : ""),
      },
    },
    scales: {
      x: {
        title: { display: true, text: "Books", font: { size: 12, weight: "bold", family: "Helvetica, sans-serif" } },
        ticks: { font: { size: 12, weight: "bold", family: "Helvetica, sans-serif" } },
      },
      y: {
        beginAtZero: true,
        title: { display: true, text: "Task Count", font: { size: 12, weight: "bold", family: "Helvetica, sans-serif" } },
        ticks: { stepSize: 1, font: { size: 12, weight: "bold", family: "Helvetica, sans-serif" } },
      },
    },
  };

  // handle filter changes
  const handleChange = async (field, value) => {
    setLoading(true);

    // update local state
    if (field === "startDate") setStartDate(value);
    if (field === "endDate") setEndDate(value);
    if (field === "book") setSelectedBook(value);


    // prepare payload for API call
    const payload = {
      book_pid: field === "book" ? value?.value : selectedBook?.value || null,
      start_date: field === "startDate" ? value : startDate || null,
      end_date: field === "endDate" ? value : endDate || null,
    };

    
    // ✅ Check date difference before API call
    if (payload.start_date && payload.end_date) {
      const startDate = new Date(payload.start_date);
      const endDate = new Date(payload.end_date);

      const diffTime = Math.abs(endDate - startDate);
      const totalDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
     
      if (totalDays > 31) {
        Swal.fire({
        //icon: "warning",
        title: "⚠️ Date Range Too Long",
        text: "The selected date range exceeds 31 days. Please choose a shorter range.",
        confirmButtonText: "Got it",
        confirmButtonColor: "#7C3AED", // purple accent
        background: "#F9FAFB", // light gray background
        color: "#1F2937", // dark text
        width: 500,
        padding: "1.5rem",
        showClass: {
          popup: "animate__animated animate__fadeInDown"
        },
        hideClass: {
          popup: "animate__animated animate__fadeOutUp"
        }
      });
        setLoading(false);
        setEndDate('');
        return; // stop further execution
      }
    } else if (!payload.start_date && payload.end_date) {
        Swal.fire({
          title: "Start Date Empty",
          text: "Please select a start date before choosing an end date.",
          confirmButtonText: "OK"
        });
        setLoading(false);
        setEndDate(''); // clear invalid end date
        return; // stop further execution
    } 
      
    try {
      const resq = await createBookReportAction(payload); // call service/API
      const res  = resq?.bookreport;

      if (res && Array.isArray(res)) {
        // group data by book_title
        const groups = Object.values(
          res.reduce((acc, row) => {
            const bookRaw = row.book_title ?? "Unknown Book";
            const book = bookRaw.charAt(0).toUpperCase() + bookRaw.slice(1).toLowerCase();
            //const book = row.book_title ?? "Unknown Book";
            if (!acc[book]) acc[book] = { book_title: book, tasks: [] };
            acc[book].tasks.push({
              task_name: row.task_name ?? "-",
              avg_time: row.avg_time_per_task ?? row.avgTime ?? "-",
              count: Number(row.task_count ?? 0),
            });
            return acc;
          }, {})
        );
        groups.forEach((g) => {
          g.total = g.tasks.reduce((s, t) => s + (Number(t.count) || 0), 0);
        });

        const reportlist = await allBookReportList(payload);
        
        const csvReport = reportlist?.bookreportlist?.reduce((acc, item) => {
          let group = acc.find(g => g.book_title === item.book_title);
          if (!group) {
            const bk_title = item.book_title.charAt(0).toUpperCase() + item.book_title.slice(1).toLowerCase();
            group = { book_title: bk_title, tasks: [] };
            acc.push(group);
          }
          group.tasks.push({
            task_name: item.task_name,
            entry_duration: item.entry_duration,
            count: item.task_count,
          });
          return acc;
        }, []);

        setFilteredGrouped(groups);
        setFilterCsvGrouped(csvReport);

        // Update chat
        const piMessage = generatePiResponseOnChange(groups);
        setChatMessages((prev) => [...prev, { from: "pi", text: piMessage }]);

        // ✅ Dynamically update title

        // Helper to format date as MM-DD-YYYY
        const formatDate = (dateString) => {
          const date = new Date(dateString);
          if (isNaN(date)) return "-"; // handle invalid date
          const mm = String(date.getMonth() + 1).padStart(2, "0");
          const dd = String(date.getDate()).padStart(2, "0");
          const yyyy = date.getFullYear();
          return `${dd}-${mm}-${yyyy}`;
        };
        
        let title = "Book Wise Reports";

        if (payload.start_date && payload.end_date) {
          const startDate = new Date(payload.start_date);
          const endDate = new Date(payload.end_date);

          // Calculate day difference
          const diffTime = Math.abs(endDate - startDate);
          const totalDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

          const startMonth = startDate.getMonth();
          const endMonth = endDate.getMonth();
          const startYear = startDate.getFullYear();
          const endYear = endDate.getFullYear();

          // ✅ Same month, same year, and ≤ 25 days
          if (startMonth === endMonth && startYear === endYear && totalDays >= 25) {
            const monthName = endDate.toLocaleString("default", { month: "long" });
            title = `Book Wise Reports - ${monthName}, ${endYear}`;
          } 
          else {
            // ✅ Otherwise show full date range
            title = `Book Wise Reports - ${formatDate(payload.start_date)} to ${formatDate(payload.end_date)}`;
          }
        } 
        else if (payload.book_pid && (field === "book" ? value?.label : selectedBook?.label)) {
          const bookLabel = field === "book" ? value?.label : selectedBook?.label;
          title = `Book Wise Report - ${bookLabel}`;
        }

        setReportTitle(title);
        
      } else {
        setFilteredGrouped([]);
        setFilterCsvGrouped()
        setReportTitle();
      }
    } catch (err) {
      console.error("Error fetching book report:", err);
      setFilteredGrouped([]);
    } finally {
      setLoading(false);
    }
  };

  const grandTotal = filteredGrouped.reduce((s, g) => s + (g.total || 0), 0);

   // --- Export Handlers ---
  const formatDate = (dateString) => {
    const date = new Date(dateString);
    if (isNaN(date)) return "-"; // handle invalid date
    const mm = String(date.getMonth() + 1).padStart(2, "0");
    const dd = String(date.getDate()).padStart(2, "0");
    const yyyy = date.getFullYear();
    return `${dd}-${mm}-${yyyy}`;
  };
  // Excel Export
  const handleExportExcel = async () => {
    const ws_data = [
      ["Book Title", "Task", "Avg. Time", "Total"],
    ];
    filteredGrouped.forEach(group => {
      group.tasks.forEach((task, idx) => {
        ws_data.push([
          idx === 0 ? group.book_title : "",
          task.task_name,
          formatTimeDisplay(task.avg_time),
          task.count,
        ]);
      });
      ws_data.push([
        "", "Total", "", group.total
      ]);
    });
    ws_data.push(["Grand Total", "", "", grandTotal]);

    const ws = XLSX.utils.aoa_to_sheet(ws_data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, "Report");

    XLSX.writeFile(wb, "book_report.xlsx");
  };

// 📊 Chart image download (PNG)
  const handleDownloadChartImage = (format = "png") => {
  const chartCanvas = chartRef.current?.canvas;
  if (!chartCanvas) return;

  const scale = 3; // higher scale = higher resolution
  const tempCanvas = document.createElement("canvas");
  tempCanvas.width = chartCanvas.width * scale;
  tempCanvas.height = chartCanvas.height * scale;

  const ctx = tempCanvas.getContext("2d");

  // Fill background white for PNG/JPG
  if (format === "png" || format === "jpg") {
    ctx.fillStyle = "#ffffff";
    ctx.fillRect(0, 0, tempCanvas.width, tempCanvas.height);
  }

  ctx.scale(scale, scale);
  ctx.drawImage(chartCanvas, 0, 0);

  let image, filename;

  if (format === "png") {
    image = tempCanvas.toDataURL("image/png", 1.0);
    filename = "BookWiseReportChart.png";
  } else if (format === "jpg") {
    image = tempCanvas.toDataURL("image/jpeg", 1.0);
    filename = "BookWiseReportChart.jpg";
  } else if (format === "svg") {
    // For SVG, use Chart.js built-in toBase64SVG() if using Chart.js v4+
    const chart = chartRef.current;
    if (chart && chart.toBase64Image) {
      image = chart.toBase64Image("image/svg+xml"); // returns SVG as data URL
      filename = "BookWiseReportChart.svg";
    }
  }

  if (image) {
    const link = document.createElement("a");
    link.href = image;
    link.download = filename;
    link.click();
  }
};

const handleDownloadLineImage = (format = "png") => {
  const lineCanvas = lineRef.current?.canvas;
  if (!lineCanvas) return;

  const scale = 3; // higher scale = higher resolution
  const tempCanvas = document.createElement("canvas");
  tempCanvas.width = lineCanvas.width * scale;
  tempCanvas.height = lineCanvas.height * scale;

  const ctx = tempCanvas.getContext("2d");

  // Fill background white for PNG/JPG
  if (format === "png" || format === "jpg") {
    ctx.fillStyle = "#ffffff";
    ctx.fillRect(0, 0, tempCanvas.width, tempCanvas.height);
  }

  ctx.scale(scale, scale);
  ctx.drawImage(lineCanvas, 0, 0);

  let image, filename;

  if (format === "png") {
    image = tempCanvas.toDataURL("image/png", 1.0);
    filename = "BookWiseReportLine.png";
  } else if (format === "jpg") {
    image = tempCanvas.toDataURL("image/jpeg", 1.0);
    filename = "BookWiseReportLine.jpg";
  } else if (format === "svg") {
    // For SVG, use Chart.js built-in toBase64SVG() if using Chart.js v4+
    const line = lineRef.current;
    if (line && line.toBase64Image) {
      image = line.toBase64Image("image/svg+xml"); // returns SVG as data URL
      filename = "BookWiseReportLine.svg";
    }
  }

  if (image) {
    const link = document.createElement("a");
    link.href = image;
    link.download = filename;
    link.click();
  }
};

// PDF Export
 const handleExportPDF = async () => {
  // 🧠 Ensure we are in the browser
  if (typeof window === "undefined") return;
  const doc = new jsPDF();
  // Title
  const pageWidth = doc.internal.pageSize.getWidth();
  const imageurl = "/logo-new.png"; // ✅ correct path (not /public/logo.png)
  doc.addImage(imageurl, "PNG", 12, 5, 40, 12);
 
  doc.setFontSize(13);
  const titleY = 12;
  let title = "Book Wise Reports";
  const drawHeader = (docInstance) => {
    docInstance.addImage(imageurl, "PNG", 15, 2, 35, 15);
    docInstance.setFontSize(13);
    docInstance.setTextColor(40);
    // ✅ Title generation
    if (startDate && endDate) {
      const start = new Date(startDate);
      const end = new Date(endDate);
      const diffTime = Math.abs(end - start);
      const totalDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

      const startMonth= start.getMonth();
      const endMonth  = end.getMonth();
      const startYear = start.getFullYear();
      const endYear   = end.getFullYear();

      if (startMonth === endMonth && startYear === endYear && totalDays >= 25) {
        const monthName = endDate.toLocaleString("default", { month: "long" });
        title = `Book Wise Reports - ${monthName}, ${endYear}`;
      } 
      else {
        // ✅ Otherwise show full date range
        title = `Book Wise Reports - ${formatDate(start)} to ${formatDate(end)}`;
      }
    } else if (selectedBook?.label) {
      title = `Book Wise Report - ${selectedBook.label}`;
    }
      docInstance.text(title, pageWidth / 2, titleY, { align: "center" });
      // Draw a line under header
      docInstance.setLineWidth(0.1);
      docInstance.line(10, 18, pageWidth - 10, 18);
  };
  drawHeader(doc); // Draw header on first page

  // --- Table Section ---
  const tableStartY = titleY + 8; // tighter space between title and table
  const body = [];

  // 🔹 Loop through each book group
  filteredGrouped.forEach(group => {
    group.tasks.forEach((task, idx) => {
      body.push([
        idx === 0 ? group.book_title : "",
        task.task_name,
        formatTimeDisplay(task.avg_time),
        task.count.toString(),
      ]);
    });

    // ✅ Add per-book total row (styled)
    body.push([
      "",
      { content: "Total", styles: { fontStyle: "bold", halign: "left" } },
      "",
      { content: group.total.toString(), styles: { fontStyle: "bold" ,halign: "left"} },
    ]);
    // ✅ Add an empty row for spacing
    //body.push(["", "", "", ""]);
  });

  body.push([
  { content: "Grand Total", colSpan: 3, styles: { fontStyle: "bold", halign: "left" } },
  { content: grandTotal.toString(), styles: { fontStyle: "bold", halign: "left" } },
]);

  // 🧾 Generate table
  autoTable(doc,{
    head: [["Book Title", "Task", "Avg. Time", "Total"]],
    body,
    startY: tableStartY,
    theme: "grid",
    styles: {
      fontSize: 8,
      cellPadding: 1,
      halign: "left",
    },
    headStyles: {
      fillColor: [200, 200, 200],
      textColor: [0, 0, 0],
      fontStyle: "bold",
    },
    // ✅ remove book_title cell bottom border for grouped rows
    didParseCell: function (data) {
      if (data.column.index === 0 && data.cell.raw === "") {
        data.cell.styles.lineWidth = { top: 0, bottom: 0, left: 0.1, right: 0.1 };
      }
    },
    didDrawPage: (data) => {
      // Draw header line on each page
      drawHeader(doc);
    }
  });

   // --- Bar Chart on New Page ---
  doc.addPage();
  drawHeader(doc); // Header for chart page
   // --- 📊 Bar Chart on New Page ---
  doc.setFontSize(12);
  const chartCanvas = document.querySelector("#barChartContainer canvas");
  
  if (chartCanvas) {
    const scale = 3; // ⬆️ 3x resolution for better image quality

    // 🖼️ Create a temporary high-res canvas
    const tempCanvas = document.createElement("canvas");
    tempCanvas.width = chartCanvas.width * scale;
    tempCanvas.height = chartCanvas.height * scale;

    const ctx = tempCanvas.getContext("2d");
    ctx.scale(scale, scale);
    ctx.drawImage(chartCanvas, 0, 0);

    // 📸 Convert to high-quality PNG image
    const chartImg = tempCanvas.toDataURL("image/png", 1.0);

    // 📏 Calculate chart dimensions for PDF
    const chartWidth = pageWidth - 20;
    const chartHeight = (chartCanvas.height * chartWidth) / chartCanvas.width;

    // 🧾 Add chart image to PDF
    doc.addImage(chartImg, "PNG", 10, 30, chartWidth, chartHeight);
  }  
  // Save file
  const date = new Date().toISOString().split("T")[0];
  //doc.save(`book_report_${date}.pdf`);
  const filename = `BookWiseReport_${date}.pdf`;
  doc.save(filename);
}; 

  // CSV Export
  const handleExportCSV = () => {
    const data = [];
    selectcsvreport.forEach(group => {
      group.tasks.forEach((task, idx) => {
        data.push({
          "Book Title": group.book_title,
          Task: task.task_name,
          "Avg. Time": formatTimeDisplay(task.entry_duration),
          Count: task.count,
        });
      });
    });

    const csv = Papa.unparse(data);
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.setAttribute("download", "BookWiseReport.csv");
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };
  
  return (
    <div className="max-w-full px-4 py-4 mx-auto rounded-lg space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-xl font-semibold text-gray-800">Book Wise Reports List</h2>
            <p className="text-gray-500 mt-1 text-sm">
              Below is a list of all Book Wise Reports.
            </p>
          </div>
        </div>
      </header>
      {/* Filters */}
      <div className="bg-purple-50 border border-purple-200 rounded-lg p-4 shadow-sm flex flex-wrap gap-4 items-center justify-start">
        <div className="flex flex-col">
          <label className="text-xs text-gray-600 font-semibold mb-1">Start Date</label>
          <input
            type="date"
            className="w-40 border border-gray-300 rounded px-3 py-2 text-sm focus:ring focus:ring-purple-200"
            value={startDate}
            onChange={(e) => handleChange("startDate", e.target.value)}
            max={endDate}
          />
        </div>
        <div className="flex flex-col">
          <label className="text-xs text-gray-600 font-semibold mb-1">End Date</label>
          <input
            type="date"
            className="w-40 border border-gray-300 rounded px-3 py-2 text-sm focus:ring focus:ring-purple-200"
            value={endDate}
            onChange={(e) => handleChange("endDate", e.target.value)}
            min={startDate}
          />
        </div>
        <div className="flex flex-col">
          <label className="text-xs text-gray-600 font-semibold mb-1">Book</label>
          <Select
            instanceId="book-filter"
            options={bookOptions}
            value={selectedBook}
            onChange={(v) => handleChange("book", v)}
            placeholder="Select Book"
            isClearable
            className="w-48 text-sm"
            menuPortalTarget={typeof document !== "undefined" ? document.body : undefined}
            styles={selectStyles}
          />
        </div>
        <div className="pt-5">
          <button
            onClick={() => {
              setStartDate("");
              setEndDate("");
              setSelectedBook(null);
              setFilteredGrouped([]);
            }}
            className="text-sm px-4 py-2 bg-purple-100 hover:bg-purple-200 text-purple-700 rounded transition-all duration-200 cursor-pointer"
          >
            Clear Filters
          </button>
        </div>
      </div>
      <header>
        <h2 className="text-xl font-semibold text-gray-800 text-center">{reportTitle}</h2>
      </header>
      {/* Table */}
      <div className="overflow-x-auto">
        <table className="min-w-full bg-white border border-gray-200 rounded-lg shadow-sm">
          <thead>
            <tr className="bg-gray-100 text-sm h-12 border border-b border-gray-300">
              <th className="py-2 px-4 border-b text-left w-1/3">Book Title</th>
              <th className="py-2 px-4 border-b text-left w-1/3">Task</th>
              <th className="py-2 px-4 border-b text-left w-1/6">Avg. Time</th>
              <th className="py-2 px-4 border-b text-left w-1/6">Total</th>
            </tr>
          </thead>
          <tbody>
            {loading ? (
              <tr>
                <td colSpan={4} className="text-center p-6">
                  Loading...
                </td>
              </tr>
            ) : filteredGrouped.length === 0 ? (
              <tr>
                <td colSpan={4} className="text-center p-6 text-gray-500">
                  No data found
                </td>
              </tr>
            ) : (
              filteredGrouped.map((group, gi) => {
                const rowSpan = group.tasks.length + 1;
                return group.tasks.map((task, ti) => (
                  <tr key={`${gi}-${ti}`} className={ti % 2 === 0 ? "bg-white" : "bg-gray-50"}>
                    {ti === 0 && (
                      <td
                        rowSpan={rowSpan}
                        className="py-2 px-4 border-l border-r border-b align-top font-medium text-gray-700 w-1/3"
                      >
                        {group.book_title}
                      </td>
                    )}
                    <td className="py-2 px-4 border-b border-r w-1/3">{task.task_name}</td>
                    <td className="py-2 px-4 border-b border-r text-left font-mono w-1/6">
                      {formatTimeDisplay(task.avg_time)}
                    </td>
                    <td className="py-2 px-4 border-b border-r text-left font-semibold w-1/6">{task.count}</td>
                  </tr>
                )).concat(
                  <tr key={`${gi}-total`} className="bg-gray-200/50">
                    <td className="py-1 px-4 border-b border-r text-left font-medium">Total</td>
                    <td className="py-1 px-4 border-b border-r text-left font-mono"></td>
                    <td className="py-1 px-4 border-b border-r text-left font-semibold">{group.total}</td>
                  </tr>
                );
              })
            )}
          </tbody>
          {filteredGrouped.length > 0 && (
            <tbody>
              <tr className="bg-gray-200/50">
                <td className="py-1 px-4 border-l border-b border-r text-left font-medium">Grand Total</td>
                <td className="py-1 px-4 border-b border-r text-left font-mono"></td>
                <td className="py-1 px-4 border-b border-r text-left font-mono"></td>
                <td className="py-1 px-4 border-b border-r text-left font-semibold">{grandTotal}</td>
              </tr>
            </tbody>
          )}
        </table>
      </div>
      {/* Export Buttons at Bottom */}
      {filteredGrouped.length > 0 && (
        <div className="flex gap-2 mt-6 justify-end">
          <button
            onClick={handleExportExcel}
            /*className="bg-green-100 hover:bg-green-200 text-green-700 px-4 py-2 rounded"  */className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
          >
            Excel
          </button>
          <button
            onClick={handleExportPDF}
            /* className="bg-blue-100 hover:bg-blue-200 text-blue-700 px-4 py-2 rounded" */
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
          >
            Export PDF
          </button>
          <button
            onClick={handleExportCSV}
            /* className="bg-yellow-100 hover:bg-yellow-200 text-yellow-700 px-4 py-2 rounded" */
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
          >
            Download CSV
          </button>
        </div>
      )}  
      {/* Pie Chart */}
      {filteredGrouped.length > 0 && (
        <>
          <div
            id="barChartContainer" // 👈 required for PDF capture
            className="bg-purple-50 border border-purple-200 rounded-lg p-4 shadow-sm 
                      flex flex-wrap gap-4 items-center justify-start"
            >
            <div className="w-full mx-auto">
              {/* 📊 Chart */}
              <div className="relative h-[400px] mb-4">
                <Bar ref={chartRef} data={barData} options={barOptions} />
              </div>

              {/* ⬇️ Download Button (centered below the chart) */}
              <div className="relative flex justify-end float-right mt-2 mb-5 w-45">
                <Select
                  value={selectedOption}
                  onChange={handleImageChange}
                  options={options}
                  placeholder="Download Image"
                  menuPortalTarget={document.body} // ensures dropdown overlays correctly
                  styles={{
                    control: (base, state) => ({
                      ...base,
                      cursor: "pointer",
                      boxShadow: "none",
                      borderColor: state.isFocused ? "#4CAF50" : "#2196F3", 
                      "&:hover": { borderColor: "#4CAF50" },
                      background: "linear-gradient(to right, #4CAF50, #2196F3)", // green → blue gradient
                      color: "#fff",
                    }),
                    menuPortal: (base) => ({
                      ...base,
                      zIndex: 9999, // ensures dropdown is on top
                    }),
                    placeholder: (base) => ({
                      ...base,
                      color: "white",
                      fontWeight: 500,
                    }),
                    singleValue: (base) => ({
                      ...base,
                      color: "white",
                    }),
                  }}
                  className="w-full"
                  classNamePrefix="react-select"
                />
              </div>
            </div>
          </div>
          <div
            id="lineChartContainer" // 👈 required for PDF capture
            className="bg-purple-50 border border-purple-200 rounded-lg p-4 shadow-sm 
                      flex flex-wrap gap-4 items-center justify-start"
          >
            <div className="w-full mx-auto">
              <div className="relative h-[400px] mb-4">
                <Line ref={lineRef} data={barData} options={lineOptions} />
              </div>
              {/* ⬇️ Download Button (centered below the Linechart) */}
              <div className="relative flex justify-end float-right  mt-2 mb-5 w-45">
                <Select
                  value={selectedLineOption}
                  onChange={handleLineImageChange}
                  options={options}
                  placeholder="Download Image"
                  menuPortalTarget={document.body} // ensures dropdown overlays correctly
                  styles={{
                    control: (base, state) => ({
                      ...base,
                      cursor: "pointer",
                      boxShadow: "none",
                      borderColor: state.isFocused ? "#4CAF50" : "#2196F3", 
                      "&:hover": { borderColor: "#4CAF50" },
                      background: "linear-gradient(to right, #4CAF50, #2196F3)", // green → blue gradient
                      color: "#fff",
                    }),
                    menuPortal: (base) => ({
                      ...base,
                      zIndex: 9999, // ensures dropdown is on top
                    }),
                    placeholder: (base) => ({
                      ...base,
                      color: "white",
                      fontWeight: 500,
                    }),
                    singleValue: (base) => ({
                      ...base,
                      color: "white",
                    }),
                  }}
                  className="w-full"
                  classNamePrefix="react-select"
                />
              </div>
            </div>  
          </div>
        </>
      )}
    </div>
  );
}
