"use client";
import { useState, useMemo, Fragment } from "react";
import {
  MagnifyingGlassIcon,
  XMarkIcon,
  ArrowUpIcon,
  ArrowDownIcon,
  TableCellsIcon,
  DocumentTextIcon,
  ChevronUpIcon,
  ChevronDownIcon,
} from "@heroicons/react/24/solid";
// import * as XLSX from "xlsx";
// import jsPDF from "jspdf";
// import autoTable from "jspdf-autotable";
import { usePathname } from "next/navigation";
import ChildDataTable from "./childDataTable";
import ChildPendingDataTable from "./childPendingDataTable";

function ModalWrapper({ content, text, maxChars = 20, modalHeader }) {
  const [open, setOpen] = useState(false);
  const strValue = text?.toString() || "";
  const isLong = strValue.length > maxChars;
  const displayText = isLong ? strValue.slice(0, maxChars) + "..." : strValue;
  if (!text) {
    return typeof content === "function" ? content(displayText, isLong) : content;
  }

  const renderedContent =
    typeof content === "function"
      ? content(displayText, isLong)
      : typeof content === "string"
        ? displayText
        : content;

  return (
    <>
      <span
        className={`${isLong ? "cursor-pointer text-blue-600 hover:underline" : "text-gray-800"
          }`}
        onClick={() => isLong && setOpen(true)}
      >
        {renderedContent}
      </span>

      {open && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm">
          <div className="relative bg-white rounded-2xl shadow-2xl max-w-2xl w-[90%] max-h-[85vh] overflow-y-auto p-8 animate-fadeIn">
            <button
              onClick={() => setOpen(false)}
              className="absolute top-4 right-4 text-gray-500 hover:text-gray-800 transition"
            >
              <XMarkIcon className="w-7 h-7" />
            </button>
            <h3 className="text-2xl font-bold mb-4 text-gray-900 border-b pb-3">
              {modalHeader}
            </h3>
            <p className="whitespace-pre-wrap text-gray-700 leading-relaxed text-[15px]">
              {strValue}
            </p>
            <div className="mt-8 flex justify-end">
              <button
                onClick={() => setOpen(false)}
                className="px-5 py-2.5 bg-gradient-to-r from-blue-500 to-blue-700 text-white font-medium rounded-lg shadow hover:from-blue-600 hover:to-blue-800 transition"
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  );
}

export default function CustomDataTable({ columns, data, defaultSort }) {
  const pathname = usePathname();
  const segments = pathname.split("/");
  const segmentValue = segments[segments.length - 1];


  const [search, setSearch] = useState("");
  const [currentPage, setCurrentPage] = useState(1);
  const [perPage, setPerPage] = useState(5);
  const [sortConfig, setSortConfig] = useState({
    key: defaultSort?.key || null,
    direction: defaultSort?.direction || null,
  });
  const [visibleColumns, setVisibleColumns] = useState(columns.map((col) => col.label));
  const [showDropdown, setShowDropdown] = useState(false);
  const [expandedRows, setExpandedRows] = useState([]);

  const displayedColumns = useMemo(() => {
    return columns.filter((col) => visibleColumns.includes(col.label));
  }, [columns, visibleColumns]);

  const handleSort = (col) => {
    if (!col.selector) return;
    let direction = "ascending";
    if (sortConfig.key === col.label && sortConfig.direction === "ascending") {
      direction = "descending";
    }
    setSortConfig({ key: col.label, direction });
  };

  const toggleRow = (rowId) => {
    setExpandedRows((prev) =>
      prev.includes(rowId)
        ? prev.filter((id) => id !== rowId)
        : [...prev, rowId]
    );
  };

  const filteredData = useMemo(() => {
    if (!Array.isArray(data)) return [];
    let filtered = data.filter((item) =>
      displayedColumns.some((col) => {
        if (!col.selector) return false;
        const value = col.selector(item);
        return (
          value && value.toString().toLowerCase().includes(search.toLowerCase())
        );
      })
    );

    if (sortConfig.key) {
      const col = columns.find((c) => c.label === sortConfig.key);
      if (col && col.selector) {
        filtered.sort((a, b) => {
          const aVal = col.selector(a);
          const bVal = col.selector(b);
          if (aVal == null) return 1;
          if (bVal == null) return -1;
          if (typeof aVal === "number" && typeof bVal === "number") {
            return sortConfig.direction === "ascending"
              ? aVal - bVal
              : bVal - aVal;
          }
          return sortConfig.direction === "ascending"
            ? aVal.toString().localeCompare(bVal.toString())
            : bVal.toString().localeCompare(aVal.toString());
        });
      }
    }
    return filtered;
  }, [search, data, columns, sortConfig, displayedColumns]);

  const totalPages = Math.ceil(filteredData.length / perPage) || 1;
  const paginatedData = useMemo(() => {
    const start = (currentPage - 1) * perPage;
    return filteredData.slice(start, start + perPage);
  }, [filteredData, currentPage, perPage]);

  const getPageNumbers = () => {
    const maxPagesToShow = 5;
    if (totalPages <= maxPagesToShow)
      return [...Array(totalPages).keys()].map((i) => i + 1);
    const pages = [];
    if (currentPage > 3) pages.push(1, "...");
    else pages.push(1, 2, 3);
    if (currentPage > 3 && currentPage < totalPages - 2) {
      pages.push(currentPage - 1, currentPage, currentPage + 1);
    }
    if (currentPage < totalPages - 2) pages.push("...", totalPages);
    else pages.push(totalPages - 2, totalPages - 1, totalPages);
    return [...new Set(pages)];
  };

  const renderCell = (col, row, rowIndex) => {
    const value = col.selector ? col.selector(row) : "";

    // ---------- Book ID column clickable ----------
    if (col.label === "Book ID" && segmentValue === "dashboard") {
      const rowId = row.bookId || rowIndex;
      const bookIdValue = row.book_CustomId || "N/A";

      // Change badge color when row is expanded
      const badgeClass = expandedRows.includes(rowId)
        ? "inline-flex items-center px-2 py-1 text-xs rounded-md bg-red-700 text-white border border-red-800"
        : "inline-flex items-center px-2 py-1 text-xs rounded-md bg-green-700 text-white border border-green-800";

      return (
        <div
          className={`flex items-center justify-between cursor-pointer ${badgeClass}`}
          onClick={() => toggleRow(rowId)}
        >
          <span>{bookIdValue}</span>
          {expandedRows.includes(rowId) ? (
            <ChevronUpIcon className="w-4 h-4 ml-1 text-white" />
          ) : (
            <ChevronDownIcon className="w-4 h-4 ml-1 text-white" />
          )}
        </div>
      );
    }

    if (col.label === "Pending Duration" && segmentValue === "lists") {

      const rowId = row.entryId || rowIndex;
      const pendingdurationvalue = row.total_pending_duration || "N/A";

      // Change badge color when row is expanded
      const badgeClass = expandedRows.includes(rowId)
        ? "inline-flex items-center px-2 py-1 text-xs rounded-md bg-red-700 text-white border border-red-800"
        : "inline-flex items-center px-2 py-1 text-xs rounded-md bg-green-700 text-white border border-green-800";

      return (
        <div
          className={`flex items-center justify-between cursor-pointer ${badgeClass}`}
          onClick={() => toggleRow(rowId)}
        >
          <span>{pendingdurationvalue}</span>
          {expandedRows.includes(rowId) ? (
            <ChevronUpIcon className="w-4 h-4 ml-1 text-white" />
          ) : (
            <ChevronDownIcon className="w-4 h-4 ml-1 text-white" />
          )}
        </div>
      );
    }



    // S.No column
    if (col.label === "S.No") return (currentPage - 1) * perPage + rowIndex + 1;
    if (col.cell) return col.cell(row, rowIndex);

    return <ModalWrapper content={value} text={value} maxChars={20} modalHeader={col?.label} />;
  };

  const handleDownloadExcel = async () => {

    const XLSX = await import("xlsx");
    const dataToExport = filteredData.map((row) => {
      const exportRow = {};
      displayedColumns.forEach((col) => {
        if (col.label === "S.No") {
          exportRow[col.label] = filteredData.indexOf(row) + 1;
        } else if (col.selector) {
          exportRow[col.label] = col.selector(row);
        }
      });
      return exportRow;
    });
    const headers = Object.keys(dataToExport[0] || {});
    const aoaData = [
      ["", "", "", `${segmentValue.charAt(0).toUpperCase() + segmentValue.slice(1)} Reports`],
      headers,
      ...dataToExport.map((row) => Object.values(row)),
    ];
    const worksheet = XLSX.utils.aoa_to_sheet(aoaData);
    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, "Sheet1");
    const filename = `${segmentValue.charAt(0).toUpperCase() + segmentValue.slice(1)}.xlsx`;
    XLSX.writeFile(workbook, filename);
  };

  const handleDownloadPdf = async () => {

     const { default: jsPDF } = await import("jspdf");
     const { default: autoTable } = await import("jspdf-autotable"); // Fix here

    const doc = new jsPDF();
    const imageurl = "/logo.jpg";
    doc.addImage(imageurl, "JPEG", 15, 5, 50, 15);
    doc.setFontSize(18);
    doc.setTextColor(40);
    doc.text(`${segmentValue.charAt(0).toUpperCase() + segmentValue.slice(1)} Report`, 70, 15);

    const startY = 25;
    const pdfColumns = displayedColumns.filter(col => col.label !== "Action");
    const tableColumn = pdfColumns.map((col) => col.label);
    const tableRows = [];

    filteredData.forEach((row, rowIndex) => {
      const rowData = pdfColumns.map((col) => {
        if (col.label === "S.No") return rowIndex + 1;
        if (col.selector) return col.selector(row)?.toString() || "";
        return "";
      });
      tableRows.push(rowData);
    });

    autoTable(doc, {
      head: [tableColumn],
      body: tableRows,
      startY: startY,
      theme: "grid",
      styles: { cellPadding: 2, fontSize: 10, valign: "middle", halign: "left" },
      headStyles: { fillColor: [30, 144, 255], textColor: [255, 255, 255], fontStyle: "bold" },
      alternateRowStyles: { fillColor: [240, 240, 240] },
      margin: { top: 10 },
    });
    const filename = `${segmentValue.charAt(0).toUpperCase() + segmentValue.slice(1)}.pdf`;
    doc.save(filename);
  };

  return (
    <div className="bg-white rounded-xl shadow-md border border-gray-200 overflow-hidden text-gray-800" style={{minHeight:'350px', display: 'flex',
    flexDirection: 'column',
    justifyContent: 'space-between'}}>
     <div style={{minHeight:'300px'}}> 
       {/* Top controls */}
      <div className="p-3 flex justify-end items-center gap-3 border-b border-gray-200 bg-gray-50">
        <div className="relative w-64">
          <MagnifyingGlassIcon className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-purple-500" />
          <input
            type="text"
            placeholder="Search..."
            value={search}
            onChange={(e) => { setSearch(e.target.value); setCurrentPage(1); }}
            className="w-full pl-10 pr-8 py-1 text-sm bg-white border border-gray-300 rounded-md focus:ring-2 focus:ring-purple-500 focus:border-purple-500"
          />
          {search && (
            <button
              onClick={() => setSearch("")}
              className="absolute right-2 top-1/2 -translate-y-1/2 text-gray-400 hover:text-purple-500"
              type="button"
            >
              <XMarkIcon className="w-5 h-5" />
            </button>
          )}
        </div>

        {/* Show/Hide Columns */}
        <div className="relative">
          <button
            onClick={() => setShowDropdown(!showDropdown)}
            className="px-2 py-1 text-sm text-white bg-green-600 border border-green-700 rounded-md hover:bg-green-700 cursor-pointer"
          >
            Show/Hide Col.
          </button>
          {showDropdown && (
            <div className="absolute right-0 mt-2 w-48 bg-white border rounded shadow-lg z-50 max-h-60 overflow-y-auto">
              <label className="flex items-center p-2 cursor-pointer hover:bg-gray-100 border-b">
                <input
                  type="checkbox"
                  checked={visibleColumns.length === columns.length}
                  onChange={() => {
                    setVisibleColumns(
                      visibleColumns.length === columns.length ? [] : columns.map((col) => col.label)
                    );
                  }}
                  className="mr-2"
                />
                Select All
              </label>
              {columns.map((col) => (
                <label key={col.label} className="flex items-center p-2 cursor-pointer hover:bg-gray-100">
                  <input
                    type="checkbox"
                    checked={visibleColumns.includes(col.label)}
                    onChange={() => {
                      setVisibleColumns(
                        visibleColumns.includes(col.label)
                          ? visibleColumns.filter((l) => l !== col.label)
                          : [...visibleColumns, col.label]
                      );
                    }}
                    className="mr-2"
                  />
                  {col.label}
                </label>
              ))}
            </div>
          )}
        </div>

        {/* Excel/PDF */}
        {(segmentValue === "pending" || segmentValue === "completed") && (
          <>
            <button onClick={handleDownloadExcel} className="px-2 py-1 text-sm text-white bg-blue-600 border border-blue-700 rounded-md hover:bg-blue-700 flex items-center gap-1 cursor-pointer">
              <TableCellsIcon className="w-4 h-4" /> Excel
            </button>
            <button onClick={handleDownloadPdf} className="px-2 py-1 text-sm text-white bg-red-600 border border-red-700 rounded-md hover:bg-red-700 flex items-center gap-1 cursor-pointer">
              <DocumentTextIcon className="w-4 h-4" /> PDF
            </button>
          </>
        )}
         {(segmentValue === "breaks") && (
          <>
            <button onClick={handleDownloadExcel} className="px-2 py-1 text-sm text-white bg-blue-600 border border-blue-700 rounded-md hover:bg-blue-700 flex items-center gap-1 cursor-pointer">
              <TableCellsIcon className="w-4 h-4" /> Excel
            </button>
          </>
        )}
      </div>

      {/* Table */}
      <div className="overflow-x-auto" style={{minHeight:'300px'}}>
        <table className="min-w-full border-separate border-spacing-0">
          <thead>
            <tr className="bg-gray-100 sticky top-0 z-10">
              {displayedColumns.map((col, i) => (
                <th
                  key={i}
                  className="px-4 py-2 text-left text-sm font-semibold text-gray-700 border-b border-gray-200 cursor-pointer hover:text-blue-600"
                  onClick={() => handleSort(col)}
                >
                  <div className="flex items-center space-x-1">
                    <span>{col.label}</span>
                    {sortConfig.key === col.label &&
                      (sortConfig.direction === "ascending" ? (
                        <ArrowUpIcon className="w-4 h-4 text-blue-500" />
                      ) : (
                        <ArrowDownIcon className="w-4 h-4 text-blue-500" />
                      ))}
                  </div>
                </th>
              ))}
            </tr>
          </thead>

          <tbody>
            {paginatedData.length > 0 ? (
              paginatedData.map((row, rowIndex) => {
                const rowId = row.bookId || rowIndex;
                return (
                  <Fragment key={rowId}>
                    <tr className="odd:bg-white even:bg-gray-50 hover:bg-blue-50 transition-colors">
                      {displayedColumns.map((col, colIndex) => (
                        <td key={colIndex} className="px-4 py-2 text-sm border-b border-gray-200">
                          {renderCell(col, row, rowIndex)}
                        </td>
                      ))}
                    </tr>

                    {expandedRows.includes(rowId)  && segmentValue === "dashboard" && (
                      <tr>
                        <td colSpan={displayedColumns.length} className="bg-gray-50 px-6 py-2">
                          {/* <ChildDataTable bookId={row.book_pid} /> */}
                          <ChildDataTable chapters={row.chapters || []} />
                        </td>
                      </tr>
                    )}
                    
                    {expandedRows.includes(rowId) && segmentValue === "lists" && (
                      <tr>
                        <td colSpan={displayedColumns.length} className="bg-gray-50 px-6 py-2">
                          <ChildPendingDataTable pendings={row.pending || []} />
                        </td>
                      </tr>
                    )}
                  </Fragment>
                );
              })
            ) : (
              <tr>
                <td colSpan={displayedColumns.length} className="text-center py-8 text-gray-500">
                  No data found
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>
     </div>

      {/* Pagination */}
      <div className="flex flex-col sm:flex-row justify-between items-center gap-3 p-1 border-t border-gray-200 text-sm bg-gray-50">
        <span className="text-gray-600">
          Page {currentPage} of {totalPages}
        </span>
        <div className="flex items-center gap-2 text-gray-700">
          <span>Rows per page:</span>
          <select
            value={perPage}
            onChange={(e) => { setPerPage(Number(e.target.value)); setCurrentPage(1); }}
            className="border border-gray-300 rounded px-3 text-sm bg-white focus:ring-2 focus:ring-blue-500"
          >
            <option value={5}>5</option>
            <option value={10}>10</option>
            <option value={25}>25</option>
          </select>
        </div>
        <div className="flex items-center space-x-2">
          <button
            onClick={() => setCurrentPage((p) => Math.max(1, p - 1))}
            disabled={currentPage === 1}
            className="px-3  border border-gray-300 rounded text-purple-600 hover:bg-purple-100 disabled:opacity-50 cursor-pointer"
          >
            Prev
          </button>
          {getPageNumbers().map((num, idx) =>
            num === "..." ? (
              <span key={idx} className="px-3 text-gray-500 select-none ">
                ...
              </span>
            ) : (
              <button
                key={idx}
                onClick={() => setCurrentPage(num)}
                className={`px-3 border rounded cursor-pointer ${currentPage === num
                  ? "bg-purple-500 text-white border-purple-500"
                  : "border-gray-300 text-gray-700 hover:bg-purple-100"
                  }`}
              >
                {num}
              </button>
            )
          )}
          <button
            onClick={() => setCurrentPage((p) => Math.min(totalPages, p + 1))}
            disabled={currentPage === totalPages}
            className="px-3 border border-gray-300 rounded text-purple-600 hover:bg-purple-100 disabled:opacity-50 cursor-pointer"
          >
            Next
          </button>
        </div>
      </div>
    </div>
  );
}
