import { getToken } from "next-auth/jwt";
import { NextResponse } from "next/server";

// Define paths that are publicly accessible even without a token.
const PUBLIC_PATHS = ["/", "/login"];
// Define paths that are public and start with a specific prefix.
const PUBLIC_PREFIXES = ["/copyediting/accept"]; // <-- ADDED THIS

export async function middleware(req) {
  const { pathname } = req.nextUrl;

  // 1. Skip Next.js internals, static assets, API, favicon, etc.
  if (
    pathname.startsWith("/_next") ||
    pathname.startsWith("/static") ||
    pathname.startsWith("/favicon.ico") ||
    pathname.startsWith("/api") ||
    pathname.match(/\.(ico|js|css|png|jpg|jpeg|svg|gif|woff|woff2|ttf|map)$/)
  ) {
    return NextResponse.next();
  }

  // 2. Check if the path is explicitly public or starts with a public prefix
  const isPublicPath = PUBLIC_PATHS.includes(pathname);
  
  // Check if the pathname starts with any of the defined public prefixes
  const isPublicPrefixPath = PUBLIC_PREFIXES.some(prefix => 
      pathname.startsWith(prefix)
  );

  const isProtectedPath = !isPublicPath && !isPublicPrefixPath; // <-- Use the combined check here

  // Get token from NextAuth
  const token = await getToken({
    req,
    secret: process.env.NEXTAUTH_SECRET,
  });

  // 3. Redirection Logic

  // If user is logged in (has token) and tries to visit a *purely* public path (like / or /login), redirect to dashboard
  if (token && isPublicPath) {
    // Note: We only redirect away from / or /login. We allow logged-in users to view /copyediting/accept 
    // in case they follow a link while already logged in.
    return NextResponse.redirect(new URL("/dashboard", req.url));
  }

  // If user is NOT logged in and tries to visit a protected page, redirect to home (or login)
  if (!token && isProtectedPath) {
    return NextResponse.redirect(new URL("/", req.url));
  }
  
  // 4. Allow access (if token exists, or if path is public/public prefix)
  return NextResponse.next();
}