// services/userService.js
"use server";
import prisma from '@/lib/prismaconfig';

// Get all active users
export const getAllUsers = async () => {
  const users = await prisma.acs_users.findMany({
    include: {
      role: {
        select: {
          role_name: true,
        },
      },
    },
    where: {
      user_is_active: "active", // 👈 filter out deleted users
    },
  });
  return users;
};

export const getAllStaffUsers = async (userId) => {
  const users = await prisma.acs_users.findMany({
    include: {
      role: {
        select: {
          role_name: true,
        },
      },
    },
    where: {
      user_is_active: "active", // 👈 filter out deleted users
      NOT: { user_pid: userId },
    },
  });
  return users;
};

export const getAllRoles = async () => {
  return await prisma.acs_roles.findMany();
};
export const getLogsWithEntriesAndBreaks = async () => {
  const logs = await prisma.acs_login_details.findMany({
    where: {
      log_isdelete: "active",
    },
    include: {
      user: {
        select: {
          user_pid:true,
          user_empid: true,
          user_name: true,
        },
      },
    },
  });

  const entries = await prisma.acs_entry.findMany({
    where: {
      entry_isdelete: "active",
    },
  });

  const breaks = await prisma.acs_break_hours_details.findMany({
    where: {
      bh_isdelete: "active",
    },
  });

  const result = logs.map((log) => {
    // filter entries for same user + same date
    const matchedEntries = entries.filter(
      (entry) =>
        entry.entry_createdBy === log.log_createdBy &&
        entry.entry_date === log.log_date
    );

    // filter breaks for same user + same date
    const matchedBreaks = breaks.filter(
      (bh) =>
        bh.bh_createdBy === log.log_createdBy &&
        bh.bh_startDate === log.log_date
    );

    // sum entry durations
    const entryDurationSum = matchedEntries.reduce(
      (sum, e) => sum + parseDuration(e.entry_duration || "00:00:00"),
      0
    );

    // sum break durations
    const breakDurationSum = matchedBreaks.reduce(
      (sum, b) => sum + parseDuration(b.bh_duration || "00:00:00"),
      0
    );

    const logDuration = parseDuration(log.log_duration || "00:00:00");
    const totalSeconds = logDuration - (entryDurationSum + breakDurationSum);

    return {
      log_date: log.log_date,
      user_pid: log.user?.user_pid,
      user_empid: log.user?.user_empid,
      user_name: log.user?.user_name,
      log_in: log.log_startTime,
      log_out: log.log_endTime,
      log_duration: formatDuration(logDuration),
      entry_duration: formatDuration(entryDurationSum),
      break_duration: formatDuration(breakDurationSum),
      total_duration: formatDuration(totalSeconds),
    };
  });

  return result;
};

// Convert "HH:mm:ss" → total seconds
function parseDuration(durationStr) {
  const parts = durationStr.split(":").map(Number);
  const [h = 0, m = 0, s = 0] = parts;
  return h * 3600 + m * 60 + s;
}

// Convert seconds → "HH:mm:ss"
function formatDuration(totalSeconds) {
  const h = Math.floor(totalSeconds / 3600);
  const m = Math.floor((totalSeconds % 3600) / 60);
  const s = totalSeconds % 60;
  return [h, m, s].map((v) => String(v).padStart(2, "0")).join(":");
}


export const getRoleById = async (role_pid) => {
  return await prisma.acs_roles.findUnique({
    where: { role_pid: Number(role_pid) },
  });
};



// Get user by ID
export const getUserById = async (user_pid) => {
  return await prisma.acs_users.findUnique({
    where: { user_pid: Number(user_pid) },
  });
};

// Get user by Email
export const getUserByEmail = async (user_email) => {
  return await prisma.acs_users.findUnique({
    include: {
      role: {
        select: {
          role_name: true,
        },
      },
    },
    where: { user_email },
  });

};

// Get user by Employee ID
export const getUserByEmpid = async (user_empid) => {
  return await prisma.acs_users.findUnique({
    where: { user_empid },
  });
};

export const createUser = async (data) => {
  return await prisma.acs_users.create({
    data,
  });
};

// Update user
export const updateUser = async (user_pid, data) => {
  return await prisma.acs_users.update({
    where: { user_pid: Number(user_pid) },
    data,
  });
};

// Delete user
export async function deleteUser(user_pid) {
  console.log("Deactivating user with ID:", user_pid);

  const user = await prisma.acs_users.update({
    where: { user_pid: Number(user_pid) },
    data: { user_is_active: "inactive" },
  });

  return user;
}


export const userLog = async (data) => {
  return await prisma.acs_login_details.create({
    data,
  });
}

export const getuserLogByEmpId = async (user_pid, log_date) => {
  return await prisma.acs_login_details.findFirst({
    where: { log_createdBy: user_pid, log_date: log_date },
  });
};

export const updateLogoutTime = async (user_pid, endDate, endTime, duration) => {
  try {
    const updatedLogs = await prisma.acs_login_details.updateMany({
      where: {
        log_createdBy: user_pid,
        log_date: endDate,
        log_isdelete: "active",
      },
      data: {
        log_endTime: endTime,
        log_duration: duration,
      },
    });
    return updatedLogs;
  } catch (error) {
    console.error("Error updating logout time:", error);
    throw error;
  }
};



export const getAllBreaks = async () => {
  const users = await prisma.acs_break_hours_details.findMany({
    include: {
      break: {
        select: {
          break_name: true,
        },
      },
      user: {
        select: {
          user_name: true,
        },
      },
    },
    where: {
      bh_isdelete: "active", // 👈 filter out deleted users
    },
  });
  return users;
};

export const getProjectUser = async () => {
  return await prisma.acs_users.findMany({
    where: {
      user_is_active: "active",
      user_role: 3 ,
    },
  });
};

export const getAllUsersByRole = async (role_id) => {
   const whereCondition =
      role_id === 1 || role_id === 2
      ? { user_is_active: "active" } // All active users
      : { user_role: role_id, user_is_active: "active" };
  const users = await prisma.acs_users.findMany({
    where: whereCondition,
    include: {
      role: {
        select: {
          role_name: true,
        },
      },
    },
  });

  return users;
};
