// services/chapterService.js
import prisma from '@/lib/prismaconfig';

// Get all chapters
// export const getAllChapters = async () => {
//   return await prisma.acs_chapters.findMany();
// };
export const getAllChapters = async () => {
  return await prisma.acs_chapters.findMany({
    include: {
      book: {
        select: {
          book_title: true,
          book_year: true,
          book_pid: true,
        },
      },
      pstatus: {
        select: {
          pstatus_name: true,
        },
      },
      user: {
        select: {
          user_name: true,
        },
      },
    },
    where: {
      chap_isdelete: "active", // 👈 filter out deleted chapter
    },
  });
};

export const getAllManuScript = async () => {
  return await prisma.acs_chapters.groupBy({
      by: ['chap_manuscriptID','chap_bookTitle'],
      where: {
        chap_isdelete: 'active',
      },
      _count: {
        _all: true,
      },
  });
};


export const getBookIdChapters = async (book_pid) => {
  //console.log(book_pid, 'Book ID in service');
  return await prisma.acs_chapters.findMany({
    include: {
      book: {
        select: {
          book_title: true,
        },
      },
      pstatus: {
        select: {
          pstatus_name: true,
        },
      },
    },
    where: {
      chap_isdelete: "active", // 👈 filter out deleted chapter
      chap_bookTitle : Number(book_pid)
    },
  });
};
// Create new chapter
// export const createChapter = async (data) => {
//   return await prisma.acs_chapters.create({
//     data,
//   });
// };

export async function createChapter(chapterData) {
  // console.log("Creating chapter with data:", chapterData);
  return await prisma.acs_chapters.create({
    data: chapterData,
    include: {
      book: true, // <-- Include related book
    },
  });
}

// Get chapter by ID
export const getChapterById = async (chap_pid) => {
  return await prisma.acs_chapters.findUnique({
    where: { chap_pid: Number(chap_pid) },
  });
};

export const fetchChapterDetailsforQC = async (chapterId) => {
  return await prisma.acs_stage_movements.findFirst({
    where: { stage_chapId: Number(chapterId) },
    orderBy: {
      stage_createdAt: 'desc', // Assumes latest record by creation date
    },
    include: {
      chapter: {
        select: {
          chap_title: true,
          chap_manuscriptID: true,
          chap_currentStage: true,
        },
      },
      book: {
        select: {
          book_title: true,
        },
      },
    },
  });
};


// Update chapter
export const updateChapter = async (chap_pid, data) => {
  return await prisma.acs_chapters.update({
    where: { chap_pid: Number(chap_pid) },
    data,
  });
};

export async function getChaptersByBookId(bookId) {
  return await prisma.acs_chapters.findMany({
    where: {
      chap_bookTitle : Number(bookId),
      chap_currentStage: {
        in: ['FC', 'FCR', 'RCR1', 'RCR2'], // This filters for either 'FC' or 'FCR'
      },
    },
    select: {
      chap_title: true,
      chap_pid: true,
    },
  });
}

export async function getFilterChaptersByBookId(bookId) {
  return await prisma.acs_chapters.findMany({
    where: {
      chap_bookTitle : Number(bookId),
    },
    select: {
      chap_title: true,
      chap_pid: true,
    },
  });
}

export const getAllPStatus = async () => {
  return await prisma.acs_pstatus.findMany({
    where: {
      pstatus_isdelete: "active", // 👈 filter out deleted books
    },
  });
};



export const getChaptersStages = async (status) => {
  // Fetch chapters with last stage movement
  return await prisma.acs_chapters.findMany({
    include: {
      book: {
        select: {
          book_title: true,
        },
      },
      pstatus: {
        select: {
          pstatus_name: true,
        },
      },     
       stage_movements: {
        orderBy: {
          stage_pid: "desc", // latest first
        },
        take: 1, // only the latest stage movement
        select: {
          stage_pid: true,
          stage_notes: true,
          stage_uploadFile: true,
          stage_status: true,
          stage_dispachDate: true,
          stage_from: true,
          stage_to: true,
          stage_createdBy: true,
          status: {                 // join status table
            select: {
              status_name: true
            }
          },
          entry: {
            select: {
              entry_email: true,
            }
          },
          user: {
            select:{
              user_role: true,
              user_name: true,
            }
          },
        },
      },
    },
    where: {
      chap_currentStage: {
        in: status,
      },
      chap_isdelete: "active",
    },
  });
};

export async function getTaskById(taskId) {
  return await prisma.acs_task.findMany({
    where: {
      task_pid : Number(taskId),
      task_isdelete : "active"
    },
    select: {
      task_name: true,
      task_pid: true,
    },
  });
};

export const getAllChaptersByCE = async () => {
  return await prisma.acs_chapters.findMany({
    where: {
      chap_copyEditing: "yes",
      chap_copyEditingStatus: 1,
      chap_isdelete: "active",
    },
    include: {
      book: {
        where: {
          book_isdelete: "active",
        },
        select: {
          book_pid: true,
          book_title: true,
          book_createdAt:true,
        },
      },
       ce_editing: {
         take: 1, // 👈 Only one record
        orderBy: {
          ce_pid: 'desc', // 👈 Or use ce_createdAt: 'desc' if you have timestamp
        },
        include: {
          user: {
            select: {
              user_name: true, // CE assigned user name
            },
          },
        },
      },
    },
  });
};

export const getAllCEStatus = async () => {
  // Fetch chapters with last stage movement
  return await prisma.acs_chapters.findMany({
    include: {
      book: {
        select: {
          book_title: true,
          book_pid: true,
        },
      },
      pstatus: {
        select: {
          pstatus_name: true,
        },
      },     
       stage_movements: {
        orderBy: {
          stage_pid: "desc", // latest first
        },
        take: 1, // only the latest stage movement
        select: {
          stage_pid: true,
          stage_notes: true,
          stage_uploadFile: true,
          stage_status: true,
          stage_dispachDate: true,
          stage_from: true,
          stage_to: true,
          stage_createdBy: true,
          status: {                 // join status table
            select: {
              status_name: true
            }
          },
          entry: {
            select: {
              entry_email: true,
            }
          },
          user: {
            select:{
              user_role: true,
              user_name: true,
              user_pid : true,
            }
          },
        },
      },
      ce_editing: {
        select: {
          ce_editor_id: true,
          ce_sent_comments: true,
          ce_received_comment: true,
          ce_received_date: true,
          ce_sent_date: true,
          ce_status: true,
            user: {
            select: {
              user_name: true,
              user_email: true,
              user_role: true,
            },
          },
        },
        where: {
          ce_isdelete: "active",
        },
        orderBy: {
          ce_pid: "desc",
        },
        take: 1,
      },
    },
    where: {
      chap_isdelete: "active",
      chap_currentStage: { in: ["CE"] },
      chap_copyEditing: { equals: "yes", not: "" },
      chap_copyEditingStatus: { not: null },
    },
  });
};

export const selectChapterList = async (chapter_id) => {
  return await prisma.acs_chapters.findMany({
    include: {
      book: {
        select: {
          book_title: true,
          book_pid: true,
        },
      },
    },
    where: {
      chap_isdelete: "active",
      chap_pid: chapter_id ,
    },
  });
};









