// services/bookService.js
import prisma from '@/lib/prismaconfig';

// Get all books

export const getAllBooks = async () => {
  return await prisma.acs_books.findMany({
    where: {
      book_isdelete: "active",

    },
    include: {
      user: {
        select: {
          user_name: true, // book_updatedBy user
        },
      },
      chapters: {
        include: {
          user: {
            select: {
              user_name: true, // chap_updatedBy user
            },
          },
        },
      },
      pm_user: {
        select: {
          user_name: true, // book projectmanager
        },
      },
    },
  });
};

export const getAllBooksList = async (user_pid) => {
  // Get books where the user is Project Manager
  const projectManagerBooks = await prisma.acs_books.findMany({
    where: {
      book_isdelete: "active",
      book_ProjectManager: user_pid,
    },
    include: {
      user: {
        select: { user_name: true },
      },
      chapters: {
        include: {
          user: { select: { user_name: true } },
        },
      },
      pm_user: {
        select: { user_name: true },
      },
      alt_pm_user: {
        select: { user_name: true },
      },
    },
  });

  // Get books where the user is Alternate Project Manager
  const altProjectManagerBooks = await prisma.acs_books.findMany({
    where: {
      book_isdelete: "active",
      book_AltProjectManager: user_pid,
    },
    include: {
      user: {
        select: { user_name: true },
      },
      chapters: {
        include: {
          user: { select: { user_name: true } },
        },
      },
      pm_user: {
        select: { user_name: true },
      },
      alt_pm_user: {
        select: { user_name: true },
      },
    },
  });

  // Combine both sets in one structured response
  return {
    projectManagerBooks,
    altProjectManagerBooks,
  };
};

// Create new book
export const createBook = async (data) => {
  return await prisma.acs_books.create({
    data,
  });
};

// Get book by ID
export const getBookById = async (book_pid) => {
  return await prisma.acs_books.findUnique({
    where: { book_pid: Number(book_pid) },
  });
};

// Update book
export const updateBook = async (book_pid, data) => {
  return await prisma.acs_books.update({
    where: { book_pid: Number(book_pid) },
    data,
  });
};

// get book by id wise chapters here
export const getBookIdChapters = async (book_pid) => {
  return await prisma.$queryRaw`
  SELECT 
  acs_books.book_pid,
  acs_books.book_year,
  acs_books.book_title,
  acs_books.book_LeadEditorName,
  acs_books.book_CustomId,
  acs_books.book_OnlineIsbn,
  acs_books.book_AuthorInvitationsSent,
  acs_books.book_AuInvited,
  acs_books.book_AuAccepted,
  acs_books.book_ChapDueDate,
  acs_books.book_DropDeadDate,
  acs_books.book_ChSubmitted,
  acs_books.book_ChAccepted,
  acs_books.book_LastDueProduction,
  acs_books.book_ProjectManager,
  acs_books.book_AltProjectManager,
  acs_books.book_NoteComments,
  acs_books.book_createdAt,
  acs_books.book_updatedAt,
  acs_books.book_isdelete,

  acs_chapters.chap_pid,
  acs_chapters.chap_bookTitle,
  acs_chapters.chap_manuscriptID,
  acs_chapters.chap_title,
  acs_chapters.chap_firstName,
  acs_chapters.chap_lastName,
  acs_chapters.chap_corresAuEmail,
  acs_chapters.chap_status,
  acs_chapters.chap_pStatus,
  acs_chapters.chap_pdueDate,
  acs_chapters.chap_copyEditing,
  acs_chapters.chap_comments,
  acs_chapters.chap_createdAt,
  acs_chapters.chap_updatedAt,
  acs_chapters.chap_isdelete

FROM acs_books
LEFT JOIN acs_chapters
  ON acs_books.book_pid = acs_chapters.chap_bookTitle
  WHERE acs_books.book_pid = ${book_pid}
  `
}
export const getAllBookYear = async () => {
  return await prisma.acs_books.findMany({
    where: {
      book_isdelete: "active",
    },
    distinct: ['book_year'], // ✅ unique years only
    select: {
      book_title: true,
      book_year: true,        // ✅ only the year column
      book_pid: true,
    },
    orderBy: {
      book_year: 'asc',       // ✅ ascending order (2022 → 2025)
    },
  });
};

