// lib/filecompressed/imageCompressor.js
import JSZip from "jszip";
import sharp from "sharp";
import path from "path";
import { writeFileSyncAtomic } from "./utils.js";

/**
 * Compress all embedded images in a DOCX to 96 ppi equivalent.
 * Aggressively converts all media to JPEG (quality ~40) and scales down dimensions.
 */
export async function compressDocxImages(inputBuffer, outputDocxPath, options = {}) {
  const quality = options.quality ?? 40;   // JPEG quality
  const targetDPI = options.targetDPI ?? 96; // target PPI
  const assumeSourceDPI = options.assumeSourceDPI ?? 300; // typical print image

  const zip = await JSZip.loadAsync(inputBuffer);
  const mediaFiles = Object.keys(zip.files).filter(p => p.startsWith("word/media/"));

  for (const mediaPath of mediaFiles) {
    try {
      const ext = path.extname(mediaPath).toLowerCase();
      if (!['.jpg', '.jpeg', '.png', '.gif', '.tif', '.tiff', '.bmp'].includes(ext)) continue;

      const original = await zip.files[mediaPath].async("nodebuffer");
      let meta = await sharp(original).metadata();

      // compute scale factor for 96 ppi equivalent
      const srcDPI = meta.density || assumeSourceDPI;
      const scale = targetDPI / srcDPI; // <1 = shrink

      const newWidth = Math.round((meta.width || 1000) * scale);
      const newHeight = Math.round((meta.height || 1000) * scale);

      // aggressively convert to JPEG @96 ppi
      const processed = await sharp(original, { density: targetDPI })
        .resize({ width: newWidth, height: newHeight, withoutEnlargement: true })
        .jpeg({ quality, mozjpeg: true })
        .toBuffer();

      zip.file(mediaPath, processed);
      console.log(`✔ Compressed ${mediaPath}: ${meta.width}×${meta.height} → ${newWidth}×${newHeight} @96 ppi`);
    } catch (err) {
      console.warn(`⚠ Skipped ${mediaPath}: ${err.message}`);
    }
  }

  const newBuffer = await zip.generateAsync({ type: "nodebuffer", compression: "DEFLATE" });
  writeFileSyncAtomic(outputDocxPath, newBuffer);
  return outputDocxPath;
}
