import fs from "fs";
import path from "path";
import { getUploadDir, getCompressedDir } from "./utils.js";
import { compressDocxImages } from "./imageCompressor.js";

export async function compressDocFile(file, dpi = "96") {
  const uploadDir = getUploadDir();
  const compressedDir = getCompressedDir();

  const originalFileName = file.name;
  const fileExtension = path.extname(originalFileName).toLowerCase();
  const baseName = path.parse(originalFileName).name;

  const prefixedFileName = `compressed_${dpi}ppi_${baseName}${fileExtension}`;

  const inputPath = path.join(uploadDir, originalFileName);
  const outputPath = path.join(compressedDir, prefixedFileName);

  const arrayBuffer = await file.arrayBuffer();
  const inputBuffer = Buffer.from(arrayBuffer);
  fs.writeFileSync(inputPath, inputBuffer);

  let finalOutputPath;

  if (fileExtension === ".docx") {
    // Convert dpi → approximate width reduction & quality
    const dpiMap = {
      "220": { quality: 80, maxWidth: 1920 },
      "150": { quality: 60, maxWidth: 1280 },
      "96": { quality: 40, maxWidth: 960 },
    };
    const settings = dpiMap[dpi] || dpiMap["96"];

    finalOutputPath = await compressDocxImages(inputBuffer, outputPath, settings);
  } else {
    fs.unlinkSync(inputPath);
    return {
      success: false,
      message: `Compression not supported for ${fileExtension}`,
    };
  }

  const statsBefore = fs.statSync(inputPath).size;
  const statsAfter = fs.statSync(finalOutputPath).size;
  fs.unlinkSync(inputPath);

  // 🧹 Auto delete compressed file after short delay
  setTimeout(() => {
    if (fs.existsSync(finalOutputPath)) {
      fs.unlinkSync(finalOutputPath);
      console.log("🧹 Auto-deleted compressed file:", finalOutputPath);
    }
  }, 10000); // 10 seconds after download

  return {
    success: true,
    originalSize: (statsBefore / 1024 / 1024).toFixed(2) + " MB",
    compressedSize: (statsAfter / 1024 / 1024).toFixed(2) + " MB",
    outputFile: prefixedFileName,
  };
}
