"use client";

import { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiX } from "react-icons/fi";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { pendingSchema } from "@/validation/validation"; // Your Zod schema
import { toast } from "react-toastify";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { createPending } from "@/lib/actions/entryActions";
import { useSession } from "next-auth/react";
export default function PendingModal({ isOpen, onClose, onCancel, entryId, onSuccess  }) {
   const { data: session, status } = useSession(); 
  const [loading, setLoading] = useState(false);
  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm({
    resolver: zodResolver(pendingSchema),
    mode: "onBlur",
    defaultValues: {
      pending_reason: "",
    },
  });

  const onSubmit = async (data) => {
    setLoading(true);
    try {
       const now = new Date();
      const payload = {
        pr_pendingDate    : now.toISOString().split("T")[0],
        pr_entry_pid      : Number(entryId),
        pr_pendingTime    : now.toTimeString().split(" ")[0],
        pr_createdBy      : session.user.user_pid,
        pr_pendingReason  : data.pending_reason
      };
      const pending_res = await createPending(payload);
      if (pending_res?.success && pending_res.pendingEntry?.pr_pid) {
        toast.success(`Pending entry saved!`);
        // Success logic: close modal and reset form
        reset();
        onClose();
        if (onSuccess) onSuccess(pending_res.pendingEntry.pr_pid);
      } else {
        toast.error(start_result?.message || "Failed to create entry.");
      }
      } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <AnimatePresence>
      {isOpen && (
        <motion.div
          className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
        >
          <motion.div
            className="bg-white rounded-lg shadow-xl w-full max-w-md overflow-hidden"
            initial={{ scale: 0.9, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            exit={{ scale: 0.9, opacity: 0 }}
            transition={{ type: "spring", stiffness: 300, damping: 25 }}
          >
            {/* Header */}
            <div className="flex items-center justify-between px-5 py-4 border-b border-blue-200 bg-blue-50">
              <h3 className="text-lg font-semibold text-blue-800">
                Pending Entry
              </h3>
              <button
                onClick={() => {
                  onClose();
                  reset();
                }}
                className="text-gray-500 hover:text-gray-700 transition cursor-pointer"
                aria-label="Close modal"
              >
                <FiX size={20} />
              </button>
            </div>

            {/* Form (Wraps body & footer) */}
            <form onSubmit={handleSubmit(onSubmit)}>
              {/* Body */}
              <div className="px-5 py-4 text-sm text-gray-700 space-y-3">
                <div>
                  <label
                    htmlFor="pending_reason"
                    className="block text-sm font-medium text-gray-700 mb-1"
                  >
                    Pending reason <span className="text-red-500">*</span>
                  </label>
                  <textarea
                    id="pending_reason"
                    placeholder="Enter pending reason"
                    rows={4}
                    {...register("pending_reason")}
                    className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${
                      errors.pending_reason
                        ? "border-red-500"
                        : "border-gray-300"
                    } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                  />
                  {errors.pending_reason && (
                    <p className="text-sm text-red-500 mt-1">
                      {errors.pending_reason.message}
                    </p>
                  )}
                </div>
              </div>

              {/* Footer */}
              <div className="flex justify-end gap-2 px-5 py-3 bg-gray-50 border-t border-gray-200">
                <button
                  type="button"
                  onClick={() => {
                    onClose();
                    reset();
                  }}
                  className="w-full sm:w-auto px-5 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  className="w-full sm:w-auto px-5 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2"
                  disabled={loading}
                >
                  <span>{loading ? "Updating..." : "Submit"}</span>
                  {loading ? (
                    <FiLoader size={20} className="animate-spin" />
                  ) : (
                    <FiArrowRightCircle size={20} />
                  )}
                </button>
              </div>
            </form>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
}
