// import React from "react";
import { useState, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiX, FiBookOpen } from "react-icons/fi";
import { fetchPendingEntriesAction } from "@/lib/actions/entryActions";

export default function PendingModal({ isOpen, onClose, selectedRow, entryId }) {
  console.log(entryId);
  if (!selectedRow) return null;
  const [pendingData, setPendingData] = useState(null);

  useEffect(() => {
    if (!entryId) return;

    async function fetchPendingEntries() {
      try {
        const pending_res = await fetchPendingEntriesAction(entryId); // ✅ directly call server action
        if (pending_res?.success) {
          console.log(pending_res.result)
          setPendingData(pending_res.result);
        } else {
          toast.error('chapter fetching error')
        }
      } catch (err) {
        console.error(err);
      }
    }
    fetchPendingEntries();
  }, [entryId]);

  return (
    <AnimatePresence>
      {isOpen && (
        <motion.div
          className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
        >
          <motion.div
            className="bg-white rounded-lg shadow-xl w-full max-w-3xl overflow-hidden"
            initial={{ scale: 0.9, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            exit={{ scale: 0.9, opacity: 0 }}
            transition={{ type: "spring", stiffness: 300, damping: 25 }}
          >
            {/* Header */}
            <div className="flex items-center justify-between px-5 py-4 border-b border-blue-200 bg-blue-50">
              <h3 className="text-lg font-semibold text-blue-800 flex items-center gap-2">
                <FiBookOpen size={20} className="text-blue-600" />
                View Entry Details
              </h3>
              <button
                onClick={() => {
                  onClose();
                  // Assuming reset is defined elsewhere, if not remove it
                  // reset();
                }}
                className="text-gray-500 hover:text-gray-700 transition cursor-pointer"
                aria-label="Close modal"
              >
                <FiX size={20} />
              </button>
            </div>

            {/* Body */}
            <div className="px-5 py-4 text-sm text-gray-700 space-y-3 max-h-[90vh] overflow-y-auto">
              <table className="min-w-full table-auto border border-gray-100 text-sm">
                <tbody className="text-gray-700">
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                      Start Date
                    </td>
                    <td className="px-4 py-2">
                      {selectedRow.entry_date}
                    </td>
                  </tr>

                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Book Title
                    </td>
                    <td className="px-4 py-2">
                      {selectedRow.book?.book_title || "—"}
                    </td>
                  </tr>
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Chapter Name
                    </td>
                    <td className="px-4 py-2">
                      {selectedRow.chapter?.chap_title || "—"}
                    </td>
                  </tr>
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Task Name
                    </td>
                    <td className="px-4 py-2">
                      {selectedRow.task?.task_name || "—"}
                    </td>
                  </tr>
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Start Time
                    </td>
                    <td className="px-4 py-2">
                      {selectedRow.entry_start_time}
                    </td>
                  </tr>
                 {pendingData?.length > 0 && (
                    <tr className="border-b border-gray-100">
                      <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                        pending / Resume Details
                      </td>
                      <td className="px-4 py-2">
                        <table className="min-w-full table-auto border border-gray-100 text-sm">
                          <thead className="text-gray-700">
                            <tr>
                              <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                                pending Date / Time
                              </td>
                              <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                                Resume Date / Time
                              </td>
                              <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                                Duration
                              </td>

                            </tr>
                          </thead>
                          <tbody className="text-gray-700">
                            {pendingData.map((item, index) => (
                              <tr key={index} className="border-t border-gray-100">
                                <td className="px-4 py-2 border-r border-gray-100">
                                  <p className="text-red-600">{item.pr_pendingDate || "—"}</p>
                                  <p className="text-red-600">{item.pr_pendingTime}</p>
                                </td>
                                <td className="px-4 py-2 border-r border-gray-100">
                                  <p className="text-green-600">{item.pr_resumeDate || "—"}</p>
                                  <p className="text-green-600">{item.pr_resumeTime}</p>
                                </td>
                                <td className="px-4 py-2">
                                  <p className="text-blue-600">{item.pr_duration || "—"}</p>
                                </td>
                              </tr>
                            ))}
                            
                          </tbody>
                        </table>
                      </td>
                    </tr>
                  )}
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      End Date
                    </td>
                    <td className="px-4 py-2">{selectedRow.entry_end_date}</td>
                  </tr>
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      End Time
                    </td>
                    <td className="px-4 py-2">{selectedRow.entry_end_time}</td>
                  </tr>
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Duration
                    </td>
                    <td className="px-4 py-2">{selectedRow.entry_duration}</td>
                  </tr>
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Comments
                    </td>
                    <td className="px-4 py-2">{selectedRow.entry_comments}</td>
                  </tr>
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Contact Email
                    </td>
                    <td className="px-4 py-2"> {selectedRow.qcUser?.user_email || "—"}</td>
                  </tr>
                  <tr className="border-b border-gray-100">
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Upload File
                    </td>
                    <td className="px-4 py-2">
                      {selectedRow.entry_file ? (
                        <a
                          href={`/First_check/${selectedRow.entry_file}`}
                          download={selectedRow.entry_file}
                          rel="noopener noreferrer"
                          className="text-blue-600 hover:underline"
                        >
                          {selectedRow.entry_file}
                        </a>
                      ) : (
                        "No file uploaded"
                      )}
                    </td>
                  </tr>
                  <tr>
                    <td className="font-medium px-4 py-2 bg-gray-50 border-r border-gray-100 font-semibold">
                      Status
                    </td>
                    <td className="px-4 py-2">
                      <span
                        className={`inline-flex items-center gap-2 font-semibold ${selectedRow.status?.status_name === "Unsubmit"
                          ? "text-red-600"
                          : "text-green-600"
                          }`}
                      >
                        {selectedRow.status?.status_name}
                      </span>
                    </td>
                  </tr>
                </tbody>
              </table>
            </div>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
}
