"use client";

import { useEffect, useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiX, FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { ceSchema } from "@/validation/validation"; // ✅ Your Zod schema
import { toast } from "react-toastify";
import { CEUpdate } from "@/lib/actions/copyeditingAction";
import { useSession } from "next-auth/react";
import { useRouter } from "next/navigation";
import Select from "react-select";

export default function CEReviewModal({
   isOpen,
  onClose,
  onCancel,
  chapterId,
  initialstatus,
  editorList,
}) {
  const { data: session } = useSession();
  const [loading, setLoading] = useState(false);
  const router = useRouter();

  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
    watch,
    setValue,
  } = useForm({
    resolver: zodResolver(ceSchema),
    mode: "onBlur",
    defaultValues: {
      ce_book_id: "",
      ce_book_title: "",
      ce_chapter_id: "",
      ce_chapter_title: "",
      ce_file_name: "",
      ce_emp_id: "",
      ce_due_date: "",
      ce_notes: "",
      ce_comments: "",
      ce_sent_file: "",
    },
  });

  // ✅ When modal opens or chapterId changes, load that chapter’s data
  useEffect(() => {
    if (isOpen && initialstatus.length > 0 && chapterId) {
      const chapterData = initialstatus.find(
        (item) => Number(item.chap_pid) === Number(chapterId)
      );

      if (chapterData) {
        // ✅ Set values dynamically based on selected chapter
        setValue("ce_book_id", chapterData.book.book_pid || "");
        setValue("ce_book_title", chapterData.book.book_title || "");
        setValue("ce_chapter_id", chapterData.chap_pid || "");
        setValue("ce_chapter_title", chapterData.chap_title || "");
        setValue("ce_file_name", chapterData.stage_movements?.[0]?.stage_uploadFile || "");
        setValue("ce_due_date", chapterData.chap_dueDate || "");
        setValue("ce_notes", chapterData.chap_notes || "");
        setValue("ce_comments", chapterData.chap_comments || "");
        setValue("ce_sent_file", chapterData.stage_movements?.[0]?.stage_uploadFile || "");
      } else {
        // ✅ Reset form if no data found for this chapter
        reset();
      }
    }
  }, [isOpen, chapterId, initialstatus, reset, setValue]);

  // ✅ Build the editor list dropdown
  const selectEditorList = editorList.map((s) => ({
    value: String(s.user_pid),
    label: s.user_email.charAt(0).toUpperCase() + s.user_email.slice(1),
  }));

  // ✅ Submit handler
  const onSubmit = async (data) => {
    setLoading(true);
    try {
      // Find the current chapter data again for reference
      const chapterData = initialstatus.find(
        (item) => Number(item.chap_pid) === Number(chapterId)
      );

      const payload = {
        ce_book_id: Number(chapterData?.chap_bookTitle || 0),
        ce_chap_id: Number(chapterId),
        ce_editor_id: Number(data.ce_emp_id),
        ce_sent_date: new Date().toISOString().split("T")[0],
        ce_due_date: String(data.ce_due_date),
        ce_sent_note: String(data.ce_notes),
        ce_sent_comments: String(data.ce_comments),
        ce_status: 1,
        ce_sent_file: String(chapterData.stage_movements?.[0]?.stage_uploadFile || ""),
      };
      
      const eda_res = await CEUpdate(payload);
      if (eda_res?.success) {
        toast.success("CE Assign File successfully!");
        reset();
        onClose();
        router.refresh();
      } else {
        toast.error(eda_res?.message || "Failed to CE Assign File.");
      }
    } catch (error) {
      console.error(error);
      toast.error(error.message || "Unexpected error occurred");
    } finally {
      setLoading(false);
    }
  };

  return (
    <AnimatePresence>
      {isOpen && (
        <motion.div
          className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
        >
          <motion.div
            className="bg-white rounded-lg shadow-xl w-[600px]  overflow-hidden"
            initial={{ scale: 0.9, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            exit={{ scale: 0.9, opacity: 0 }}
            transition={{ type: "spring", stiffness: 300, damping: 25 }}
          >
            {/* Header */}
            <div className="flex items-center justify-between px-6 py-2 border-b border-blue-200 bg-blue-50" >
              <h3 className="text-lg font-semibold text-blue-800">
                Assign File to Editor
              </h3>
              <button
                onClick={() => {
                  onClose();
                  reset();
                }}
                className="text-gray-500 hover:text-gray-700 transition cursor-pointer"
                aria-label="Close modal"
              >
                <FiX size={20} />
              </button>
            </div>

            {/* Form */}
            <form onSubmit={handleSubmit(onSubmit)}>
              <div className="px-6 py-1 text-sm text-gray-800 space-y-1">
                <table className="min-w-full table-auto border border-gray-100 text-sm">
                  <tbody className="text-gray-700">
                    {/* Book Title */}
                    <tr className="border-b border-gray-100">
                      <td className="font-medium px-4 py-1 bg-gray-50 w-1/3 border-r border-gray-100">
                        Book Title
                      </td>
                      <td className="px-4 py-1">
                        <input
                          type="text"
                          id = "ce_book_id"
                          placeholder="Book Title"
                          value={watch("ce_book_title") || ""}
                          readOnly
                          className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${
                            errors.book_title
                              ? "border-red-500"
                              : "border-gray-300"
                          } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                        />
                        <input
                          type="hidden"
                          {...register("ce_book_id")}
                        />
                      </td>
                    </tr>

                    {/* Chapter Name */}
                    <tr className="border-b border-gray-100">
                      <td className="font-medium px-4 py-1 bg-gray-50 border-r border-gray-100">
                        Chapter Name
                      </td>
                      <td className="px-4 py-1">
                        <input
                          type="text"
                          id = "ce_chap_id"
                          value={watch("ce_chapter_title")}
                          readOnly
                          className="w-full px-4 py-1.5 rounded-md border border-gray-300 focus:outline-none focus:ring-2 focus:ring-blue-500"
                        />
                        <input type="hidden" {...register("ce_chap_id")} />
                      </td>
                    </tr>

                    {/* File Name */}
                    <tr className="border-b border-gray-100">
                      <td className="font-medium px-4 py-1 bg-gray-50 border-r border-gray-100">
                        File Name
                      </td>
                      <td className="px-4 py-1">
                        <span className={`w-full px-2 py-1.5 rounded-md  placeholder:text-sm   focus:ring-2 focus:ring-blue-500`}>{watch("ce_file_name")}</span>
                        
                        <input type="hidden" {...register("ce_file_name")} />
                      </td>
                    </tr>

                    {/* Editor Name */}
                    <tr className="border-b border-gray-100">
                      <td className="font-medium px-4 py-1 bg-gray-50 border-r border-gray-100">
                        Editor Name <span className="text-red-500">*</span>
                      </td>
                      <td className="px-4 py-1">
                        <Controller
                          name="ce_emp_id"
                          id="ce_editor_id"
                          control={control}
                          render={({ field }) => (
                            <Select
                              {...field}
                              instanceId="entry-book-pid-select"
                              options={selectEditorList}
                              onChange={(selected) =>
                                field.onChange(selected ? selected.value : "")
                              }
                              value={
                                selectEditorList.find(
                                  (option) => option.value === field.value
                                ) || null
                              }
                              placeholder="Select an Editor"
                              isClearable
                              styles={{
                                control: (base) => ({
                                  ...base,
                                  borderColor: errors.ce_emp_id
                                    ? "red"
                                    : base.borderColor,
                                }),
                              }}
                            />
                          )}
                        />
                        {errors.ce_emp_id && (
                          <p className="text-sm text-red-500 mt-1">
                            {errors.ce_emp_id.message}
                          </p>
                        )}
                      </td>
                    </tr>

                    {/* Due Date */}
                    <tr className="border-b border-gray-100">
                      <td className="font-medium px-4 py-1 bg-gray-50 border-r border-gray-100">
                        Due Date <span className="text-red-500">*</span>
                      </td>
                      <td className="px-4 py-1">
                        <input
                          type="date"
                          id="ce_due_date"
                          {...register("ce_due_date")}
                          className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm bg-white text-gray-700 ${
                            errors.ce_due_date
                              ? "border-red-500"
                              : "border-gray-300"
                          } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                        />
                        {errors.ce_due_date && (
                          <p className="text-sm text-red-500 mt-1">
                            {errors.ce_due_date.message}
                          </p>
                        )}
                      </td>
                    </tr>

                    {/* Notes */}
                    <tr className="border-b border-gray-100">
                      <td className="font-medium px-4 py-1 bg-gray-50 border-r border-gray-100">
                        Notes
                      </td>
                      <td className="px-4 py-1">
                        <textarea
                          placeholder="Enter Notes"
                          id="ce_notes"
                          rows={4}
                          {...register("ce_notes")}
                          className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${
                            errors.ce_notes
                              ? "border-red-500"
                              : "border-gray-300"
                          } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                        />
                        {errors.ce_notes && (
                          <p className="text-sm text-red-500 mt-1">
                            {errors.ce_notes.message}
                          </p>
                        )}
                      </td>
                    </tr>

                    {/* Comments */}
                    <tr className="border-b border-gray-100">
                      <td className="font-medium px-4 py-1 bg-gray-50 border-r border-gray-100">
                        Comments
                      </td>
                      <td className="px-4 py-1">
                        <textarea
                          placeholder="Enter comments"
                          id="ce_comments"
                          rows={4}
                          {...register("ce_comments")}
                          className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${
                            errors.ce_comments
                              ? "border-red-500"
                              : "border-gray-300"
                          } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                        />
                        {errors.ce_comments && (
                          <p className="text-sm text-red-500 mt-1">
                            {errors.ce_comments.message}
                          </p>
                        )}
                      </td>
                    </tr>
                  </tbody>
                </table>
              </div>

              {/* Footer */}
              <div className="flex justify-end gap-2 px-6 py-2 bg-gray-50 border-t border-gray-200">
                <button
                  type="button"
                  onClick={() => {
                    onClose();
                    reset();
                  }}
                  className="w-full sm:w-auto px-5 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 transition shadow-sm cursor-pointer flex items-center gap-2 text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
                >
                  Cancel
                </button>

                <button
                  type="submit"
                  disabled={loading}
                  className="w-full sm:w-auto px-5 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition shadow-sm cursor-pointer flex items-center gap-2 text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2"
                >
                  <span>{loading ? "Updating..." : "Submit"}</span>
                  {loading ? (
                    <FiLoader size={20} className="animate-spin" />
                  ) : (
                    <FiArrowRightCircle size={20} />
                  )}
                </button>
              </div>
            </form>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
}
