"use client";

import { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiX } from "react-icons/fi";
import { useForm,Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { breakSchema } from "@/validation/validation"; // Your Zod schema
import { toast } from "react-toastify";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { createBreak } from "@/lib/actions/entryActions";
import { useSession } from "next-auth/react";
import Select from "react-select";
export default function TakeBreakModal({ isOpen, onClose, onCancel, initialbreaks, onSuccess  }) {
   const { data: session, status } = useSession(); 
  const [loading, setLoading] = useState(false);
  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
  } = useForm({
    resolver: zodResolver(breakSchema),
    mode: "onBlur",
    defaultValues: {
      entry_break_type: "",
    },
  });

  const onSubmit = async (data) => {
    setLoading(true);
    try {
      console.log(data.entry_break_type);
       const now = new Date();
      const payload = {        
        bh_createdBy : session.user.user_pid,        
        bh_breakId   : Number(data.entry_break_type),
        bh_startDate : now.toISOString().split("T")[0],
        bh_startTime : now.toTimeString().split(" ")[0],
        bh_duration  : null,
        bh_status    : "pending",
      };
      const break_res = await createBreak(payload);
      if (break_res?.success && break_res.createdEntry?.bh_pid) {
        toast.success(`Break started successfully!`);
        // Success logic: close modal and reset form
        reset();
        onClose();
        if (onSuccess) onSuccess(break_res.createdEntry.bh_pid);
      } else {
        toast.error(break_res?.message || "Failed to create entry.");
      }
      } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };
 const breakOptions = initialbreaks.map((b) => ({
        value: String(b.break_pid),
        label: b.break_name.charAt(0).toUpperCase() + b.break_name.slice(1),
    }));
  return (
    <AnimatePresence>
      {isOpen && (
        <motion.div
          className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
        >
          <motion.div
            className="bg-white rounded-lg shadow-xl w-full max-w-md overflow-hidden"
            initial={{ scale: 0.9, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            exit={{ scale: 0.9, opacity: 0 }}
            transition={{ type: "spring", stiffness: 300, damping: 25 }}
          >
            {/* Header */}
            <div className="flex items-center justify-between px-5 py-4 border-b border-blue-200 bg-blue-50">
              <h3 className="text-lg font-semibold text-blue-800">
                Take Break
              </h3>
              <button
                onClick={() => {
                  onClose();
                  reset();
                }}
                className="text-gray-500 hover:text-gray-700 transition cursor-pointer"
                aria-label="Close modal"
              >
                <FiX size={20} />
              </button>
            </div>

            {/* Form (Wraps body & footer) */}
            <form onSubmit={handleSubmit(onSubmit)}>
              {/* Body */}
              <div className="px-5 py-4 text-sm text-gray-700 space-y-3">
                 <div>
                                    <label
                                      htmlFor="entry_break_type"
                                      className="block text-sm font-medium text-gray-700 mb-1"
                                    >
                                      Break Type <span className="text-red-500"></span>
                                    </label>
                                    <Controller
                                      name="entry_break_type"
                                      control={control}
                                      render={({ field }) => (
                                        <Select
                                          {...field}
                                          instanceId="entry-book-pid-select"
                                          options={breakOptions}
                                          onChange={(selected) =>
                                            field.onChange(selected ? selected.value : "")
                                          }
                                          value={
                                            breakOptions.find(
                                              (option) => option.value === field.value
                                            ) || null
                                          }
                                          placeholder="Select a break"
                                          isClearable
                                          inputId="entry_break_type"
                                          styles={{
                                            control: (base) => ({
                                              ...base,
                                              borderColor: errors.entry_break_type
                                                ? "red"
                                                : base.borderColor,
                                            }),
                                          }}
                                        />
                                      )}
                                    />
                                    {errors.entry_break_type && (
                                      <p className="text-sm text-red-500 mt-1">
                                        {errors.entry_break_type.message}
                                      </p>
                                    )}
                                  </div>
              </div>

              {/* Footer */}
              <div className="flex justify-end gap-2 px-5 py-3 bg-gray-50 border-t border-gray-200">
                <button
                  type="button"
                  onClick={() => {
                    onClose();
                    reset();
                  }}
                  className="w-full sm:w-auto px-5 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  className="w-full sm:w-auto px-5 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2"
                  disabled={loading}
                >
                  <span>{loading ? "Updating..." : "Submit"}</span>
                  {loading ? (
                    <FiLoader size={20} className="animate-spin" />
                  ) : (
                    <FiArrowRightCircle size={20} />
                  )}
                </button>
              </div>
            </form>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
}
