"use client"

import { useState, useEffect, useRef } from "react"
import Link from "next/link"
import { usePathname } from "next/navigation"
import { useSession, signOut } from "next-auth/react"
import Image from "next/image"
import { toast } from "react-toastify"
import { FiHome, FiBook, FiUser, FiBookOpen, FiCalendar, FiSettings, FiLock, FiBriefcase, FiEdit, FiFileText, FiLogOut, FiChevronUp } from "react-icons/fi"
import { motion, AnimatePresence } from "framer-motion"

export default function SidebarClient({ links = [] }) {
    const [isExpanded, setIsExpanded] = useState(false);
    const [openMenu, setOpenMenu] = useState(null);
    const [profileDropdownOpen, setProfileDropdownOpen] = useState(false);
    const sidebarRef = useRef(null);
    const pathname = usePathname();
    const { data: session } = useSession();
    const user = session?.user;
console.log(user,"user")
    const iconMap = {
        FiHome, FiBook, FiUser, FiBookOpen, FiCalendar, FiSettings, FiLock, FiBriefcase, FiEdit, FiFileText,
    };

    // Set CSS variable for sidebar width
    useEffect(() => {
        const sidebarWidth = isExpanded ? 240 : 80;
        document.body.style.setProperty('--sidebar-width', `${sidebarWidth}px`);
    }, [isExpanded]);


    const toggleMenu = (label) => {
        setOpenMenu(prev => (prev === label ? null : label));
    };


    useEffect(() => {
        links.forEach(({ label, submenu }) => {
            if (submenu?.some((item) => pathname.startsWith(item.href))) {
                setOpenMenu(label);
            }
        });

        const handleClickOutside = (event) => {
            if (sidebarRef.current && !sidebarRef.current.contains(event.target)) {
                setProfileDropdownOpen(false);
            }
        };

        document.addEventListener("mousedown", handleClickOutside);
        return () => {
            document.removeEventListener("mousedown", handleClickOutside);
        };

    }, [pathname, links]);

    const handleLogout = async () => {
        try {
            fetch('/api/auth/logout', { method: 'POST' }).catch(console.error);
            toast.success('You have been logged out.');
            await signOut({ callbackUrl: '/' });
        } catch (error) {
            console.error("Logout error:", error);
            toast.error("An error occurred during logout.");
        }
    };

    const NavItem = ({ href, label, icon, submenu, isExpanded, onToggle }) => {
        const Icon = icon ? iconMap[icon.trim()] : null;
        const isParentActive = submenu?.some(item => pathname.startsWith(item.href));
        const isActive = (pathname === href && !isParentActive) || isParentActive;

        const commonClasses = "flex items-center h-12 w-full rounded-lg transition-all duration-300 cursor-pointer overflow-hidden";
        const activeClasses = "bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] text-white shadow-lg shadow-pink-500/20";
        const inactiveClasses = "text-gray-200 hover:bg-white/10 hover:text-white";

        const content = (
            <>
                <div className="flex-shrink-0 w-16 h-12 flex items-center justify-center">
                    {Icon && <Icon size={22} />}
                </div>
                <AnimatePresence>
                    {isExpanded && (
                        <motion.span
                            initial={{ opacity: 0, x: -10 }}
                            animate={{ opacity: 1, x: 0 }}
                            exit={{ opacity: 0, x: -10 }}
                            transition={{ duration: 0.2, delay: 0.1 }}
                            className="whitespace-nowrap font-medium text-sm pr-4"
                        >
                            {label}
                        </motion.span>
                    )}
                </AnimatePresence>
            </>
        );

        return (
            <div className="px-2">
                {submenu ? (
                    <button type="button" onClick={onToggle} className={`${commonClasses} ${isActive ? activeClasses : inactiveClasses}`} >
                        {content}
                    </button>
                ) : (
                    <Link href={href} className={`${commonClasses} ${isActive ? activeClasses : inactiveClasses}`}>
                        {content}
                    </Link>
                )}
            </div>
        );
    };

    return (
        <>
            <motion.aside
                onMouseEnter={() => setIsExpanded(true)}
                onMouseLeave={() => setIsExpanded(false)}
                ref={sidebarRef}
                className="fixed top-1/2 left-3 -translate-y-1/2 flex flex-col h-[97vh]  bg-gray-900/80 backdrop-blur-lg rounded-2xl shadow-2xl z-40"
                animate={{ width: isExpanded ? 240 : 80 }}
                transition={{ type: "spring", stiffness: 400, damping: 40 }}
            >
                {/* Logo */}
                <div className="flex-shrink-0 w-full h-20 flex items-center justify-center">
                    <Image src={"/logo.png"} alt="Logo" width={120} height={36} className="object-cover p-3" />
                </div>

                {/* Navigation */}
                <nav className="flex-1 overflow-y-auto overflow-x-hidden hide-scrollbar">
                    <ul className="space-y-2 py-2">
                        {links.map((link) => (
                            <li key={link.label}>
                                <NavItem
                                    {...link}
                                    isExpanded={isExpanded}
                                    onToggle={() => toggleMenu(link.label)}
                                />
                                <AnimatePresence>
                                    {isExpanded && openMenu === link.label && link.submenu && (
                                        <motion.ul
                                            initial={{ height: 0, opacity: 0 }}
                                            animate={{ height: "auto", opacity: 1 }}
                                            exit={{ height: 0, opacity: 0 }}
                                            transition={{ duration: 0.3, ease: "easeInOut" }}
                                            className="ml-8 pl-5 border-l border-white/20 overflow-hidden"
                                        >
                                            {link.submenu.map(subItem => (
                                                <li key={subItem.href} className="py-1">
                                                    <Link href={subItem.href} className={`block w-full px-3 py-1.5 text-sm rounded-md transition-colors ${pathname.startsWith(subItem.href) ? "text-white font-semibold" : "text-gray-200 hover:text-white"}`}>
                                                        {subItem.label}
                                                    </Link>
                                                </li>
                                            ))}
                                        </motion.ul>
                                    )}
                                </AnimatePresence>
                            </li>
                        ))}
                    </ul>
                </nav>

                {/* User Profile */}
                <div className="flex-shrink-0 w-full h-20 flex items-center justify-center px-2">
                    <div className="relative w-full group">
                        <AnimatePresence>
                            {isExpanded && profileDropdownOpen && (
                                <motion.div
                                    initial={{ opacity: 0, y: 10 }}
                                    animate={{ opacity: 1, y: 0 }}
                                    exit={{ opacity: 0, y: 10 }}
                                    transition={{ duration: 0.2 }}
                                    className="absolute bottom-14 left-0 w-full p-2"
                                >
                                    <div className="bg-gray-800 rounded-lg shadow-lg overflow-hidden">
                                        <Link href="/profile" onClick={() => setProfileDropdownOpen(false)} className="flex items-center gap-3 w-full text-left px-4 py-2.5 text-sm text-gray-200 hover:bg-white/10">
                                            <FiUser />
                                            <span>View Profile</span>
                                        </Link>
                                        <button onClick={handleLogout} className="flex items-center gap-3 w-full text-left px-4 py-2.5 text-sm text-gray-200 hover:bg-white/10">
                                            <FiLogOut />
                                            <span>Logout</span>
                                        </button>
                                    </div>
                                </motion.div>
                            )}
                        </AnimatePresence>
                        <button
                            onClick={() => isExpanded && setProfileDropdownOpen(p => !p)}
                            className="flex items-center h-12 w-full rounded-lg text-gray-200 hover:bg-white/10 hover:text-white transition-colors cursor-pointer overflow-hidden"
                        >
                            <div className="flex-shrink-0 w-16 h-12 flex items-center justify-center">
                                {user?.profile ? (
                                    <Image src={`/emp_profile/${user.profile}`} alt="Profile" width={36} height={36} className="rounded-full object-cover w-9 h-9" />
                                ) : (
                                    <div className="w-9 h-9 bg-gray-500 rounded-full flex items-center justify-center">
                                        <span className="font-bold text-sm">{user?.name?.[0]?.toUpperCase() || "U"}</span>
                                    </div>
                                )}
                            </div>
                            <AnimatePresence>
                                {isExpanded && (
                                    <motion.div
                                        initial={{ opacity: 0, x: -10 }}
                                        animate={{ opacity: 1, x: 0 }}
                                        exit={{ opacity: 0, x: -10 }}
                                        transition={{ duration: 0.2, delay: 0.1 }}
                                        className="flex-1 flex items-center justify-between pr-4"
                                    >
                                        <span className="whitespace-nowrap font-medium text-sm">
                                            {user?.name || "User"}
                                        </span>
                                        <FiChevronUp className={`transition-transform duration-300 ${profileDropdownOpen ? 'rotate-180' : ''}`} />
                                    </motion.div>
                                )}
                            </AnimatePresence>
                        </button>
                    </div>
                </div>
            </motion.aside>

        </> // This closing tag is redundant
    )
}
