"use client"

import { useState, useEffect } from "react"
import Link from "next/link"
import { usePathname } from "next/navigation"
import {
    FiHome,
    FiBook,
    FiUser,
    FiBookOpen,
    FiCalendar,
    FiSettings,
    FiMenu,
    FiChevronDown,
    FiChevronRight,
    FiLock,
} from "react-icons/fi"

export default function SidebarClient({ links }) {
    const [collapsed, setCollapsed] = useState(false)
    const [openMenus, setOpenMenus] = useState({})
    const pathname = usePathname()

    const toggleMenu = (label) => {
        setOpenMenus((prev) => ({ ...prev, [label]: !prev[label] }))
    }

    // Auto-open submenu if active
    useEffect(() => {
        links.forEach(({ label, submenu }) => {
            if (submenu && submenu.some((item) => pathname.startsWith(item.href))) {
                setOpenMenus((prev) => ({ ...prev, [label]: true }))
            }
        })
    }, [pathname, links])

    // Icon map
    const iconMap = {
        FiHome,
        FiBook,
        FiUser,
        FiBookOpen,
        FiCalendar,
        FiSettings,
        FiLock,
    }

    return (
        <aside
            className={`flex flex-col h-full transition-width duration-300 bg-gradient-to-b from-gray-900 via-gray-800 to-gray-900 text-gray-200 shadow-xl ${collapsed ? "w-18" : "w-60"
                }`}
        >
            {/* Header */}
            <div className="flex items-center justify-between px-4 py-2 border-b border-gray-700">
                {!collapsed && (
                    <h1 className="text-lg font-bold tracking-wide select-none flex items-center space-x-3">
                        <img
                            src="/ebook-icon.svg"
                            alt="Ebooks"
                            className="w-6 h-6 object-contain"
                            style={{
                                backgroundColor: "#ededed",
                                width: "40px",
                                height: "35px",
                                padding: "2px",
                                borderRadius: "6px",
                            }}
                        />
                        <span>EBOOKS</span>
                    </h1>
                )}
                <button
                    onClick={() => setCollapsed(!collapsed)}
                    aria-label="Toggle sidebar"
                    className="p-2 rounded-md hover:bg-gray-700 transition-colors"
                >
                    <FiMenu size={24} />
                </button>
            </div>

            {/* Navigation */}
            <nav className="flex-1 overflow-y-auto mt-2">
                <ul className="flex flex-col">
                    {links.map(({ href, label, icon, submenu }) => {
                        const Icon = iconMap[icon] || FiMenu
                        const isActive =
                            pathname === href ||
                            (submenu && submenu.some((item) => pathname.startsWith(item.href)))
                        const isOpen = openMenus[label]

                        if (submenu) {
                            return (
                                <li key={label} className="relative">
                                    <button
                                        onClick={() => toggleMenu(label)}
                                        title={collapsed ? label : undefined}
                                        className={`flex items-center w-full px-4 py-2.5 space-x-2 rounded-r-lg transition-colors duration-200 cursor-pointer ${isActive
                                                ? "bg-gray-700 text-indigo-400 border-l-4 border-indigo-400 shadow-lg"
                                                : "hover:border-indigo-600 hover:text-indigo-300 border-l-4 border-transparent"
                                            }`}
                                    >
                                        <Icon size={20} />
                                        {!collapsed && <span className="flex-grow text-left">{label}</span>}
                                        {!collapsed &&
                                            (isOpen ? (
                                                <FiChevronDown size={20} className="text-indigo-400" />
                                            ) : (
                                                <FiChevronRight size={20} className="text-gray-500" />
                                            ))}
                                    </button>

                                    <ul
                                        className={`overflow-hidden transition-[max-height,opacity] duration-300 ease-in-out ${isOpen ? "max-h-60 opacity-100" : "max-h-0 opacity-0"
                                            }`}
                                    >
                                        {submenu.map(({ href: subHref, label: subLabel }) => {
                                            const isSubActive =
                                                pathname === subHref || pathname.startsWith(subHref)
                                            return (
                                                <li key={subHref}>
                                                    <Link
                                                        href={subHref}
                                                        title={collapsed ? subLabel : undefined}
                                                        className={`block px-10 py-2 text-sm rounded-r-lg transition-colors duration-200 ${isSubActive
                                                                ? "bg-indigo-600 text-white font-semibold"
                                                                : "text-gray-400 hover:bg-gray-700 hover:text-indigo-300"
                                                            }`}
                                                    >
                                                        {!collapsed && subLabel}
                                                    </Link>
                                                </li>
                                            )
                                        })}
                                    </ul>
                                </li>
                            )
                        }

                        return (
                            <li key={href}>
                                <Link
                                    href={href}
                                    title={collapsed ? label : undefined}
                                    className={`flex items-center px-4 py-2.5 space-x-3 rounded-r-lg border-l-4 transition-colors duration-200 cursor-pointer ${isActive
                                            ? "bg-gray-700 text-indigo-400 border-indigo-400 shadow-lg"
                                            : "border-transparent hover:text-indigo-300 hover:border-indigo-600"
                                        }`}
                                >
                                    <Icon size={20} />
                                    {!collapsed && <span>{label}</span>}
                                </Link>
                            </li>
                        )
                    })}
                </ul>
            </nav>
        </aside>
    )
}
