"use client";

import { useSession } from "next-auth/react";
import { motion, useMotionValue, useTransform } from "framer-motion";
import Image from "next/image";
import { FiMail, FiBriefcase, FiShield, FiClock,FiUser } from "react-icons/fi";
import React from "react";

export default function UserProfilePage() {
  const bannerRef = React.useRef(null);
  const { data: session, status } = useSession();

  // ✅ All hooks at the top level
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  const bubble1X = useTransform(mouseX, [0, 300], [-15, 15]);
  const bubble1Y = useTransform(mouseY, [0, 160], [-10, 10]);
  const bubble2X = useTransform(mouseX, [0, 300], [20, -20]);
  const bubble2Y = useTransform(mouseY, [0, 160], [15, -15]);
  const bubble3X = useTransform(mouseX, [0, 300], [-10, 10]);
  const bubble3Y = useTransform(mouseY, [0, 160], [5, -5]);
  const bubble4X = useTransform(mouseX, [0, 300], [5, -5]);
  const bubble4Y = useTransform(mouseY, [0, 160], [-12, 12]);

  const handleMouseMove = (e) => {
    if (!bannerRef.current) return;
    const rect = bannerRef.current.getBoundingClientRect();
    mouseX.set(e.clientX - rect.left);
    mouseY.set(e.clientY - rect.top);
  };

  const handleMouseLeave = () => {
    if (!bannerRef.current) return;
    const rect = bannerRef.current.getBoundingClientRect();
    mouseX.set(rect.width / 2);
    mouseY.set(rect.height / 2);
  };

  // ✅ Conditional rendering happens AFTER hooks
  if (status === "loading") {
    return (
      <div className="flex items-center justify-center min-h-screen bg-gray-100">
        <div className="text-xl font-semibold text-gray-600 animate-pulse">
          Loading Profile...
        </div>
      </div>
    );
  }

  if (status === "unauthenticated") {
    return (
      <div className="flex items-center justify-center min-h-screen bg-gray-100">
        <div className="text-xl font-semibold text-red-500 text-center p-4">
          Access Denied. Please log in to view your profile.
        </div>
      </div>
    );
  }

  const user = session?.user;
  console.log(user);

  return (
    <div className="w-full font-sans" style={{ height: "100%" }}>
      <div
        style={{ height: "100%" }}
        className="relative w-full"
        initial={{ opacity: 0, y: 50 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.7, ease: "easeOut" }}
      >
        {/* Banner */}
        <div
          ref={bannerRef}
          onMouseMove={handleMouseMove}
          onMouseLeave={handleMouseLeave}
          className="h-40 bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] relative overflow-hidden"
          style={{ borderTopLeftRadius: "20px", borderTopRightRadius: "20px" }}
        >
          {/* Animated Bubbles - smoother & cooler version */}
          <div
            style={{ x: bubble1X, y: bubble1Y }}
            className="absolute -bottom-10 -left-10 w-32 h-32 bg-white/10 rounded-full"
            animate={{ y: [0, -15, 0], scale: [1, 1.05, 1] }}
            transition={{ duration: 20, repeat: Infinity, repeatType: "mirror", ease: "easeInOut" }}
          />
          <div
            style={{ x: bubble2X, y: bubble2Y }}
            className="absolute top-5 right-20 w-24 h-24 bg-white/10 rounded-full"
            animate={{ y: [0, 10, 0], x: [0, -10, 0] }}
            transition={{ duration: 18, repeat: Infinity, repeatType: "mirror", ease: "easeInOut", delay: 2 }}
          />
          <div
            style={{ x: bubble3X, y: bubble3Y }}
            className="absolute bottom-8 right-8 w-16 h-16 bg-white/15 rounded-full"
            animate={{ y: [0, -10, 0] }}
            transition={{ duration: 15, repeat: Infinity, repeatType: "mirror", ease: "easeInOut", delay: 4 }}
          />
          <div
            style={{ x: bubble4X, y: bubble4Y }}
            className="absolute top-1/2 left-1/3 w-12 h-12 bg-white/5 rounded-full"
            animate={{ y: [0, 15, 0] }}
            transition={{ duration: 25, repeat: Infinity, repeatType: "mirror", ease: "easeInOut", delay: 1 }}
          />
          {/* Extra Bubbles for more depth */}
          <div
            className="absolute top-10 left-20 w-20 h-20 bg-white/5 rounded-full"
            animate={{
              y: [0, 8, -8, 0],
              x: [0, 12, -12, 0],
              scale: [1, 1.03, 0.97, 1],
            }}
            transition={{
              duration: 22,
              repeat: Infinity,
              repeatType: "mirror",
              ease: "easeInOut",
              delay: 5,
            }}
          />
          <div
            className="absolute bottom-2 left-1/2 w-8 h-8 bg-white/10 rounded-full"
            animate={{
              y: [0, -12, 0],
              x: [0, 5, -5, 0],
            }}
            transition={{
              duration: 19,
              repeat: Infinity,
              repeatType: "mirror",
              ease: "easeInOut",
              delay: 6,
            }}
          />

          <div
            className="absolute inset-0 bg-black/20"
            style={{
              borderTopLeftRadius: "20px",
              borderTopRightRadius: "20px",
            }}
          ></div>
        </div>

        {/* Profile Info */}
        <div className="px-8 flex flex-col sm:flex-row items-center sm:items-end -mt-20 sm:-mt-16 relative z-10">
          <div className="w-36 h-36 rounded-full shadow-2xl overflow-hidden ring-4 ring-white bg-gray-200 flex-shrink-0">
            {user?.profile ? (
              <Image
                src={`/emp_profile/${user.profile}`}
                alt="Profile"
                width={144}
                height={144}
                className="object-cover w-full h-full"
                priority
              />
            ) : (
              <div className="w-full h-full flex items-center justify-center bg-gray-200">
                <span className="text-6xl font-bold text-slate-400">
                  {user?.name?.[0]?.toUpperCase() || "U"}
                </span>
              </div>
            )}
          </div>
          <div className="mt-4 sm:mt-0 sm:ml-8 text-center sm:text-left pb-4">
            <h1 className="text-3xl font-bold tracking-tight text-gray-800">
              {user?.name}
            </h1>
            <p className="text-md text-purple-600 mt-1 capitalize tracking-wide font-medium">
              {user?.role_name}
            </p>
          </div>
        </div>

        {/* Account Details */}
        <div className="p-8 bg-slate-50/50">
          <h2 className="text-2xl font-bold text-gray-700 mb-6 border-b-2 border-gray-200/80 pb-4">
            Account Details
          </h2>
          <div
            className="grid grid-cols-1 md:grid-cols-2 gap-6"
            initial="hidden"
            animate="visible"
            variants={{
              visible: { transition: { staggerChildren: 0.08 } },
            }}
          >
            <div
              variants={{
                hidden: { opacity: 0, y: 20 },
                visible: { opacity: 1, y: 0 },
              }}
              className="bg-gray-50/70 p-5 rounded-xl flex items-center gap-4 border border-gray-200/80 transition-all duration-300 hover:bg-white hover:shadow-lg hover:border-blue-300"
            >
              <div className="p-3 bg-blue-100 rounded-full">
                <FiBriefcase className="w-6 h-6 text-blue-600" />
              </div>
              <div>
                <p className="text-sm text-gray-400">Employee ID</p>
                <p className="font-semibold text-gray-800">
                  {user?.user_empid}
                </p>
              </div>
            </div>
             <div
              variants={{
                hidden: { opacity: 0, y: 20 },
                visible: { opacity: 1, y: 0 },
              }}
              className="bg-gray-50/70 p-5 rounded-xl flex items-center gap-4 border border-gray-200/80 transition-all duration-300 hover:bg-white hover:shadow-lg hover:border-blue-300"
            >
              <div className="p-3 bg-blue-100 rounded-full">
                <FiUser className="w-6 h-6 text-blue-600" />
              </div>
              <div>
                <p className="text-sm text-gray-400">Employee Name</p>
                <p className="font-semibold text-gray-800">
                  {user?.name}
                </p>
              </div>
            </div>

            <div
              variants={{
                hidden: { opacity: 0, y: 20 },
                visible: { opacity: 1, y: 0 },
              }}
              className="bg-gray-50/70 p-5 rounded-xl flex items-center gap-4 border border-gray-200/80 transition-all duration-300 hover:bg-white hover:shadow-lg hover:border-pink-300"
            >
              <div className="p-3 bg-pink-100 rounded-full">
                <FiMail className="w-6 h-6 text-pink-600" />
              </div>
              <div>
                <p className="text-sm text-gray-400">Email Address</p>
                <p className="font-semibold text-gray-800">{user?.email}</p>
              </div>
            </div>

            <div
              variants={{
                hidden: { opacity: 0, y: 20 },
                visible: { opacity: 1, y: 0 },
              }}
              className="bg-gray-50/70 p-5 rounded-xl flex items-center gap-4 border border-gray-200/80 transition-all duration-300 hover:bg-white hover:shadow-lg hover:border-purple-300"
            >
              <div className="p-3 bg-purple-100 rounded-full">
                <FiShield className="w-6 h-6 text-purple-600" />
              </div>
              <div>
                <p className="text-sm text-gray-400">Role</p>
                <p className="font-semibold text-gray-800 capitalize">
                  {user?.role_name}
                </p>
              </div>
            </div>

            <div
              variants={{
                hidden: { opacity: 0, y: 20 },
                visible: { opacity: 1, y: 0 },
              }}
              className="bg-gray-50/70 p-5 rounded-xl flex items-center gap-4 border border-gray-200/80 transition-all duration-300 hover:bg-white hover:shadow-lg hover:border-green-300"
            >
              <div className="p-3 bg-green-100 rounded-full">
                <FiClock className="w-6 h-6 text-green-600" />
              </div>
              <div>
                <p className="text-sm text-gray-400">Member Since</p>
                <p className="font-semibold text-gray-800">January 1, 2023</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
