"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import CustomDataTable from "../../dataTable/customDataTable";
import { getChapterPendingColumns } from "./columns";
import EntryModal from "@/app/components/modals/entryModel";
import Select from "react-select";
import { BookFilterChapterAction } from "@/lib/actions/entryActions";
import { toast } from "react-toastify";
import selectStyles from "@/app/components/selectStyles";

export default function ChaptersPendingList({ initialPendings, initialBooks, initialtasks, initialUsers}) {
  const [data, setData] = useState(initialPendings || []);
  const [filteredData, setFilteredData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedRow, setSelectedRow] = useState(null);
  const [isModalOpen, setIsModalOpen] = useState(false);
 const [selectedUser, setSelectedUser] = useState(null);
  const [selectedBook, setSelectedBook] = useState(null);
  const [chapterOptions, setChapterOptions] = useState([]);
  const [selectedChapter, setSelectedChapter] = useState(null);
  const [selectedTask, setSelectedTask] = useState(null);
  const [startDate, setStartDate] = useState("");
  const [endDate, setEndDate] = useState("");
  const [entryId, setEntryId] = useState(null);
  const router = useRouter();

  // Initial load
  useEffect(() => {
    setData(initialPendings || []);
    setFilteredData(initialPendings || []);
    setLoading(false);
  }, [initialPendings]);

  // Fetch chapters when book is selected
  const selectedBookId = selectedBook?.value;
  useEffect(() => {
    if (!selectedBookId) {
      setChapterOptions([]);
      setSelectedChapter(null);
      return;
    }

    const fetchChapters = async (bookId) => {
      try {
        const res = await BookFilterChapterAction(bookId);
        const options = res.chapters.map((c) => ({
          value: String(c.chap_pid),
          label: c.chap_title.charAt(0).toUpperCase() + c.chap_title.slice(1),
        }));
        setChapterOptions(options);
      } catch (error) {
        toast.error("Failed to load chapters");
      }
    };

    fetchChapters(selectedBookId);
  }, [selectedBookId]);

  // Filter options
  const bookOptions = initialBooks.map((b) => ({
    value: String(b.book_pid),
    label: b.book_title.charAt(0).toUpperCase() + b.book_title.slice(1),
  }));

  const taskOptions = initialtasks.map((t) => ({
    value: String(t.task_pid),
    label: t.task_name.charAt(0).toUpperCase() + t.task_name.slice(1),
  }));
   const userOptions = initialUsers.map((u) => ({
    value: String(u.user_pid),
    label: u.user_name.charAt(0).toUpperCase() + u.user_name.slice(1),
  }));

  // Filter logic
  useEffect(() => {
    const filtered = data.filter((item) => {
      const matchesUser = selectedUser ? String(item.entry_createdBy) === selectedUser.value : true;
      const matchesBook = selectedBook ? String(item.entry_bookId) === selectedBook.value : true;
      const matchesChapter = selectedChapter ? String(item.entry_chapterId) === selectedChapter.value : true;
      const matchesTask = selectedTask ? String(item.entry_taskId) === selectedTask.value : true;

      const itemDate = new Date(item.entry_date);
      const matchesStartDate = startDate ? itemDate >= new Date(startDate) : true;
      const matchesEndDate = endDate ? itemDate <= new Date(endDate) : true;

      return (
        matchesUser &&
        matchesBook &&
        matchesChapter &&
        matchesTask &&
        matchesStartDate &&
        matchesEndDate
      );
    });
    setFilteredData(filtered);
  }, [selectedUser, selectedBook, selectedChapter, selectedTask, startDate, endDate, data]);

  const columns = useMemo(
    () => getChapterPendingColumns(router, setSelectedRow, setIsModalOpen, setEntryId),
    [router]
  );

  const filteredColumns = columns.filter(
    (col) =>
      col.name !== "Comments" &&
      col.name !== "Contact Email" &&
      col.name !== "Upload File"
  );

  // 🔥 Common react-select styles (so dropdown always above table)



  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">
              Pending Files
            </h2>
            <p className="text-gray-500 mt-1 text-sm">
              Manage all submitted pending files here.
            </p>
          </div>
        </div>
      </header>

      <div className="bg-purple-50 border border-purple-200 rounded-lg p-2 shadow-sm">
        <div className="flex flex-wrap gap-4 items-center justify-start">
             {/* User Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              User
            </label>
            <Select
              instanceId="user-filter"
              options={userOptions}
              value={selectedUser}
              onChange={setSelectedUser}
              placeholder="Select User"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>
          {/* Book Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Book
            </label>
            <Select
              instanceId="book-filter"
              options={bookOptions}
              value={selectedBook}
              onChange={setSelectedBook}
              placeholder="Select Book"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>

          {/* Chapter Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Chapter
            </label>
            <Select
              instanceId="chapter-filter"
              options={chapterOptions}
              value={selectedChapter}
              onChange={setSelectedChapter}
              placeholder="Select Chapter"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>

          {/* Task Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Task
            </label>
            <Select
              instanceId="task-filter"
              options={taskOptions}
              value={selectedTask}
              onChange={setSelectedTask}
              placeholder="Select Task"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>

          {/* Start Date */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Start Date
            </label>
            <input
              type="date"
              className="w-40 border border-gray-300 rounded px-3 py-2 text-sm focus:ring focus:ring-purple-200"
              value={startDate}
              onChange={(e) => setStartDate(e.target.value)}
            />
          </div>

          {/* End Date */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              End Date
            </label>
            <input
              type="date"
              className="w-40 border border-gray-300 rounded px-3 py-2 text-sm focus:ring focus:ring-purple-200"
              value={endDate}
              onChange={(e) => setEndDate(e.target.value)}
              min={startDate}
            />
          </div>

          {/* Clear Filters Button */}
          <div className="pt-5">
            <button
              onClick={() => {
                setSelectedBook(null);
                setSelectedChapter(null);
                setSelectedTask(null);
                setStartDate("");
                setEndDate("");
              }}
              className="text-sm px-4 py-2 bg-purple-100 hover:bg-purple-200 text-purple-700 rounded transition-all duration-200 cursor-pointer"
            >
              Clear Filters
            </button>
          </div>
        </div>
      </div>

      <CustomDataTable
        columns={filteredColumns}
        data={filteredData}
        progressPending={loading}
        defaultSort={{ key: "Owned By", direction: "ascending" }}
      />

      {isModalOpen && selectedRow && (
        <EntryModal
          isOpen={isModalOpen}
          onClose={() => setIsModalOpen(false)}
          entryId={entryId}
          selectedRow={selectedRow}
        />
      )}
    </div>
  );
}
