"use client";
import React, { useEffect, useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { queriesAddSchema } from '@/validation/validation';
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { createQueryAction } from "@/lib/actions/queryAction";
import BackButton from "@/app/components/BackButton";
import { GetChapterByIdAction } from "@/lib/actions/chapterActions";
import { updateQueriesAction } from "@/lib/actions/queryAction";
import { number } from "zod";


export default function CreateQueriesform({ booksData, queriesPid, user_pid, queriesData, taskData }) {
  const [loading, setLoading] = useState(false);
  const [books, setBooks] = useState(booksData || []);
  const [tasks, setTask] = useState(taskData || []);
  const [chapters, setChapters] = useState([]);
  const [PreviousQUeryReply, setPreviousQUeryReply] = useState(queriesData ? queriesData.queries_name_reply : '');
  const isEditMode = !!queriesPid;
  const router = useRouter();

  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
    watch,
    setValue,
  } = useForm({
    resolver: zodResolver(queriesAddSchema(isEditMode)),
    mode: 'onBlur',
    defaultValues: {
      queries_name: '',
      queries_book_pid: '',
      queries_chap_pid: '',
      queries_task_pid: '',
      queries_name_reply: ''
    },
  });

  useEffect(() => {
    const loadDataForEdit = async () => {
      if (queriesData) {
        const bookId = queriesData.queries_book_pid;

        // Set book_pid first so chapter fetching is triggered
        setValue('queries_book_pid', bookId);

        // Fetch chapters for this book
        const res = await GetChapterByIdAction(bookId);
        const chapterOptions = res.chapters.map((c) => ({
          value: String(c.chap_pid),
          label: c.chap_title.charAt(0).toUpperCase() + c.chap_title.slice(1),
        }));
        setChapters(chapterOptions);

        // Now set the rest of the values
        // setValue('queries_chap_pid', queriesData.queries_chap_pid);
        setValue('queries_name', queriesData.queries_name);
        // setValue('queries_task_pid', queriesData.queries_task_pid);
        setValue('queries_book_pid', String(queriesData.queries_book_pid));
        setValue('queries_chap_pid', String(queriesData.queries_chap_pid));
        setValue('queries_task_pid', String(queriesData.queries_task_pid));
        setValue('queries_name_reply', String(queriesData.queries_name_reply != null? queriesData.queries_name_reply : ''));

        setPreviousQUeryReply(queriesData.queries_name_reply || '');

      } else {
        reset();
      }
  };

  loadDataForEdit();
}, [queriesData, setValue, reset]);


  const onSubmit = async (data) => {
    setLoading(true);
    try {
      // const payload = { 
      //   //  queries_createdBy: Number(user_pid),
      //    queries_name: data.queries_name,
      //    queries_book_pid: Number(data.queries_book_pid),
      //    queries_chap_pid: Number(data.queries_chap_pid),
      //    queries_task_pid: Number(data.queries_task_pid),
      //    queries_name_reply: data.queries_name_reply,
      //    queries_reply_by: data.queries_name_reply,

      //   };
      //   payload.user = {
      //     connect: { user_pid: Number(user_pid) } // sets `queries_createdBy`
      //   };

       const payload = {
  queries_name: data.queries_name,
  // Remove this line:
  // queries_reply_by: Number(user_pid),  <-- DO NOT set directly
};

// 🔗 Connect user (queries_createdBy)
payload.user = {
  connect: { user_pid: Number(user_pid) }
};

// 🔗 Connect book
payload.book = {
  connect: { book_pid: Number(data.queries_book_pid) }
};

// 🔗 Connect chapter (if present)
if (data.queries_chap_pid) {
  payload.chapter = {
    connect: { chap_pid: Number(data.queries_chap_pid) }
  };
}

// 🔗 Connect task (if present)
if (data.queries_task_pid) {
  payload.task = {
    connect: { task_pid: Number(data.queries_task_pid) }
  };
}

// ✏️ Conditionally update reply
if (PreviousQUeryReply != data.queries_name_reply) {
  payload.queries_name_reply = data.queries_name_reply;

  // Instead of queries_reply_by, update relation properly:
  payload.user_repliedBy = {
    connect: { user_pid: Number(user_pid) }
  };
}

if (isEditMode) {
  const res = await updateQueriesAction(queriesPid, payload);
  if (res?.success) {
    router.push("/queries/list");
    toast.success("Query updated successfully!");
  } else {
    toast.error(res?.message || "Failed to update Query.");
  }
}
else {
        const res = await createQueryAction(payload);
        if (res?.success) {
          router.push("/queries/list");
          toast.success("Query created successfully!");
        } else {
          toast.error(res?.message || "Failed to create Querys.");
        }
      }
    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const selectedBookId = watch("queries_book_pid");
  useEffect(() => {
    if (!selectedBookId) return;

    const fetchChapters = async (selectedBookId) => {
      try {
        console.log(selectedBookId, 'Selected Book ID');
        
        const res = await GetChapterByIdAction(selectedBookId);
        console.log(res, 'Chapters data');
        
        const options = res.chapters.map((c) => ({
          value: String(c.chap_pid),
          label: c.chap_title.charAt(0).toUpperCase() + c.chap_title.slice(1),
        }));
        setChapters(options);
      } catch (error) {
        console.log(error, 'Error fetching chapters');
        
        toast.error("Failed to load chapters");
      }
    };

    fetchChapters(selectedBookId);
  }, [selectedBookId]);

  const handleReset = () => {
    reset();
    toast.info('Form reset.');
  };

  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">
              {isEditMode ? 'Edit Query' : 'Create New Query'}
            </h2>
            <p className="text-gray-500 mt-1 text-sm">
              {isEditMode
                ? 'Modify the details of the selected Query.'
                : 'Fill in the details below to add a new Query.'}
            </p>
          </div>
          <BackButton />
        </div>
      </header>

      <form onSubmit={handleSubmit(onSubmit)} className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl mx-auto">
        {/* Left Column */}
        <div className="space-y-4">
          {/* Book Name */}
          <div>
            <label htmlFor="queries_book_pid" className="block text-sm font-medium text-gray-700 mb-1">
              Book Name <span className="text-red-500">*</span>
            </label>
            <select
              id="queries_book_pid"
              {...register('queries_book_pid')}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.queries_book_pid ? 'border-red-500' : 'border-gray-300'
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            >
              <option value="">Select a Book</option>
              {books.map((book) => (
                <option key={book.book_pid} value={book.book_pid}>
                  {book.book_title}
                </option>
              ))}
            </select>
            {errors.queries_book_pid && (
              <p className="text-sm text-red-500 mt-1">{errors.queries_book_pid.message}</p>
            )}
          </div>

          {/* Chapter Name */}
          <div>
            <label htmlFor="queries_chap_pid" className="block text-sm font-medium text-gray-700 mb-1">
              Chapter Name <span className="text-red-500">*</span>
            </label>
            <select
              id="queries_chap_pid"
              {...register('queries_chap_pid')}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.queries_chap_pid ? 'border-red-500' : 'border-gray-300'
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            >
              <option value="">Select a Chapter</option>
              {chapters.map((chapter) => (
                <option key={chapter.value} value={chapter.value}>
                  {chapter.label}
                </option>
              ))}
            </select>
            {errors.queries_chap_pid && (
              <p className="text-sm text-red-500 mt-1">{errors.queries_chap_pid.message}</p>
            )}
          </div>

          {/* Task Name */}
          <div>
            <label htmlFor="queries_task_pid" className="block text-sm font-medium text-gray-700 mb-1">
              Task Name <span className="text-red-500">*</span>
            </label>
            <select
              id="queries_task_pid"
              {...register('queries_task_pid')}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.queries_task_pid ? 'border-red-500' : 'border-gray-300'
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            >
              <option value="">Select a Task</option>
              {tasks.map((task) => (
                <option key={task.task_pid} value={task.task_pid}>
                  {task.task_name}
                </option>
              ))}
            </select>
            {errors.queries_task_pid && (
              <p className="text-sm text-red-500 mt-1">{errors.queries_task_pid.message}</p>
            )}
          </div>
        </div>

    {/* Right Column */}
    <div className="space-y-4">
      {/* Query Name */}
      <div>
        <label htmlFor="queries_name" className="block text-sm font-medium text-gray-700 mb-1" >
          Query <span className="text-red-500">*</span>
        </label>
        <textarea
          id="queries_name"
          placeholder="Enter Queryname"
          {...register('queries_name')} style={{ height:"6rem"}}
          className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.queries_name ? 'border-red-500' : 'border-gray-300'
            } focus:outline-none focus:ring-2 focus:ring-blue-500`}
        />
        {errors.queries_name && (
          <p className="text-sm text-red-500 mt-1">{errors.queries_name.message}</p>
        )}
      </div>

      {/* Query Reply */}
      {queriesPid && (
        <div>
          <label htmlFor="queries_name_reply" className="block text-sm font-medium text-gray-700 mb-1">
            Reply Query <span className="text-red-500"></span>
          </label>
          <textarea
            id="queries_name_reply"
            placeholder="Enter Query Reply"
            {...register('queries_name_reply')}
            style={{ height: "6rem" }}
            className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.queries_name_reply ? 'border-red-500' : 'border-gray-300'} focus:outline-none focus:ring-2 focus:ring-blue-500`}
          />
          {errors.queries_name_reply && (
            <p className="text-sm text-red-500 mt-1">{errors.queries_name_reply.message}</p>
          )}
        </div>
      )}
    </div>



        {/* Buttons */}
        <div className="col-span-1 md:col-span-2 flex flex-col sm:flex-row justify-end gap-4">
          {!isEditMode && (
            <button
              type="button"
              onClick={handleReset}
              className="w-full sm:w-auto px-5 py-2 border border-red-600 text-red-600 rounded-md hover:bg-red-100 transition cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
            >
            </button>
          )}
          <button
            type="submit"
            className="w-full sm:w-auto px-5 py-2 bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white rounded-md transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2 d-flex justify-center"
          >
            <span>{loading ? (isEditMode ? 'Updating...' : 'Creating...') : isEditMode ? 'Update Query' : 'Create Query'}</span>
            {loading ? (
              <FiLoader size={20} className="animate-spin" />
            ) : (
              <FiArrowRightCircle size={20} />
            )}
          </button>
        </div>
      </form>
    </div>
  );
}
