"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import Swal from "sweetalert2";
import { toast } from "react-toastify";
import CustomDataTable from "../../dataTable/customDataTable";
import { getKnowledgeColumns } from "./columns";
import { updateDeleteAction } from "@/lib/actions/knowledgeActions";

export default function HolidaysList({ initialKnowledges, role }) {
  const [data, setData] = useState(initialKnowledges || []);
  const [loading, setLoading] = useState(true);
  const router = useRouter();

  useEffect(() => {
    setData(initialKnowledges || []);
    setLoading(false);
  }, [initialKnowledges]);

  const handleDelete = async (knowledgeId, knowledgeName) => {
    const result = await Swal.fire({
      title: "Are you sure?",
      text: `Do you really want to delete the knowledge named "${knowledgeName}"?`,
      icon: "warning",
      showCancelButton: true,
      confirmButtonColor: "#d33",
      cancelButtonColor: "#3085d6",
      confirmButtonText: "Yes, delete it!",
    });

    if (!result.isConfirmed) return;

    try {
      const payload = {
        knowledge_isdelete: "inactive",
      };

      const res = await updateDeleteAction(knowledgeId, payload);

      if (res?.success) {
        // ✅ Filter out deleted knowledge from local state
        setData((prevData) =>
          prevData.filter((data) => data.knowledge_pid !== knowledgeId)
        );
        toast.success("Knowledge has been deleted.");
      } else {
        toast.error(res?.message || "Failed to delete knowledge.");
      }

      // ✅ Filter out deleted book from local state



    } catch (error) {
      toast.error(error.message || "Error deleting knowledge");
    }
  };

  const columns = useMemo(() => getKnowledgeColumns(router, handleDelete), [router]);

  return (
    <div className="max-w-full px-4 py-4 mx-auto bg-white rounded-lg  space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">Knowledge Base List</h2>
            <p className="text-gray-500 mt-1 text-sm">
              Below is a list of all Knowledge.
            </p>
          </div>
          {role !== 3 && (
          <button
            onClick={() => router.push('/knowledge/create')}
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-2 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105"
          >
            + New knowledge
          </button>
          )}
        </div>
      </header>

      <CustomDataTable
        columns={columns}
        data={data}
        progressPending={loading}
        defaultSort={{ key: "Knowledge Name", direction: "ascending" }}
      />
    </div>
  );
}
