"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import Swal from "sweetalert2";
import { toast } from "react-toastify";
import CustomDataTable from "../../dataTable/customDataTable";
import { getHolidayColumns } from "./columns";
import { updateHolidayAction } from "@/lib/actions/holidayActions";
import Select from "react-select";
import selectStyles from "@/app/components/selectStyles";

export default function HolidaysList({ initialHolidays, role }) {
  const [data, setData] = useState(initialHolidays || []);
  const [loading, setLoading] = useState(true);
  const [filteredData, setFilteredData] = useState([]);
  const [yearOptions, setYearOptions] = useState([]);
  const [selectedYear, setSelectedYear] = useState(null);
  const [selectedType, setSelectedType] = useState(null);
  const router = useRouter();
  useEffect(() => {
    setLoading(true);
    setData(initialHolidays || []);
    setFilteredData(initialHolidays || []);
    setLoading(false);
  }, [initialHolidays]);
  const handleDelete = async (holidayId, holidayName) => {
    const result = await Swal.fire({
      title: "Are you sure?",
      text: `Do you really want to delete the holiday named "${holidayName}"?`,
      icon: "warning",
      showCancelButton: true,
      confirmButtonColor: "#d33",
      cancelButtonColor: "#3085d6",
      confirmButtonText: "Yes, delete it!",
    });
    if (!result.isConfirmed) return;
    try {
      const payload = {
        holiday_isdelete: "inactive",
      };
      const res = await updateHolidayAction(holidayId, payload);
      if (res?.success) {
        // ✅ Filter out deleted holiday from local state
        setData((prevData) =>
          prevData.filter((data) => data.holiday_pid !== holidayId)
        );
        toast.success("Holiday has been deleted.");
      } else {
        toast.error(res?.message || "Failed to delete holiday.");
      }
    } catch (error) {
      toast.error(error.message || "Error deleting holiday");
    }
  };
  useEffect(() => {
    const currentYear = new Date().getFullYear();
    const years = [];
    for (let i = 0; i < 10; i++) {
      const year = currentYear + i;
      years.push({ value: year.toString(), label: `${year}` });
    }
    setYearOptions(years);
    setSelectedYear({ value: currentYear.toString(), label: `${currentYear}` });
  }, []);
  const typeOptions = [
    { value: 'PDMR', label: 'PDMR' },
    { value: 'Compuscript', label: 'Compuscript' },
  ];
  // Filter logic
  useEffect(() => {
    const filtered = data.filter((item) => {
      const itemDate = new Date(item.holiday_date).getFullYear().toString();
      const matchesYear = selectedYear ? itemDate === selectedYear.value : true;
      const matchesType = selectedType ? String(item.holiday_OfficeType) === selectedType.value : true;
      return (
        matchesYear &&
        matchesType
      );
    });
    setFilteredData(filtered);
  }, [selectedYear, selectedType, data]);
  const columns = useMemo(() => getHolidayColumns(router, handleDelete, role), [router]);


  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">Holiday Lists</h2>
            <p className="text-gray-500 mt-1 text-sm">
              Below is a list of all holidays.
            </p>
          </div>
          {role !== 3 && (
          <button
            onClick={() => router.push("/holidays/create")}
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-2 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105"
          >
            + New Holiday
          </button>
          )}
        </div>
      </header>
      <div className="bg-purple-50 border border-purple-200 rounded-lg p-2 shadow-sm">
        <div className="flex flex-wrap gap-4 items-center justify-start">
          {/* Year Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Year
            </label>
            <Select
              instanceId="year-filter"
              options={yearOptions}
              value={selectedYear}
              onChange={setSelectedYear}
              placeholder="Select Year"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>
          {/* Type Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Type
            </label>
            <Select
              instanceId="type-filter"
              options={typeOptions}
              value={selectedType}
              onChange={setSelectedType}
              placeholder="Select Type"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>
          {/* Clear Filters Button */}
          <div className="pt-5">
            <button
              onClick={() => {
                setSelectedYear(null);
                setSelectedType(null);
              }}
              className="text-sm px-4 py-2 bg-purple-100 hover:bg-purple-200 text-purple-700 rounded transition-all duration-200 cursor-pointer"
            >
              Clear Filters
            </button>
          </div>
        </div>
      </div>
      <CustomDataTable
        columns={columns}
        data={filteredData}
        progressPending={loading}
        defaultSort={{ key: "Holiday Name", direction: "ascending" }}
      />
    </div>
  );
}
