"use client";

import React, { useState, useEffect } from "react";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { entryStartSchema, entryEndSchema, entryUpdateSchema } from "@/validation/validation";
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import Select from "react-select";
import PendingModal from "@/app/components/modals/pendingModal";
import ResumeModal from "@/app/components/modals/resumeModal";
import TakeBreakModal from "@/app/components/modals/breakModel";
import { createEntryAction, updateEntryAction, updateFinalEntryAction, updateBreak, checkBreaksAction, checkEntryAction, BookChapterAction, BookTaskAction } from "@/lib/actions/entryActions";
import { getChaptersByBookId } from "@/lib/services/chapterService";
import { useSession } from "next-auth/react";
// import { useWatch } from "react-hook-form";

export default function CreateEntryForm({ initialBooks, initialtasks, initialstatus, initialchapters, initialemails, initialbreaks, entry, initialPending, fentryId }) {
  const { data: session, status } = useSession();
  const [loading, setLoading] = useState(false);
  const router = useRouter();
  const [entryStarted, setEntryStarted] = useState(false);
  const [entryEnded, setEntryEnded] = useState(false);
  const [buttonType, setButtonType] = useState("start");
  const [entryId, setEntryId] = useState(null);
  const [chapterOptions, setChapterOptions] = useState([]);
  const [taskOptions, setTaskOptions] = useState([]);
  const [pendingId, setPendingId] = useState(null);
  const [breakId, setBreakId] = useState(null);
  const [breakEntryId, setBreakEntryId] = useState(null);
  const [isPendingModalOpen, setIsPendingModalOpen] = useState(false);
  const [isResumeModalOpen, setIsResumeModalOpen] = useState(false);
  const [isbreakModelOpen, setIsbreakModelOpen] = useState(false);
  //  const [onBreak, setOnBreak] = useState(false);
  const [isEntryPendingStatus, setIsEntryPendingStatus] = useState(false);
  const isEditMode = !!fentryId;
  // console.log(fentryId);
  const user_pid = session?.user?.user_pid || "";
  useEffect(() => {
    if (!user_pid) return;

    async function fetchBreakStatus() {
      try {
        const break_res = await checkBreaksAction(user_pid); // ✅ directly call server action
        if (break_res?.success && break_res?.count) {
          setBreakId(break_res.count.bh_pid);
          setIsEntryPendingStatus(false);
        } else {
          const user_res = await checkEntryAction(user_pid);
          //console.log(user_res);
          if (user_res?.success && user_res?.count) {
              setBreakEntryId(user_res.count);
            if (isEditMode) {
              setIsEntryPendingStatus(user_res.count);
            } else {
              setIsEntryPendingStatus(true);
            }

          } else {
            setIsEntryPendingStatus(false);
            setBreakId(false);
          }

        }
      } catch (err) {
        console.error(err);
        //toast.error("Error fetching break status");
      }
    }

    fetchBreakStatus();
  }, [user_pid]);

  function getSchemaByButtonType(type) {
    // console.log(type)
    switch (type) {
      case "start":
        return entryStartSchema;
      case "end":
        return entryEndSchema;
      case "update":
        return entryUpdateSchema;
      default:
        return entryStartSchema; // fallback
    }
  }

  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
    watch,
    setValue,
  } = useForm({
    resolver: zodResolver(getSchemaByButtonType(buttonType)),
    mode: "onBlur",
    defaultValues: {
      entry_book_pid: "",
      entry_chapter_pid: "",
      entry_task_pid: "",
      entry_start_time: "",
      entry_end_date: "",
      entry_end_time: "",
      entry_wordcount: "",
      entry_comments: "",
      entry_contact_email: "",
      entry_status: "",
      entry_upload_file: null,
    },
  });

  const selectedBookId = watch("entry_book_pid");

  useEffect(() => {
    if (!selectedBookId) return;

    const fetchChapters = async (selectedBookId) => {
      try {
       // console.log(selectedBookId);
        const res = await BookChapterAction(selectedBookId);
        const options = res.chapters.map((c) => ({
          value: String(c.chap_pid),
          label: c.chap_title.charAt(0).toUpperCase() + c.chap_title.slice(1),
        }));
        setChapterOptions(options);
      } catch (error) {
        toast.error("Failed to load chapters");
      }
    };
    fetchChapters(selectedBookId);
  }, [selectedBookId]);

  const selectedChapterId = watch("entry_chapter_pid");


  useEffect(() => {
    if (!selectedChapterId) return;

    const fetchTasks = async (selectedChapterId) => {
      try {
        // console.log(selectedBookId);
        const res = await BookTaskAction(selectedChapterId);
        const options = res.tasks.map((t) => ({
          value: String(t.task_pid),
          label: t.task_name.charAt(0).toUpperCase() + t.task_name.slice(1),
        }));
        setTaskOptions(options);
        setValue("entry_wordcount", res.chapterDetails.chap_wordcount);
      } catch (error) {
        toast.error("Failed to load tasks");
      }
    };
    fetchTasks(selectedChapterId);
  }, [selectedChapterId]);




  const formatDate = (dateString) => {
    if (!dateString) return "";
    return new Date(dateString).toISOString().split("T")[0];
  };
  // If editing, prefill form
  useEffect(() => {
    if (entry) {
      setValue("entry_book_pid", String(entry.entry_bookId));
      setValue("entry_chapter_pid", String(entry.entry_chapterId));
      setValue("entry_task_pid", String(entry.entry_taskId));
      setValue("entry_start_time", String(entry.entry_start_time));
      if (initialPending) {
        setPendingId(initialPending.pr_pid);
      }
      if (entry.entry_start_time != null) {
        setEntryStarted(true);
        setEntryId(entry.entry_pid);
      }
      setValue("entry_end_date", new Date().toISOString().split("T")[0]);

      if (entry.entry_end_time != null) {
        setValue("entry_end_time", String(entry.entry_end_time));
        setEntryEnded(true);
      }
      setValue("entry_comments", entry.entry_comments);
      setValue("entry_wordcount", entry.entry_wordcount);
      setValue("entry_contact_email", String(entry.entry_email));
      setValue("entry_status", String(entry.entry_status));
    } else {
      reset();
      setIsEntryPendingStatus(breakEntryId);
      setEntryStarted(false);
      setEntryEnded(false);
      setButtonType("start");
    }
  }, [entry, setValue, reset]);

  const onSubmit = async (data) => {
    setLoading(true);
    try {
      if (buttonType === "start") {
        const now = new Date();
        const payload = {
          entry_date: now.toISOString().split("T")[0], // "YYYY-MM-DD"
          entry_bookId: Number(data.entry_book_pid),
          entry_chapterId: Number(data.entry_chapter_pid),
          entry_taskId: Number(data.entry_task_pid),
          entry_start_time: now.toTimeString().split(" ")[0], // "HH:MM:SS"
          entry_createdBy: session.user.user_pid,
        };

        const start_result = await createEntryAction(payload);
        if (start_result?.success && start_result.createdEntry?.entry_pid) {
          const createdEntry = start_result.createdEntry;

          toast.success("Entry Start time saved!");
          setEntryId(createdEntry.entry_pid);
          reset({
            entry_book_pid: String(createdEntry.entry_bookId),
            entry_chapter_pid: String(createdEntry.entry_chapterId),
            entry_task_pid: String(createdEntry.entry_taskId),
            entry_start_time: createdEntry.entry_start_time,
            entry_end_date: new Date().toISOString().split("T")[0]
          });
          setEntryStarted(true);
        } else {
          toast.error(start_result?.message || "Failed to create entry.");
        }
      }

      if (buttonType === "end") {
        if (!entryId) {
          toast.error("Entry ID is missing. Cannot end entry.");
          return;
        }

        const payload = {
          entry_end_date: data.entry_end_date,
        };

        const end_result = await updateEntryAction(entryId, payload);
        if (end_result?.success && end_result.entry?.entry_pid) {
          const entry = end_result.entry;
          const chapterDetails = end_result.chapterDetails;
          toast.success("End time updated!");
          setEntryId(entry.entry_pid);
          reset({
            entry_book_pid: entry.entry_bookId.toString(),
            entry_chapter_pid: entry.entry_chapterId.toString(),
            entry_task_pid: entry.entry_taskId.toString(),
            entry_start_time: entry.entry_start_time,
            entry_end_date: entry.entry_end_date,
            entry_end_time: entry.entry_end_time,
            entry_wordcount: chapterDetails?.chap_wordcount || "",
          });
          setEntryEnded(true);
        } else {
          toast.error(end_result?.message || "Failed to update entry.");
        }
      }

      if (buttonType === "update") {
        if (!entryId) {
          toast.error("Entry ID is missing. Cannot update entry.");
          return;
        }

        const formData = new FormData();
        formData.append("entry_chapterId", data.entry_chapter_pid);
        formData.append("entry_comments", data.entry_comments);
        formData.append("entry_wordcount", data.entry_wordcount);
        formData.append("entry_contact_email", data.entry_contact_email);
        formData.append("entry_status", data.entry_status);
        formData.append("action", "update");

        if (data.entry_upload_file && data.entry_upload_file.length > 0) {
          formData.append("entry_upload_file", data.entry_upload_file[0]);
        }

        const Final_result = await updateFinalEntryAction(entryId, formData);
        if (Final_result?.success) {
          toast.success("Work Entry updated successfully!");
          router.push("/entrys/completed");
        } else {
          toast.error(Final_result?.message || "Failed to update entry.");
        }

      }
    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const allProjectManger    = initialBooks.projectManagerBooks;
  const allAltProjectManger = initialBooks.altProjectManagerBooks;
  
  const bookPMOptions = allProjectManger.map((b) => ({
    value: String(b.book_pid),
    label: b.book_title.charAt(0).toUpperCase() + b.book_title.slice(1),
  }));

  const bookAltPMOptions = allAltProjectManger.map((b) => ({
    value: String(b.book_pid),
    label: b.book_title.charAt(0).toUpperCase() + b.book_title.slice(1),
  }));

  // Combine both lists (Project Manager books first, then Alt Project Manager)
  //const combinedBookOptions = [...bookPMOptions, ...bookAltPMOptions];
  // ✅ Grouped options
  const groupedBookOptions = [
    {
      //label: "Select Book",
      options: bookPMOptions,
    },
    {
      label: "Alternative Books",
      options: bookAltPMOptions,
    },
  ];
  

 /*  const bookOptions = initialBooks.map((b) => ({
    value: String(b.book_pid),
    label: b.book_title.charAt(0).toUpperCase() + b.book_title.slice(1),
  })); */

  // const taskOptions = initialtasks.map((t) => ({
  //   value: String(t.task_pid),
  //   label: t.task_name.charAt(0).toUpperCase() + t.task_name.slice(1),
  // }));

  const emailOptions = initialemails.map((e) => ({
    value: String(e.user_pid),
    label: e.user_email.charAt(0).toUpperCase() + e.user_email.slice(1),
  }));


  // const chapterOptions = initialchapters.map((c) => ({
  //   value: String(c.chap_pid),
  //   label: c.chap_title.charAt(0).toUpperCase() + c.chap_title.slice(1),
  // }));

  const selectedTaskId = watch("entry_task_pid");
  const entry_status = watch("entry_status");
  const isStatusOne = entry_status == 1; // 👈 true when status = 1
  const selectedTask = taskOptions.find(
    (task) => task.value === selectedTaskId
  );
  const showConditionalFields =
    selectedTask?.label.toLowerCase() === "first check" ||
    selectedTask?.label.toLowerCase() === "first check - resubmitted";

  // const statusOptions = initialstatus.map((s) => ({
  //   value: String(s.status_pid),
  //   label: s.status_name.charAt(0).toUpperCase() + s.status_name.slice(1),
  // }));
  const statusOptions = initialstatus
  .filter((s) => {
    if (selectedTaskId == 1 || selectedTaskId == 2) {
      return s.status_pid === 1 || s.status_pid === 2;
    } else {
      return s.status_pid === 1 || s.status_pid === 6;
    }
  })
  .map((s) => ({
    value: String(s.status_pid),
    label: s.status_name.charAt(0).toUpperCase() + s.status_name.slice(1),
  }));



  const HandleEndBreak = async () => {
    setLoading(true);
    if (breakId) {
      try {
        const payload = {
          bh_status: "Completed",
        };

        const break_res = await updateBreak(breakId, payload);
        if (break_res?.success) {
          setBreakId(null);
          toast.success(`Break ended successfully!`);
        } else {
          toast.error(break_res?.message || "Failed to create entry.");
        }
      } catch (error) {
        toast.error(error.message || "Unexpected error occurred");
        console.error(error);
      } finally {
        setLoading(false);
      }
    }
  };

  return (
    <div className="max-w-full  px-4 py-4 mx-auto  rounded-lg  space-y-4" style={{ height: '100%' }}>

      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">
              {isEditMode ? "Edit Entry" : "Create New Entry"}
            </h2>
            <p className="text-gray-500 mt-1 text-sm">
              {isEditMode
                ? "Modify the details of the selected entry."
                : "Fill in the details below to add a new entry."}
            </p>
          </div>
          {breakId ? (
            <button
              type="submit"
              onClick={HandleEndBreak}
              className=" px-5 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
            >
              <span>{loading ? "Updating..." : "End Break"}</span>
             {loading && <FiLoader size={20} className="animate-spin" />}

            </button>
          ) : (
            <button
              type="submit"
              onClick={() => setIsbreakModelOpen(true)}
              className=" px-5 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2"
            >
              <span>+ Add Break</span>

            </button>
          )}
        </div>
      </header>
      <div style={{ position: "relative" }}>
        <div
          // style={{
          //   filter: (breakId || isEntryPendingStatus) && !isEditMode ? "blur(4px)" : "none",
          //   pointerEvents: (breakId || isEntryPendingStatus) && !isEditMode ? "none" : "auto",
          //   userSelect: (breakId || isEntryPendingStatus) && !isEditMode ? "none" : "auto",
          // }}
          style={{
            filter: breakId ? "blur(4px)" : (isEntryPendingStatus && !isEditMode) ? "blur(4px)" : "none",
            pointerEvents: breakId ? "none" : (isEntryPendingStatus && !isEditMode) ? "none" : "auto",
            userSelect: breakId ? "none" : (isEntryPendingStatus && !isEditMode) ? "none" : "auto",
          }}
        >
          <form
            onSubmit={handleSubmit(onSubmit)}
            className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl mx-auto"
          >
            {/* LEFT SIDE */}
            <div className="space-y-4">
              {/* Book Title*/}
              {/* <div>
                <label
                  htmlFor="entry_book_pid"
                  className="block text-sm font-medium text-gray-700 mb-1"
                >
                  Book Title <span className="text-red-500">*</span>
                </label>
                <Controller
                  name="entry_book_pid"
                  control={control}
                  render={({ field }) => (
                    <Select
                      {...field}
                      instanceId="entry-book-pid-select"
                      options={bookOptions}
                      onChange={(selected) =>
                        field.onChange(selected ? selected.value : "")
                      }
                      value={
                        bookOptions.find(
                          (option) => option.value === field.value
                        ) || null
                      }
                      placeholder="Select a Book"
                      isClearable
                      inputId="entry_book_pid"
                      styles={{
                        control: (base) => ({
                          ...base,
                          borderColor: errors.entry_book_pid
                            ? "red"
                            : base.borderColor,
                        }),
                      }}
                    />
                  )}
                />
                {errors.entry_book_pid && (
                  <p className="text-sm text-red-500 mt-1">
                    {errors.entry_book_pid.message}
                  </p>
                )}
              </div> */}

              {/* Section List  */} 
              <div>
                <label
                  htmlFor="entry_book_pid"
                  className="block text-sm font-medium text-gray-700 mb-1"
                >
                  Book Title <span className="text-red-500">*</span>
                </label>
                <Controller
                  name="entry_book_pid"
                  control={control}
                  render={({ field }) => (
                    <Select
                      {...field}
                      instanceId="entry-book-pid-select"
                      options={groupedBookOptions}  // ✅ Use grouped options here
                      onChange={(selected) =>
                        field.onChange(selected ? selected.value : "")
                      }
                      value={
                        [...bookPMOptions, ...bookAltPMOptions].find(
                          (option) => option.value === field.value
                        ) || null
                      }
                      placeholder="Select a Book"
                      isClearable
                      inputId="entry_book_pid"
                      styles={{
                        control: (base) => ({
                          ...base,
                          borderColor: errors.entry_book_pid
                            ? "red"
                            : base.borderColor,
                        }),
                      }}
                    />
                  )}
                />
                {errors.entry_book_pid && (
                  <p className="text-sm text-red-500 mt-1">
                    {errors.entry_book_pid.message}
                  </p>
                )}
              </div>

              {/* Chapter Name*/}
              <div>
                <label
                  htmlFor="entry_chapter_pid"
                  className="block text-sm font-medium text-gray-700 mb-1"
                >
                  Chapter Name <span className="text-red-500">*</span>
                </label>
                <Controller
                  name="entry_chapter_pid"
                  control={control}
                  render={({ field }) => (
                    <Select
                      {...field}
                      instanceId="entry-chapter-pid-select"
                      options={chapterOptions}
                      onChange={(selected) =>
                        field.onChange(selected ? selected.value : "")
                      }
                      value={
                        chapterOptions.find(
                          (option) => option.value === field.value
                        ) || null
                      }
                      placeholder="Select a chapter"
                      isClearable
                      inputId="entry_chapter_pid"
                      styles={{
                        control: (base) => ({
                          ...base,
                          borderColor: errors.entry_chapter_pid
                            ? "red"
                            : base.borderColor,
                        }),
                      }}
                    />
                  )}
                />
                {errors.entry_chapter_pid && (
                  <p className="text-sm text-red-500 mt-1">
                    {errors.entry_chapter_pid.message}
                  </p>
                )}
              </div>

              {/* Task Name*/}
              <div>
                <label
                  htmlFor="entry_task_pid"
                  className="block text-sm font-medium text-gray-700 mb-1"
                >
                  Task Name <span className="text-red-500">*</span>
                </label>
                <Controller
                  name="entry_task_pid"
                  control={control}
                  render={({ field }) => (
                    <Select
                      {...field}
                      instanceId="entry-task-pid-select"
                      options={taskOptions}
                      onChange={(selected) =>
                        field.onChange(selected ? selected.value : "")
                      }
                      value={
                        taskOptions.find(
                          (option) => option.value === field.value
                        ) || null
                      }
                      placeholder="Select a task"
                      isClearable
                      inputId="entry_task_pid"
                      styles={{
                        control: (base) => ({
                          ...base,
                          borderColor: errors.entry_task_pid
                            ? "red"
                            : base.borderColor,
                        }),
                      }}
                    />
                  )}
                />
                {errors.entry_task_pid && (
                  <p className="text-sm text-red-500 mt-1">
                    {errors.entry_task_pid.message}
                  </p>
                )}
              </div>

              {/* Comments */}
              {entryEnded && (
                <div>
                  <label
                    htmlFor="entry_comments"
                    className="block text-sm font-medium text-gray-700 mb-1"
                  >
                    Comments <span className="text-red-500">*</span>
                  </label>
                  <textarea
                    id="entry_comments"
                    placeholder="Enter comments"
                    rows={4} // you can change the number of rows as needed
                    {...register("entry_comments")}
                    className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.entry_comments ? "border-red-500" : "border-gray-300"
                      } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                  />
                  {errors.entry_comments && (
                    <p className="text-sm text-red-500 mt-1">
                      {errors.entry_comments.message}
                    </p>
                  )}
                </div>
              )}
              {/* Status*/}
              {entryEnded && (
                <div>
                  <label
                    htmlFor="entry_status"
                    className="block text-sm font-medium text-gray-700 mb-1"
                  >
                    Status<span className="text-red-500">*</span>
                  </label>
                  <Controller
                    name="entry_status"
                    control={control}
                    render={({ field }) => (
                      <Select
                        {...field}
                        instanceId="entry-task-pid-select"
                        options={statusOptions}
                        onChange={(selected) =>
                          field.onChange(selected ? selected.value : "")
                        }
                        value={
                          statusOptions.find(
                            (option) => option.value === field.value
                          ) || null
                        }
                        placeholder="Select a task"
                        isClearable
                        inputId="entry_status"
                        styles={{
                          control: (base) => ({
                            ...base,
                            borderColor: errors.entry_status
                              ? "red"
                              : base.borderColor,
                          }),
                        }}
                      />
                    )}
                  />
                  {errors.entry_status && (
                    <p className="text-sm text-red-500 mt-1">
                      {errors.entry_status.message}
                    </p>
                  )}
                </div>
              )}

            </div>

            {/* RIGHT SIDE */}
            <div className="space-y-4">
              {/* Start Time and Start Button in Same Row */}
              <div className="flex flex-col md:flex-row items-start md:items-end gap-4 w-full">
                <div className="flex-1">
                  <label
                    htmlFor="entry_start_time"
                    className="block text-sm font-medium text-gray-700 mb-1"
                  >
                    Start Time <span className="text-red-500">*</span>
                  </label>
                  <input
                    id="entry_start_time"
                    type="text"
                    readOnly
                    placeholder="Select start time"
                    {...register("entry_start_time")}
                    className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm bg-gray-100 text-gray-700 cursor-not-allowed ${errors.entry_start_time
                      ? "border-red-500"
                      : "border-gray-300"
                      } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                  />
                  {errors.entry_start_time && (
                    <p className="text-sm text-red-500 mt-1">
                      {errors.entry_start_time.message}
                    </p>
                  )}
                </div>
                {!entryStarted ? (
                  <div>
                    <button
                      type="submit"
                      onClick={() => setButtonType("start")}
                      className="px-2 py-1.5 border border-green-600 text-green-600 rounded-md hover:bg-green-100 transition cursor-pointer shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 flex items-center gap-2 justify-center"
                      disabled={loading}
                    >
                      <span>{loading ? "Creating..." : "Start"}</span>
                      {loading ? (
                        <FiLoader size={20} className="animate-spin" />
                      ) : (
                        <FiArrowRightCircle size={20} />
                      )}
                    </button>
                  </div>
                ) : (
                  !entryEnded && (
                    <div>
                      {pendingId ? (
                        <button
                          type="button"
                          onClick={() => setIsResumeModalOpen(true)}
                          className="px-3 py-1.5 border border-green-600 text-green-600 rounded-md hover:bg-green-100 transition cursor-pointer shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 flex items-center gap-2 justify-center"
                        >
                          Resume
                        </button>
                      ) : (
                        <button
                          type="button"
                          onClick={() => setIsPendingModalOpen(true)}
                          className="px-3 py-1.5 border border-yellow-600 text-yellow-600 rounded-md hover:bg-yellow-100 transition cursor-pointer shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-yellow-500 focus:ring-offset-2 flex items-center gap-2 justify-center"
                        >
                          Pending
                        </button>
                      )}
                    </div>
                  )
                )}
              </div>

              {/* End Date */}
              {entryStarted && !pendingId && (
                <>
                  <div style={{ display: 'none' }}>
                    <label
                      htmlFor="entry_end_date"
                      className="block text-sm font-medium text-gray-700 mb-1"
                    >
                      End Date <span className="text-red-500">*</span>
                    </label>
                    <input
                      id="entry_end_date"
                      type="date"
                      // min={getTodayDate()}
                      // max={getTodayDate()}
                      placeholder="Select end date"
                      {...register("entry_end_date")}
                      className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm bg-white text-gray-700 ${errors.entry_end_date
                        ? "border-red-500"
                        : "border-gray-300"
                        } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                    />
                    {errors.entry_end_date && (
                      <p className="text-sm text-red-500 mt-1">
                        {errors.entry_end_date.message}
                      </p>
                    )}
                  </div>
                  {/* End Time and Button in same row */}
                  <div className="flex flex-col md:flex-row items-start md:items-end gap-4 w-full">
                    <div className="flex-1">
                      <label
                        htmlFor="entry_end_time"
                        className="block text-sm font-medium text-gray-700 mb-1"
                      >
                        End Time <span className="text-red-500">*</span>
                      </label>
                      <input
                        id="entry_end_time"
                        type="text"
                        readOnly
                        placeholder="Select end time"
                        {...register("entry_end_time")}
                        className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm bg-gray-100 text-gray-700 cursor-not-allowed ${errors.entry_end_time
                          ? "border-red-500"
                          : "border-gray-300"
                          } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                      />
                      {errors.entry_end_time && (
                        <p className="text-sm text-red-500 mt-1">
                          {errors.entry_end_time.message}
                        </p>
                      )}
                    </div>
                    {!entryEnded && (
                      <div>
                        <button
                          type="submit"
                          onClick={() => setButtonType("end")}
                          className="px-2 py-1.5 border border-green-600 text-green-600 rounded-md hover:bg-green-100 transition cursor-pointer shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 flex items-center gap-2 justify-center"
                          disabled={loading}
                        >
                          <span>{loading ? "Updating..." : "End"}</span>
                          {loading ? (
                            <FiLoader size={20} className="animate-spin" />
                          ) : (
                            <FiArrowRightCircle size={20} />
                          )}
                        </button>
                      </div>
                    )}
                  </div>
                </>
              )}

              {/* Word Count */}
              {entryEnded && (

                <div>
                  <label
                    htmlFor="entry_wordcount"
                    className="block text-sm font-medium text-gray-700 mb-1"
                  >
                    Word Count
                  </label>
                  <input
                    id="entry_wordcount"
                    placeholder="Enter Word Count"
                    type="number"
                    {...register("entry_wordcount")}
                    className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.entry_wordcount ? "border-red-500" : "border-gray-300"
                      } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                  />
                  {errors.entry_wordcount && (
                      <p className="text-sm text-red-500 mt-1">
                        {errors.entry_wordcount.message}
                      </p>
                    )}
                </div>
              )}


              {/* Contact email */}
              {entryEnded && showConditionalFields && (

                <div>
                  <label
                    htmlFor="entry_contact_email"
                    className="block text-sm font-medium text-gray-700 mb-1"
                  >
                    Contact Email {isStatusOne && <span className="text-red-500">*</span>}
                  </label>
                  <Controller
                    name="entry_contact_email"
                    control={control}
                    render={({ field }) => (
                      <Select
                        {...field}
                        instanceId="entry-book-pid-select"
                        options={emailOptions}
                        onChange={(selected) =>
                          field.onChange(selected ? selected.value : "")
                        }
                        value={
                          emailOptions.find(
                            (option) => option.value === field.value
                          ) || ""
                        }
                        placeholder="Select a email"
                        isClearable
                        inputId="entry_contact_email"
                        styles={{
                          control: (base) => ({
                            ...base,
                            borderColor: isStatusOne && errors.entry_contact_email
                              ? "red"
                              : base.borderColor,
                          }),
                        }}
                      />
                    )}
                  />
                  {isStatusOne && errors.entry_contact_email && (
                    <p className="text-sm text-red-500 mt-1">
                      {errors.entry_contact_email.message}
                    </p>
                  )}
                </div>
              )}
              {/* Upload File */}
              {entryEnded && showConditionalFields && (
                <div>
                  <label
                    htmlFor="entry_upload_file"
                    className="block text-sm font-medium text-gray-700 mb-1"
                  >
                    Upload File  {isStatusOne && <span className="text-red-500">*</span>} (Allowed:
                     pdf, doc, docx)
                  </label>

                  <input
                    type="file"
                    id="entry_upload_file"
                    {...register("entry_upload_file")}
                    accept=".pdf,.docx,application/pdf,application/vnd.openxmlformats-officedocument.wordprocessingml.document"
                    className={`block w-full text-sm text-gray-700
                  file:bg-blue-100 file:text-blue-700 file:rounded file:px-4 file:py-2 file:border-0
                  cursor-pointer
                  border rounded-md
                    ${isStatusOne && errors.entry_upload_file
                        ? "border-red-500"
                        : "border-gray-300"
                      }
                  focus:outline-none focus:ring-2 focus:ring-blue-500
                `}
                  />
                  {isStatusOne && errors.entry_upload_file && (
                    <p className="text-sm text-red-500 mt-1">
                      {errors.entry_upload_file.message}
                    </p>
                  )}
                </div>
              )}
            </div>
            {/* Buttons */}
            {entryEnded && (
              <div className="col-span-1 md:col-span-2 flex flex-col sm:flex-row justify-end gap-4">
                <button
                  type="submit"
                  onClick={() => setButtonType("update")}
                  className="w-full sm:w-auto px-5 py-2 text-white rounded-md bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B]transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
                  disabled={loading}
                >
                  <span>{loading ? "Updating..." : "Update"}</span>
                  {loading ? (
                    <FiLoader size={20} className="animate-spin" />
                  ) : (
                    <FiArrowRightCircle size={20} />
                  )}
                </button>
              </div>
            )}
          </form>

        </div>

        {breakId && (
          <div
            style={{
              position: "absolute",
              top: "50%",
              left: "50%",
              transform: "translate(-50%, -50%)",
              background: "rgba(255, 255, 255, 0.9)",
              borderRadius: "8px",
              padding: "20px",
              boxShadow: "0 4px 12px rgba(0,0,0,0.15)",
              fontWeight: "bold",
              fontSize: "1.2rem",
              color: "#1a202c",
              zIndex: 10,
              textAlign: "center",
            }}
          >
            Please end the break to continue.

          </div>
        )}

        {isEntryPendingStatus && !breakId && !isEditMode && (
          <div
            style={{
              position: "absolute",
              top: "50%",
              left: "50%",
              transform: "translate(-50%, -50%)",
              background: "rgba(255, 255, 255, 0.9)",
              borderRadius: "8px",
              padding: "20px",
              boxShadow: "0 4px 12px rgba(0,0,0,0.15)",
              fontWeight: "bold",
              fontSize: "1.2rem",
              color: "#1a202c",
              zIndex: 10,
              textAlign: "center",
            }}
          >
            Please complete the pending entry.
          </div>
        )}
      </div>
      {/* Pending Modal */}
      <PendingModal
        isOpen={isPendingModalOpen}
        onClose={() => setIsPendingModalOpen(false)}
        entryId={entryId}
        onSuccess={(pid) => setPendingId(pid)}
      />

      {/* Resume Modal */}
      <ResumeModal
        isOpen={isResumeModalOpen}
        onClose={() => setIsResumeModalOpen(false)}
        pendingId={pendingId}
        onSuccess={() => setPendingId(null)} // reset after resume
      />

      {/* <TakeBreakModal
          isOpen={isbreakModelOpen}
          onClose={() => setIsbreakModelOpen(false)}
          isEntrySuccess={isbreakEntrySuccess}
          setIsEntrySuccess={setIsbreakEntrySuccess}
          onEndBreak={handleEndBreak}
          setOnBreak={setOnBreak}
          onBreak={onBreak}
          initialbreaks = {initialbreaks}
        /> */}
      <TakeBreakModal
        isOpen={isbreakModelOpen}
        onClose={() => setIsbreakModelOpen(false)}
        initialbreaks={initialbreaks}
        onSuccess={(pid) => setBreakId(pid)}
      />
    </div>
  );
}