"use client";
import { useState, useMemo, useEffect } from "react";
import {
  MagnifyingGlassIcon,
  XMarkIcon,
  EnvelopeIcon,
  ArrowUpIcon,
  ArrowDownIcon,
  TableCellsIcon,
  DocumentTextIcon,
} from "@heroicons/react/24/solid";
import autoTable from "jspdf-autotable";
import { usePathname } from 'next/navigation';
import Pagination from "./Pagination";

function ModalWrapper({ content, text, maxChars = 20, modalHeader }) {
  const [open, setOpen] = useState(false);

  if (!text) return content;

  const strValue = text.toString();
  const isLong = strValue.length > maxChars;
  const displayText = isLong ? strValue.slice(0, maxChars) + "..." : strValue;

  const renderedContent =
    typeof content === "function"
      ? content(displayText, isLong)
      : typeof content === "string"
        ? displayText
        : content;

  return (
    <>
      <span
        className={`${isLong ? "cursor-pointer text-blue-600 hover:underline" : "text-gray-800"}`}
        onClick={() => isLong && setOpen(true)}
      >
        {renderedContent}
      </span>

      {open && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm">
          <div className="relative bg-white rounded-2xl shadow-2xl max-w-2xl w-[90%] max-h-[85vh] overflow-y-auto p-8 animate-fadeIn">
            <button
              onClick={() => setOpen(false)}
              className="absolute top-4 right-4 text-gray-500 hover:text-gray-800 transition"
            >
              <XMarkIcon className="w-7 h-7" />
            </button>
            <h3 className="text-2xl font-bold mb-4 text-gray-900 border-b pb-3">
              {modalHeader}
            </h3>
            <p className="whitespace-pre-wrap text-gray-700 leading-relaxed text-[15px]">
              {strValue}
            </p>
            <div className="mt-8 flex justify-end">
              <button
                onClick={() => setOpen(false)}
                className="px-5 py-2.5 bg-gradient-to-r from-blue-500 to-blue-700 text-white font-medium rounded-lg shadow hover:from-blue-600 hover:to-blue-800 transition"
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  );
}

export default function CustomDataTable({ columns, data, defaultSort, page, limit, totaldata }) {
  console.log( page, limit, totaldata ," page, limit, totaldata ")
  const pathname = usePathname();
  const segments = pathname.split('/');
  const segmentValue = segments[segments.length - 1];

  const [search, setSearch] = useState("");
  const [sortConfig, setSortConfig] = useState({
    key: defaultSort?.key || null,
    direction: defaultSort?.direction || null,
  });
  const [visibleColumns, setVisibleColumns] = useState(columns.map((col) => col.label));
  const [showDropdown, setShowDropdown] = useState(false);

  // Reset sort when new data comes from server
  useEffect(() => {
    setSortConfig({ key: defaultSort?.key || null, direction: defaultSort?.direction || null });
  }, [data, defaultSort]);

  const displayedColumns = useMemo(() => {
    return columns.filter((col) => visibleColumns.includes(col.label));
  }, [columns, visibleColumns]);

  const handleSort = (col) => {
    if (!col.selector) return;
    let direction = "ascending";
    if (sortConfig.key === col.label && sortConfig.direction === "ascending") {
      direction = "descending";
    }
    setSortConfig({ key: col.label, direction });
  };

  // Client-side search + sort on already fetched page data
  const processedData = useMemo(() => {
    if (!Array.isArray(data)) return [];

    let filtered = data.filter((item) =>
      displayedColumns.some((col) => {
        if (!col.selector) return false;
        const value = col.selector(item);
        return (
          value && value.toString().toLowerCase().includes(search.toLowerCase())
        );
      })
    );

    if (sortConfig.key) {
      const col = columns.find((c) => c.label === sortConfig.key);
      if (col && col.selector) {
        filtered.sort((a, b) => {
          const aVal = col.selector(a);
          const bVal = col.selector(b);
          if (aVal == null) return 1;
          if (bVal == null) return -1;
          if (typeof aVal === "number" && typeof bVal === "number") {
            return sortConfig.direction === "ascending" ? aVal - bVal : bVal - aVal;
          }
          return sortConfig.direction === "ascending"
            ? aVal.toString().localeCompare(bVal.toString())
            : bVal.toString().localeCompare(aVal.toString());
        });
      }
    }

    return filtered;
  }, [search, data, columns, sortConfig, displayedColumns]);

  const renderCell = (col, row, rowIndex) => {
    const value = col.selector ? col.selector(row) : "";
    if (col.label === "S.No") return (page - 1) * limit + rowIndex + 1;
    if (col.cell) return col.cell(row, rowIndex);

    return (
      <ModalWrapper
        content={value}
        text={value}
        maxChars={20}
        modalHeader={col?.label}
      />
    );
  };

  const basePath = `/${segments.slice(1, segments.length - 1).join('/')}/${segmentValue}`;

  return (
    <div className="bg-white rounded-xl shadow-md border border-gray-200 overflow-hidden text-gray-800">
      {/* Toolbar */}
      <div className="p-3 flex justify-end items-center gap-3 border-b border-gray-200 bg-gray-50">
        <div className="relative w-64">
          <MagnifyingGlassIcon className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-blue-500" />
          <input
            type="text"
            placeholder="Search..."
            value={search}
            onChange={(e) => setSearch(e.target.value)}
            className="w-full pl-10 pr-8 py-1 text-sm bg-white border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          />
          {search && (
            <button
              onClick={() => setSearch("")}
              className="absolute right-2 top-1/2 -translate-y-1/2 text-gray-400 hover:text-blue-500"
              type="button"
            >
              <XMarkIcon className="w-5 h-5" />
            </button>
          )}
        </div>

        {/* Column visibility toggle */}
        <div className="relative">
          <button
            onClick={() => setShowDropdown(!showDropdown)}
            className="px-2 py-1 text-sm text-white bg-green-600 border border-green-700 rounded-md hover:bg-green-700"
          >
            Show/Hide Col.
          </button>
          {showDropdown && (
            <div className="absolute right-0 mt-2 w-48 bg-white border rounded shadow-lg z-50 max-h-60 overflow-y-auto">
              <label className="flex items-center p-2 cursor-pointer hover:bg-gray-100 border-b">
                <input
                  type="checkbox"
                  checked={visibleColumns.length === columns.length}
                  onChange={() => {
                    if (visibleColumns.length === columns.length) {
                      setVisibleColumns([]);
                    } else {
                      setVisibleColumns(columns.map((col) => col.label));
                    }
                  }}
                  className="mr-2"
                />
                Select All
              </label>
              {columns.map((col) => (
                <label key={col.label} className="flex items-center p-2 cursor-pointer hover:bg-gray-100">
                  <input
                    type="checkbox"
                    checked={visibleColumns.includes(col.label)}
                    onChange={() => {
                      setVisibleColumns((prev) =>
                        prev.includes(col.label)
                          ? prev.filter((l) => l !== col.label)
                          : [...prev, col.label]
                      );
                    }}
                    className="mr-2"
                  />
                  {col.label}
                </label>
              ))}
            </div>
          )}
        </div>

        {/* Exports (optional) */}
        {segmentValue === "pending" || segmentValue === "completed" ? (
          <>
            <button className="px-2 py-1 text-sm text-white bg-blue-600 border border-blue-700 rounded-md hover:bg-blue-700 flex items-center gap-1">
              <TableCellsIcon className="w-4 h-4" /> Excel
            </button>
            <button className="px-2 py-1 text-sm text-white bg-red-600 border border-red-700 rounded-md hover:bg-red-700 flex items-center gap-1">
              <DocumentTextIcon className="w-4 h-4" /> PDF
            </button>
          </>
        ) : null}
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="min-w-full border-separate border-spacing-0">
          <thead>
            <tr className="bg-gray-100 sticky top-0 z-10">
              {displayedColumns.map((col, i) => (
                <th
                  key={i}
                  className="px-4 py-2 text-left text-sm font-semibold text-gray-700 border-b border-gray-200 cursor-pointer hover:text-blue-600"
                  onClick={() => handleSort(col)}
                >
                  <div className="flex items-center space-x-1">
                    <span>{col.label}</span>
                    {sortConfig.key === col.label &&
                      (sortConfig.direction === "ascending" ? (
                        <ArrowUpIcon className="w-4 h-4 text-blue-500" />
                      ) : (
                        <ArrowDownIcon className="w-4 h-4 text-blue-500" />
                      ))}
                  </div>
                </th>
              ))}
            </tr>
          </thead>
          <tbody>
            {processedData.length > 0 ? (
              processedData.map((row, rowIndex) => (
                <tr key={rowIndex} className="odd:bg-white even:bg-gray-50 hover:bg-blue-50 transition-colors">
                  {displayedColumns.map((col, colIndex) => (
                    <td key={colIndex} className="px-6 py-2 text-sm border-b border-gray-200">
                      {renderCell(col, row, rowIndex)}
                    </td>
                  ))}
                </tr>
              ))
            ) : (
              <tr>
                <td colSpan={displayedColumns.length} className="text-center py-8 text-gray-500">
                  No data found
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Server-side Pagination */}
      <Pagination
        page={page}
        limit={limit}
        totaldata={totaldata}
        basePath={basePath}
      />
    </div>
  );
}