"use client";
import { useState, useMemo, useEffect } from "react";
import {
  MagnifyingGlassIcon,
  XMarkIcon,
  EnvelopeIcon,
  ArrowUpIcon,
  ArrowDownIcon,
  TableCellsIcon,
  DocumentTextIcon,
} from "@heroicons/react/24/solid";
// import * as XLSX from "xlsx";
// import jsPDF from "jspdf";
import autoTable from "jspdf-autotable"; // Explicit import
import { usePathname } from 'next/navigation';
import Pagination from "./Pagination";

function ModalWrapper({ content, text, maxChars = 20, modalHeader }) {
  const [open, setOpen] = useState(false);

  if (!text) return content;

  const strValue = text.toString();
  const isLong = strValue.length > maxChars;
  const displayText = isLong ? strValue.slice(0, maxChars) + "..." : strValue;

  const renderedContent =
    typeof content === "function"
      ? content(displayText, isLong)
      : typeof content === "string"
        ? displayText
        : content;

  return (
    <>
      <span
        className={`${isLong
            ? "cursor-pointer text-blue-600 hover:underline"
            : "text-gray-800"
          }`}
        onClick={() => isLong && setOpen(true)}
      >
        {renderedContent}
      </span>

      {open && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/60 backdrop-blur-sm">
          <div className="relative bg-white rounded-2xl shadow-2xl max-w-2xl w-[90%] max-h-[85vh] overflow-y-auto p-8 animate-fadeIn">
            <button
              onClick={() => setOpen(false)}
              className="absolute top-4 right-4 text-gray-500 hover:text-gray-800 transition"
            >
              <XMarkIcon className="w-7 h-7" />
            </button>
            <h3 className="text-2xl font-bold mb-4 text-gray-900 border-b pb-3">
              {modalHeader}
            </h3>
            <p className="whitespace-pre-wrap text-gray-700 leading-relaxed text-[15px]">
              {strValue}
            </p>
            <div className="mt-8 flex justify-end">
              <button
                onClick={() => setOpen(false)}
                className="px-5 py-2.5 bg-gradient-to-r from-blue-500 to-blue-700 text-white font-medium rounded-lg shadow hover:from-blue-600 hover:to-blue-800 transition"
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  );
}

export default function CustomDataTable({ columns, data, defaultSort, page = 1, limit, totalUsers}) {
  const pathname = usePathname();
  const segments = pathname.split('/');
  const segmentValue = segments[segments.length - 1];
  const [search, setSearch] = useState("");
  const [currentPage, setCurrentPage] = useState(1);
  const [perPage, setPerPage] = useState(limit || 10);
  const [sortConfig, setSortConfig] = useState({
    key: defaultSort?.key || null,
    direction: defaultSort?.direction || null,
  });
  const [visibleColumns, setVisibleColumns] = useState(
    columns.map((col) => col.label)
  );
  const [showDropdown, setShowDropdown] = useState(false);

  const displayedColumns = useMemo(() => {
    return columns.filter((col) => visibleColumns.includes(col.label));
  }, [columns, visibleColumns]);

  const handleSort = (col) => {
    if (!col.selector) return;
    let direction = "ascending";
    if (sortConfig.key === col.label && sortConfig.direction === "ascending") {
      direction = "descending";
    }
    setSortConfig({ key: col.label, direction });
  };

  const filteredData = useMemo(() => {
    if (!Array.isArray(data)) return [];
    let filtered = data.filter((item) =>
      displayedColumns.some((col) => {
        if (!col.selector) return false;
        const value = col.selector(item);
        return (
          value && value.toString().toLowerCase().includes(search.toLowerCase())
        );
      })
    );

    if (sortConfig.key) {
      const col = columns.find((c) => c.label === sortConfig.key);
      if (col && col.selector) {
        filtered.sort((a, b) => {
          const aVal = col.selector(a);
          const bVal = col.selector(b);
          if (aVal == null) return 1;
          if (bVal == null) return -1;
          if (typeof aVal === "number" && typeof bVal === "number") {
            return sortConfig.direction === "ascending"
              ? aVal - bVal
              : bVal - aVal;
          }
          return sortConfig.direction === "ascending"
            ? aVal.toString().localeCompare(bVal.toString())
            : bVal.toString().localeCompare(aVal.toString());
        });
      }
    }
    return filtered;
  }, [search, data, columns, sortConfig, displayedColumns]);

  const totalPages = Math.ceil(filteredData.length / perPage) || 1;
  const paginatedData = useMemo(() => {
    const start = (currentPage - 1) * perPage;
    return filteredData.slice(start, start + perPage);
  }, [filteredData, currentPage, perPage]);

  const getPageNumbers = () => {
    const maxPagesToShow = 5;
    if (totalPages <= maxPagesToShow)
      return [...Array(totalPages).keys()].map((i) => i + 1);
    const pages = [];
    if (currentPage > 3) pages.push(1, "...");
    else pages.push(1, 2, 3);
    if (currentPage > 3 && currentPage < totalPages - 2) {
      pages.push(currentPage - 1, currentPage, currentPage + 1);
    }
    if (currentPage < totalPages - 2) pages.push("...", totalPages);
    else pages.push(totalPages - 2, totalPages - 1, totalPages);
    return [...new Set(pages)];
  };

  const renderCell = (col, row, rowIndex) => {
    const value = col.selector ? col.selector(row) : "";
    if (col.label === "S.No") return (currentPage - 1) * perPage + rowIndex + 1;
    if (col.cell) return col.cell(row, rowIndex);

    const colorMap = {
      success:
        "bg-green-100/30 text-green-700 backdrop-blur-sm border border-green-200",
      danger:
        "bg-red-100/30 text-red-700 backdrop-blur-sm border border-red-200",
      secondary:
        "bg-gray-100/30 text-gray-700 backdrop-blur-sm border border-gray-200",
      warning:
        "bg-yellow-100/30 text-yellow-700 backdrop-blur-sm border border-yellow-200",
      info: "bg-blue-100/30 text-blue-700 backdrop-blur-sm border border-blue-200",
      light:
        "bg-white/20 text-gray-800 backdrop-blur-sm border border-gray-200",
      dark: "bg-gray-200/30 text-gray-800 backdrop-blur-sm border border-gray-300",
      primary:
        "bg-blue-200/30 text-blue-800 backdrop-blur-sm border border-blue-300",
      white:
        "bg-white/20 text-gray-700 backdrop-blur-sm border border-gray-200",
      red: "bg-red-200/30 text-red-800 backdrop-blur-sm border border-red-300",
      orange:
        "bg-orange-200/30 text-orange-800 backdrop-blur-sm border border-orange-300",
      yellow:
        "bg-yellow-200/30 text-yellow-800 backdrop-blur-sm border border-yellow-300",
      green:
        "bg-green-200/30 text-green-800 backdrop-blur-sm border border-green-300",
      teal: "bg-teal-200/30 text-teal-800 backdrop-blur-sm border border-teal-300",
      text_success: "text-green-600 font-medium backdrop-blur-sm",
      text_danger: "text-red-600 font-medium backdrop-blur-sm",
      text_secondary: "text-purple-700 font-medium backdrop-blur-sm",
      text_warning: "text-yellow-600 font-medium backdrop-blur-sm",
      text_info: "text-blue-600 font-medium backdrop-blur-sm",
      text_light: "text-gray-500 font-medium backdrop-blur-sm",
      text_dark: "text-gray-900 font-medium backdrop-blur-sm",
      text_primary: "text-blue-800 font-medium backdrop-blur-sm",
      text_white: "text-gray-200 font-medium backdrop-blur-sm",
      text_red: "text-red-700 font-medium backdrop-blur-sm",
      text_orange: "text-orange-600 font-medium backdrop-blur-sm",
      text_yellow: "text-yellow-700 font-medium backdrop-blur-sm",
      text_green: "text-green-700 font-medium backdrop-blur-sm",
      text_teal: "text-teal-600 font-medium backdrop-blur-sm",
    };

    const colorClass = col.styleTypeColors
      ? colorMap[col.styleTypeColors]
      : "text-gray-800";
    let content = value;

    if (
      col.styleTypeUiPatterns === "badge" &&
      col.styleTypeIcons === "mailIcon"
    ) {
      content = (displayText) => (
        <span
          className={`inline-flex items-center px-2 py-1 text-xs rounded-full ${colorClass}`}
        >
          <EnvelopeIcon className="w-3 h-3 mr-1" />
          {displayText}
        </span>
      );
    } else if (col.styleTypeIcons === "avatar") {
      const strValue = value?.toString() || "";
      const initials =
        strValue
          .split(" ")
          .map((word) => word[0])
          .join("")
          .substring(0, 2)
          .toUpperCase() || "?";
      content = (
        <div className="flex items-center space-x-2">
          <div className="h-8 w-8 flex items-center justify-center rounded-full bg-blue-500 text-white font-bold text-sm">
            {initials}
          </div>
          <span>{value}</span>
        </div>
      );
    } else if (col.styleTypeIcons === "mailIcon") {
      content = (
        <div className="flex items-center text-blue-600">
          <EnvelopeIcon className="w-4 h-4 mr-1" />
          <span>{value}</span>
        </div>
      );
    } else if (col.styleTypeUiPatterns === "badge") {
      content = (displayText) => (
        <span
          className={`inline-block px-2 py-1 text-xs rounded-full ${colorClass}`}
        >
          {displayText}
        </span>
      );
    } else if (col.styleTypeUiPatterns === "boxModal") {
      content = (displayText) => (
        <div className={`p-2 rounded-lg shadow-sm ${colorClass}`}>
          {displayText}
        </div>
      );
    } else if (col.styleTypeColors) {
      content = (displayText) => (
        <span className={`${colorClass} transition-colors duration-200`}>
          {displayText}
        </span>
      );
    }

    return (
      <ModalWrapper
        content={content}
        text={value}
        maxChars={20}
        modalHeader={col?.label}
      />
    );
  };

  // const handleDownloadExcel = () => {
  //   const dataToExport = filteredData.map((row) => {
  //     const exportRow = {};
  //     displayedColumns.forEach((col) => {
  //       if (col.label === "S.No") {
  //         exportRow[col.label] = filteredData.indexOf(row) + 1;
  //       } else if (col.selector) {
  //         exportRow[col.label] = col.selector(row);
  //       }
  //     });
  //     return exportRow;
  //   });
  //   const headers = Object.keys(dataToExport[0]);
  //   const aoaData = [
  //     ["","","",`${segmentValue.charAt(0).toUpperCase() + segmentValue.slice(1)} Reports`],
  //     headers,
  //     ...dataToExport.map(row => Object.values(row))
  //   ];
  //   const worksheet = XLSX.utils.aoa_to_sheet(aoaData);
  //   // const worksheet = XLSX.utils.json_to_sheet(dataToExport);
  //   const workbook = XLSX.utils.book_new();
  //   XLSX.utils.book_append_sheet(workbook, worksheet, "Sheet1");
  //   const filename = `${segmentValue.charAt(0).toUpperCase() + segmentValue.slice(1)}.xlsx`;
  //   XLSX.writeFile(workbook, filename);
  // };

  // const handleDownloadPdf = () => {
  //   const doc = new jsPDF();

  //   const imageurl = '/logo.jpg';
  //   doc.addImage(imageurl, 'JPEG', 15, 5, 50, 15);
  //   doc.setFontSize(18);
  //   doc.setTextColor(40);
  //   doc.text(`${segmentValue.charAt(0).toUpperCase() + segmentValue.slice(1)} Report`, 70, 15); 

  //   const startY = 25;
  //   const pdfColumns = displayedColumns.filter(col => col.label !== "Action");
  //   const tableColumn = pdfColumns.map((col) => col.label);
  //   const tableRows = [];

  //   filteredData.forEach((row, rowIndex) => {
  //     const rowData = pdfColumns.map((col) => {
  //       if (col.label === "S.No") {
  //         return rowIndex + 1;
  //       } else if (col.selector) {
  //         return col.selector(row)?.toString() || "";
  //       }
  //       return "";
  //     });
  //     tableRows.push(rowData);
  //   });

  //   autoTable(doc, {
  //     head: [tableColumn],
  //     body: tableRows,
  //     startY: startY,
  //     theme: "grid",
  //     styles: {
  //       cellPadding: 2,
  //       fontSize: 10,
  //       valign: "middle",
  //       halign: "left",
  //     },
  //     headStyles: {
  //       fillColor: [30, 144, 255],
  //       textColor: [255, 255, 255],
  //       fontStyle: "bold",
  //     },
  //     alternateRowStyles: {
  //       fillColor: [240, 240, 240],
  //     },
  //     margin: { top: 10 },
  //   });
  //   const filename = `${segmentValue.charAt(0).toUpperCase() + segmentValue.slice(1)}.pdf`;
  //   doc.save(filename);
  // };

  return (
    <div className="bg-white rounded-xl shadow-md border border-gray-200 rounded-md overflow-hidden text-gray-800">
      <div className="p-3 flex justify-end items-center gap-3 border-b border-gray-200 bg-gray-50">
        <div className="relative w-64">
          <MagnifyingGlassIcon className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-blue-500" />
          <input
            type="text"
            placeholder="Search..."
            value={search}
            onChange={(e) => {
              setSearch(e.target.value);
              setCurrentPage(1);
            }}
            className="w-full pl-10 pr-8 py-1 text-sm bg-white border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          />
          {search && (
            <button
              onClick={() => setSearch("")}
              className="absolute right-2 top-1/2 -translate-y-1/2 text-gray-400 hover:text-blue-500"
              type="button"
            >
              <XMarkIcon className="w-5 h-5" />
            </button>
          )}
        </div>
        <div className="relative">
          <button
            onClick={() => setShowDropdown(!showDropdown)}
            className="px-2 py-1 text-sm text-white bg-green-600 border border-green-700 rounded-md hover:bg-green-700 cursor-pointer"
          >
            Show/Hide Col.
          </button>
          {showDropdown && (
            // <div className="absolute right-0 mt-2 w-48 bg-white border rounded shadow-lg z-50 max-h-60 overflow-y-auto">
            //     {columns.map((col) => (
            //         <label key={col.label} className="flex items-center p-2 cursor-pointer hover:bg-gray-100">
            //             <input
            //                 type="checkbox"
            //                 checked={visibleColumns.includes(col.label)}
            //                 onChange={() => {
            //                     if (visibleColumns.length === 1 && visibleColumns.includes(col.label)) {
            //                         alert("You must have at least one column visible.");
            //                     } else {
            //                         setVisibleColumns(prev =>
            //                             prev.includes(col.label)
            //                                 ? prev.filter(l => l !== col.label)
            //                                 : [...prev, col.label]
            //                         );
            //                     }
            //                 }}
            //                 className="mr-2"
            //             />
            //             {col.label}
            //         </label>
            //     ))}
            // </div>
            <div className="absolute right-0 mt-2 w-48 bg-white border rounded shadow-lg z-50 max-h-60 overflow-y-auto">
              {/* Enable All / Select All */}
              <label className="flex items-center p-2 cursor-pointer hover:bg-gray-100 border-b">
                <input
                  type="checkbox"
                  checked={visibleColumns.length === columns.length}
                  onChange={() => {
                    if (visibleColumns.length === columns.length) {
                      // Deselect all columns
                      setVisibleColumns([]);
                    } else {
                      // Select all columns
                      setVisibleColumns(columns.map((col) => col.label));
                    }
                  }}
                  className="mr-2"
                />
                Select All
              </label>

              {/* Individual Checkboxes */}
              {columns.map((col) => (
                <label
                  key={col.label}
                  className="flex items-center p-2 cursor-pointer hover:bg-gray-100"
                >
                  <input
                    type="checkbox"
                    checked={visibleColumns.includes(col.label)}
                    onChange={() => {
                      setVisibleColumns(
                        (prev) =>
                          prev.includes(col.label)
                            ? prev.filter((l) => l !== col.label) // remove
                            : [...prev, col.label] // add
                      );
                    }}
                    className="mr-2"
                  />
                  {col.label}
                </label>
              ))}
            </div>
          )}
        </div>
        {segmentValue === "pending" || segmentValue === "completed" ? (
          <>
            <button className="px-2 py-1 text-sm text-white bg-blue-600 border border-blue-700 rounded-md hover:bg-blue-700 cursor-pointer flex items-center gap-1">
              <TableCellsIcon className="w-4 h-4" />
              Excel
            </button>
            <button className="px-2 py-1 text-sm text-white bg-red-600 border border-red-700 rounded-md hover:bg-red-700 cursor-pointer flex items-center gap-1">
              <DocumentTextIcon className="w-4 h-4" />
              PDF
            </button>
          </>
        ) : null}
      </div>


      <div className="overflow-x-auto">
        <table className="min-w-full border-separate border-spacing-0">
          <thead>
            <tr className="bg-gray-100 sticky top-0 z-10">
              {displayedColumns.map((col, i) => (
                <th
                  key={i}
                  className="px-4 py-2 text-left text-sm font-semibold text-gray-700 border-b border-gray-200 cursor-pointer hover:text-blue-600"
                  onClick={() => handleSort(col)}
                >
                  <div className="flex items-center space-x-1">
                    <span>{col.label}</span>
                    {sortConfig.key === col.label &&
                      (sortConfig.direction === "ascending" ? (
                        <ArrowUpIcon className="w-4 h-4 text-blue-500" />
                      ) : (
                        <ArrowDownIcon className="w-4 h-4 text-blue-500" />
                      ))}
                  </div>
                </th>
              ))}
            </tr>
          </thead>
          <tbody>
            {paginatedData.length > 0 ? (
              paginatedData.map((row, rowIndex) => (
                <tr
                  key={rowIndex}
                  className="odd:bg-white even:bg-gray-50 hover:bg-blue-50 transition-colors"
                >
                  {displayedColumns.map((col, colIndex) => (
                    <td
                      key={colIndex}
                      className="px-6 py-2 text-sm border-b border-gray-200"
                    >
                      {renderCell(col, row, rowIndex)}
                    </td>
                  ))}
                </tr>
              ))
            ) : (
              <tr>
                <td
                  colSpan={displayedColumns.length}
                  className="text-center py-8 text-gray-500"
                >
                  No data found
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>
      <Pagination
        page={page}
        limit={limit}
        totalUsers={totalUsers}
        basePath="/users/list"
      />
      {/* <div className="flex flex-col sm:flex-row justify-between items-center gap-3 p-2 border-t border-gray-200 text-sm bg-gray-50">
        <span className="text-gray-600">
          Page {currentPage} of {totalPages}
        </span>
        <div className="flex items-center gap-2 text-gray-700">
          <span>Rows per page:</span>
          <select
            value={perPage}
            onChange={(e) => {
              setPerPage(Number(e.target.value));
              setCurrentPage(1);
            }}
            className="border border-gray-300 rounded px-3 py-1 text-sm bg-white focus:ring-2 focus:ring-blue-500"
          >
            <option value={5}>5</option>
            <option value={10}>10</option>
            <option value={25}>25</option>
          </select>
        </div>
        <div className="flex items-center space-x-2">
          <button
            onClick={() => setCurrentPage((p) => Math.max(1, p - 1))}
            disabled={currentPage === 1}
            className="px-4 py-1 border border-gray-300 rounded text-blue-600 hover:bg-blue-100 disabled:opacity-50"
          >
            Prev
          </button>
          {getPageNumbers().map((num, idx) =>
            num === "..." ? (
              <span key={idx} className="px-3 text-gray-500 select-none">
                ...
              </span>
            ) : (
              <button
                key={idx}
                onClick={() => setCurrentPage(num)}
                className={`px-4 py-1 border rounded ${
                  currentPage === num
                    ? "bg-blue-500 text-white border-blue-500"
                    : "border-gray-300 text-gray-700 hover:bg-blue-100"
                }`}
              >
                {num}
              </button>
            )
          )}
          <button
            onClick={() => setCurrentPage((p) => Math.min(totalPages, p + 1))}
            disabled={currentPage === totalPages}
            className="px-4 py-1 border border-gray-300 rounded text-blue-600 hover:bg-blue-100 disabled:opacity-50"
          >
            Next
          </button>
        </div>
      </div> */}
    </div>
  );
}
