// components/dataTable/childDataTable.js
import { useMemo } from "react";
import { FiEdit, FiDownload, FiLayers } from "react-icons/fi";
import { Tooltip } from "react-tooltip";
import { useRouter } from "next/navigation";
import { encryptId } from "@/app/actions/encrypt";

export default function ChildDataTable({ chapters = [] }) {
    const router = useRouter();
    const stageCodeToName = {
        FC: "First Check",
        FCR: "First Check Resubmit",
        FCQC: "First Check QC",
        FCRQC: "First Check Resubmit QC",
        EDA: "Editor Assignment",
        PR: "Peer Review",
        FD: "Final Decision",
    };
    const columns = useMemo(
        () => [
            {
                label: 'S.No',
                selector: (_row, index) => index + 1,
            },
            {
                label: "Manuscript ID",
                selector: (row) => row.chap_manuscriptID,
            },
            {
                label: "Chapter Title",
                selector: (row) => row.chap_title,
            },
            {
                label: "Word Count",
                selector: (row) => row.chap_wordcount,
            },
            {
                label: "Author",
                selector: (row) => `${row.chap_firstName} ${row.chap_lastName}`.trim(),
            },
            {
                label: "Email",
                selector: (row) => row.chap_corresAuEmail,
            },
            {
                label: "Status",
                selector: (row) => row.chap_status,
            },
            {
                label: "P+ Status",
                selector: (row) => row.chap_pStatus,
            },
            {
                label: "Due Date",
                selector: (row) => row.chap_pdueDate,
            },
            {
                label: "Copy Editing",
                selector: (row) => row.chap_copyEditing,
            },
            {
                label: "Comments",
                selector: (row) => row.chap_comments,
                grow: 2, // custom grow factor for width
                wrap: true,
            },
            {
                label: "Current Stage",
                selector: (row) => stageCodeToName[row.chap_currentStage] || row.chap_currentStage,
            },
            {
                label: "Action",
                selector: (row) => (
                    <div className="flex space-x-3">

                        <button
                            onClick={async () => {
                                try {
                                    const encryptedId = await encryptId(row.chap_pid);
                                    router.push(`/stage_movements/${encryptedId}`);
                                } catch (error) {
                                    console.error('Encryption failed:', error);
                                }
                            }}
                            data-tooltip-id={`edit-tooltip-${row.chap_pid}`}
                            data-tooltip-content="View Stage movements1"
                            className="text-blue-600 hover:text-blue-800 cursor-pointer"
                            type="button"
                        >
                            <FiLayers size={20} />
                            <Tooltip
                                id={`edit-tooltip-${row.chap_pid}`}
                                place="bottom"
                                style={{
                                    backgroundColor: "#3b82f6",
                                    color: "#fff",
                                    fontSize: "12px",
                                    borderRadius: "4px",
                                    padding: "6px 10px",
                                }}
                            />
                        </button>
                    </div>
                ),
                ignoreRowClick: true,
            },
        ],
        []
    );

    if (!chapters.length) {
        return (
            <div className="p-4 text-center text-gray-500">
                No chapters found for this book.
            </div>
        );
    }

    return (
        <div className="bg-blue-50 rounded-lg shadow-inner mt-4 border border-blue-100 overflow-x-auto">
            <table className="min-w-full">
                <thead>
                    <tr className="bg-pink-50">
                        {columns.map((col, i) => (
                            <th
                                key={i}
                                className="px-6 py-3 text-left text-xs font-bold text-black uppercase tracking-wider"
                                style={col.grow ? { width: `${col.grow * 150}px` } : {}}
                            >
                                {col.label}
                            </th>
                        ))}
                    </tr>
                </thead>
                <tbody>
                    {chapters.map((row, rowIndex) => (
                        <tr key={rowIndex} className="even:bg-gray-50">
                            {columns.map((col, colIndex) => {
                                const value = col.selector(row, rowIndex);

                                // Apply badge for Current Stage column
                                if (col.label === "Current Stage") {
                                    return (
                                        <td
                                            key={colIndex}
                                            className="px-6 py-4 whitespace-nowrap text-sm"
                                            style={col.grow ? { width: `${col.grow * 150}px` } : {}}
                                        >
                                            <span className="inline-block px-2 py-1 text-xs font-semibold text-white bg-green-500 rounded-md">
                                                {value}
                                            </span>
                                        </td>
                                    );
                                }

                                return (
                                    <td
                                        key={colIndex}
                                        className={`px-6 py-4 whitespace-pre-wrap text-sm text-gray-800 ${col.wrap ? "break-words" : ""
                                            }`}
                                        style={col.grow ? { width: `${col.grow * 150}px` } : {}}
                                    >
                                        {value}
                                    </td>
                                );
                            })}
                        </tr>
                    ))}
                </tbody>
            </table>
        </div>
    );
}
