"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import CustomDataTable from "../dataTable/customDataTable";
import { getBookColumns } from "./columns";
import { getCopyeditingColumns } from "./columnsCE";
import { FiBookOpen, FiCheckCircle, FiUsers, FiCalendar, FiEdit } from 'react-icons/fi';
import { motion } from 'framer-motion';
import Select from "react-select";
import selectStyles from "@/app/components/selectStyles";

export default function BooksList({ initialBooks, initialCopyediting, role }) {
  const [data, setData] = useState(initialBooks || []);
  const [copyeditingData, setCopyeditingData] = useState(initialCopyediting || []);
  const [loading, setLoading] = useState(true);
  const [filteredData, setFilteredData] = useState([]);
  const [filteredCopyeditingData, setFilteredCopyeditingData] = useState([]);
  const [yearOptions, setYearOptions] = useState([]);
  const [activeTab, setActiveTab] = useState('yearWise');
  const [selectedYear, setSelectedYear] = useState(null);

  useEffect(() => {
    setLoading(false);
    setData(initialBooks || []);
    setFilteredData(initialBooks || []);
    setLoading(false);
  }, [initialBooks]);

  useEffect(() => {
    setLoading(false);
    setCopyeditingData(initialCopyediting || []);
    setFilteredCopyeditingData(initialCopyediting || []);
    setLoading(false);
  }, [initialCopyediting]);

  useEffect(() => {
    const currentYear = new Date().getFullYear();
    const years = [];

    for (let i = 0; i < 10; i++) {
      const year = currentYear + i;
      years.push({ value: year.toString(), label: `${year}_Working Title` });
    }

    setYearOptions(years);
    setSelectedYear({ value: currentYear.toString(), label: `${currentYear}_Working Title` });
  }, []);

  useEffect(() => {
    const filtered = data.filter((item) => {
      const itemDate = new Date(item.book_createdAt).getFullYear().toString();
      const matchesYear = selectedYear ? itemDate === selectedYear.value : true;

      return (
        matchesYear
      );
    });
    setFilteredData(filtered);
  }, [selectedYear, data]);

  useEffect(() => {
    const filtered = copyeditingData.filter((item) => {
      const itemDate = new Date(item.ce_createdAt).getFullYear().toString();
      const matchesYear = selectedYear ? itemDate === selectedYear.value : true;

      return (
        matchesYear
      );
    });
    setFilteredCopyeditingData(filtered);
  }, [selectedYear, copyeditingData]);

  const columns = useMemo(() => getBookColumns(), []);
  const copyeditingColumns = useMemo(() => getCopyeditingColumns(), []);
console.log('copyeditingColumns in BookList:', copyeditingColumns);
  // const selectStyles = {
  //   control: (base) => ({ ...base, minWidth: 220 }),
  //   menuPortal: (base) => ({ ...base, zIndex: 9999 }),
  // };
  return (
    <div className="relative max-w-full px-2 py-2 mx-auto space-y-8 h-full">
      {/* New Header Section */}
      <header className="relative z-10 bg-white rounded-2xl shadow-lg p-4 overflow-hidden">
        <div className="flex flex-col md:flex-row justify-between items-start md:items-center">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">Dashboard</h2>
            <p className="text-gray-500 mt-1 text-sm">
              Welcome back! Here's your book data overview.
            </p>
          </div>
          <div className="flex items-center space-x-4 mt-4 md:mt-0">
            <div className="flex -space-x-2">
              <div className="w-10 h-10 rounded-full bg-gradient-to-br from-purple-500 to-indigo-600 flex items-center justify-center text-white shadow-md border-2 border-white">
                <FiBookOpen size={20} />
              </div>
              <div className="w-10 h-10 rounded-full bg-gradient-to-br from-green-500 to-teal-600 flex items-center justify-center text-white shadow-md border-2 border-white">
                <FiCheckCircle size={20} />
              </div>
              <div className="w-10 h-10 rounded-full bg-gradient-to-br from-pink-500 to-rose-600 flex items-center justify-center text-white shadow-md border-2 border-white">
                <FiUsers size={20} />
              </div>
            </div>
          </div>
        </div>
        <div className="absolute top-0 right-0 h-24 w-64 bg-gradient-to-l from-purple-500/20 to-transparent -z-10 blur-2xl"></div>
        <div className="absolute bottom-0 left-0 h-24 w-64 bg-gradient-to-r from-pink-500/20 to-transparent -z-10 blur-2xl"></div>
      </header>

      {/* Tabs UI */}
      <div className="bg-white/30 backdrop-blur-sm rounded-xl shadow-lg border border-gray-200/70 overflow-hidden">
        <div className="relative border-b border-gray-200/80 px-6">

          <div className="absolute inset-y-0 left-0 w-1/4 bg-gradient-to-r from-sky-400/20 to-transparent -z-10 blur-2xl"></div>
          <div className="absolute inset-y-0 right-0 w-1/4 bg-gradient-to-l from-sky-400/20 to-transparent -z-10 blur-2xl"></div>


          <nav className="relative flex space-x-1" aria-label="Tabs">
            <button
              onClick={() => setActiveTab('yearWise')}
              className={`relative whitespace-nowrap py-4 px-4 font-semibold text-sm transition-colors focus:outline-none flex items-center gap-2 ${activeTab === 'yearWise' ? 'text-[#8C3FE1]' : 'text-gray-500 hover:text-gray-700 cursor-pointer'
                }`}
            >
              <FiCalendar />
              <span className={activeTab === 'yearWise' ? 'bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] text-transparent bg-clip-text' : ''}>Year Wise</span>
              {activeTab === 'yearWise' && (
                <motion.div layoutId="active-tab-underline" className="absolute bottom-0 left-0 right-0 h-0.5 bg-gradient-to-r from-[#8C3FE1] to-[#E557AE]" />
              )}
            </button>
            {role !== 3 && (
              <button
                onClick={() => setActiveTab('copyediting')}
                className={`relative whitespace-nowrap py-4 px-4 font-semibold text-sm transition-colors focus:outline-none flex items-center gap-2 ${activeTab === 'copyediting' ? 'text-[#8C3FE1]' : 'text-gray-500 hover:text-gray-700 cursor-pointer'
                  }`}
              >
                <FiEdit />
                <span className={activeTab === 'copyediting' ? 'bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] text-transparent bg-clip-text' : ''}>Copyediting</span>
                {activeTab === 'copyediting' && (
                  <motion.div layoutId="active-tab-underline" className="absolute bottom-0 left-0 right-0 h-0.5 bg-gradient-to-r from-[#8C3FE1] to-[#E557AE]" />
                )}
              </button>
            )}
          </nav>
        </div>

        {activeTab === 'yearWise' && (

          <section>
            <div className="px-10 py-2 flex flex-col sm:flex-row justify-between items-center gap-3">
              <h2 className="text-xl font-semibold text-gray-800">&nbsp;</h2>
              <div className="flex items-center space-x-3">
                <Select
                  instanceId="year-wise-year-filter"
                  id="year-filter"
                  options={yearOptions}
                  value={selectedYear}
                  onChange={setSelectedYear}
                  placeholder="Select Year"
                  isClearable
                  className="w-48 text-sm"
                  menuPortalTarget={document.body}
                  styles={selectStyles}
                />
              </div>
            </div>
            <CustomDataTable
              columns={columns}
              data={filteredData}
              defaultSort={{ key: 'Working Title', direction: 'ascending' }}
            />
          </section>
        )}

        {activeTab === 'copyediting' && (
          <section>
            <div className="px-10 py-2 flex flex-col sm:flex-row justify-between items-center gap-3">
              <h2 className="text-xl font-semibold text-gray-800">&nbsp;</h2>
              <div className="flex items-center space-x-3">
                <Select
                  instanceId="copyediting-year-filter"
                  id="year-filter-copyediting"
                  options={yearOptions}
                  value={selectedYear}
                  onChange={setSelectedYear}
                  placeholder="Select Year"
                  isClearable
                  className="w-48 text-sm"
                  menuPortalTarget={document.body}
                  styles={selectStyles}
                />
              </div>
            </div>
            <CustomDataTable
              columns={copyeditingColumns}
              data={filteredCopyeditingData}
              defaultSort={{ key: 'Working Title', direction: 'ascending' }}
            />
          </section>
        )}
      </div>
    </div>
  );
}