"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import Swal from "sweetalert2";
import { toast } from "react-toastify";
import CustomDataTable from "../../dataTable/customDataTable";
import { getChapterColumns } from "./columns";
import { updateChapterAction } from "@/lib/actions/chapterActions";
import Select from "react-select";
import selectStyles from "@/app/components/selectStyles";

export default function ChaptersList({ initialChapters, initialChapterEntries, initialChapterId, role, booklist,bookyear, manuscriptid }) {
  const [data, setData] = useState(initialChapters || []);
  const [filteredData, setFilteredData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [bookOptions, setBookOptions] = useState([]);
  const [selectedBook, setSelectedBook] = useState(null);
  const [selectedCopyType, setSelectedCopyType] = useState(null);
  const [selectedManuScriptId, setSelectedManuScriptId] = useState(null);
  const [filteredManuscripts, setFilteredManuscripts] = useState([]);
  const [yearOptions, setYearOptions] = useState([]);
  const [selectedYear, setSelectedYear] = useState(null);
  const router = useRouter();

  //loading use effect
  useEffect(() => {
    setData(initialChapters || []);
    setFilteredData(initialChapters || []);
    setLoading(false);
  }, [initialChapters]);

  const copyEditng = [
    { value: 'yes', label: 'yes' },
    { value: 'no', label: 'no' },
  ];
  
  //year Use effect
  useEffect(() => {
    if (bookyear && bookyear.length > 0) {
      const formattedYears = bookyear.map((y) => ({
        value: y.book_year.toString(),
        label: y.book_year.toString(),
      }));
      setYearOptions(formattedYears);
    }
  }, [bookyear]);

  // Fetch chapters when book is selected
  const book_Year = selectedYear?.value;
  useEffect(() => {
    if (!book_Year) {
      setBookOptions([]);
      setSelectedBook(null);
      return;
    }
    const fetchBooks = async () => {
      try {
        const filteredBooks = booklist.filter(
            (b) => String(b.book_year) === String(book_Year)
        );
        const options = filteredBooks.map((c) => ({
          value: String(c.book_pid),
          label: c.book_title.charAt(0).toUpperCase() + c.book_title.slice(1),
        }));
       
        //filter the chapter count
        const filterBookData = data.filter(
            (d) => String(d.book?.book_year) === String(book_Year)
        );

        if (filterBookData.length > 0) {
          setBookOptions(options);
        } else {
          setBookOptions([]);
        }
        setSelectedBook(null); 
        console.log("Book options:", options);
      } catch (error) {
        toast.error("Failed to load books");
        console.error(error);
      }
    };
    fetchBooks();
  }, [book_Year, booklist]);

  //manuscript id filter
  useEffect(() => {
    if (!selectedBook) {
      setSelectedManuScriptId(null);
      setFilteredManuscripts([]);
      return;
    }
    try {
      const filtered = manuscriptid.filter(
        (m) => String(m.chap_bookTitle) === String(selectedBook.value) ||
              String(m.book_pid) === String(selectedBook.value) 
      );

      const manuscriptOptions = filtered.map((m) => ({
        value: String(m.chap_manuscriptID),
        label: m.chap_manuscriptID,
      }));

      setFilteredManuscripts(manuscriptOptions);
      setSelectedManuScriptId(null);
    } catch (error) {
      toast.error("Failed to load manuscript IDs");
      console.error(error);
    }
  }, [selectedBook, manuscriptid]); 

  useEffect(() => {
    const filtered = data.filter((item) => {
      const matchesYear = selectedYear ? Number(item.book?.book_year) === Number(selectedYear.value) : true;
      const matchesBook = selectedBook
        ? Number(item.chap_bookTitle) === Number(selectedBook.value)
        : true;
      const matchesManuScript = selectedManuScriptId ? String(item.chap_manuscriptID) === selectedManuScriptId.value : true; 

      const matchesType = selectedCopyType ? String(item.chap_copyEditing) === selectedCopyType.value : true;

      return (
        matchesYear &&
        matchesBook &&
        matchesManuScript && 
        matchesType 
      ); 
    });
    setFilteredData(filtered);
  }, [selectedYear,selectedBook, selectedManuScriptId, selectedCopyType, data]);
  
  const handleDelete = async (chapterId, chapterName) => {
    const result = await Swal.fire({
      title: "Are you sure?",
      text: `Do you really want to delete the chapter named "${chapterName}"?`,
      icon: "warning",
      showCancelButton: true,
      confirmButtonColor: "#d33",
      cancelButtonColor: "#3085d6",
      confirmButtonText: "Yes, delete it!",
    });

    if (!result.isConfirmed) return;

    try {
      const payload = {
        chap_isdelete: "inactive",
      };

      const res = await updateChapterAction(chapterId, payload);
      if (res?.success) {
        // ✅ Filter out deleted chapter from local state
         setData((prevData) =>
        prevData.filter((data) => data.chap_pid !== chapterId)
      );
       toast.success("Chapter has been deleted.");
      } else {
        toast.error(res?.message || "Failed to delete chapter.");
      }

    } catch (error) {
      toast.error(error.message || "Error deleting chapter");
    }
  };

  const combinedChapter = initialChapterId.map((id, index) => ({
    id,
    value: initialChapterEntries[index],
  }));

  const columns = useMemo(() => getChapterColumns(router, handleDelete, combinedChapter, role), [router]);

  // 🔥 Common react-select styles (so dropdown always above table)



  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg  space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">
              Chapters Lists
            </h2>
            <p className="text-gray-500 mt-1 text-sm">
              Below is a list of all chapters.
            </p>
          </div>
          <button
            onClick={() => router.push("/chapters/create")}
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-2 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105"
          >
            + New Chapter
          </button>
        </div>
      </header>

      <div className="bg-purple-50 border border-purple-200 rounded-lg p-2 shadow-sm">
        <div className="flex flex-wrap gap-4 items-center justify-start">
          
          {/* Year Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Year
            </label>
            <Select
              instanceId="year-filter"
              options={yearOptions}
              value={selectedYear}
              onChange={setSelectedYear}
              placeholder="Select Year"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>
          
          
          {/* Role Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Book Title
            </label>
            <Select
              instanceId="book-filter"
              options={bookOptions}
              value={selectedBook}
              onChange={setSelectedBook}
              placeholder="Select Book"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>
            {/* manuscript Id Filter */}
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Manuscript Id
            </label>
            <Select
              instanceId="manuscript-filter"
              options={filteredManuscripts}
              value={selectedManuScriptId}
              onChange={setSelectedManuScriptId}
              placeholder="Select Manuscript Id"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>
          <div className="flex flex-col">
            <label className="text-xs text-gray-600 font-semibold mb-1">
              Copy Editing
            </label>
            <Select
              instanceId="copytype-filter"
              options={copyEditng}
              value={selectedCopyType}
              onChange={setSelectedCopyType}
              placeholder="Select Type"
              isClearable
              className="w-48 text-sm"
              menuPortalTarget={document.body}
              styles={selectStyles}
            />
          </div>
          {/* Clear Filters Button */}
          <div className="pt-5">
            <button
              onClick={() => {
                setSelectedYear(null);
                setSelectedBook(null);
                setSelectedManuScriptId(null);
                setSelectedCopyType(null);
                setFilteredData(initialChapters);
              }}
              className="text-sm px-4 py-2 bg-purple-100 hover:bg-purple-200 text-purple-700 rounded transition-all duration-200 cursor-pointer"
            >
              Clear Filters
            </button>
          </div>
        </div>
      </div>

      <CustomDataTable
        columns={columns}
        data={filteredData}
        progressPending={loading}
        defaultSort={{ key: "Book Title", direction: "ascending" }}
      />
    </div>
  );
}
