"use client";

import React, { useState, useEffect } from "react";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { chapterSchema } from "@/validation/validation";
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import Select from "react-select";
import { createChapterAction, updateChapterAction } from "@/lib/actions/chapterActions";
import BackButton from "@/app/components/BackButton";
export default function CreateChapterForm({ initialBooks, initialPStatus, chapter, chapterId, user_pid }) {
  const [loading, setLoading] = useState(false);
  const router = useRouter();
  const isEditMode = !!chapterId;

  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
    setValue,
  } = useForm({
    resolver: zodResolver(chapterSchema),
    mode: "onBlur",
    defaultValues: {
      chap_book_title: "",
      chap_manuscript_id: "",
      chap_title: "",
      chap_wordcount: "",
      chap_first_name: "",
      chap_last_name: "",
      chap_corres_au_email: "",
      chap_status: "",
      chap_p_status: "",
      chap_pdue_date: "",
      chap_copy_editing: "",
      chap_comments: "",
      
    },
  });
  const formatDate = (dateString) => {
    if (!dateString) return "";
    return new Date(dateString).toISOString().split("T")[0];
  };
  // If editing, prefill form
  useEffect(() => {
    if (chapter) {
      setValue("chap_book_title", String(chapter.chap_bookTitle));
      setValue("chap_manuscript_id", chapter.chap_manuscriptID);
      setValue("chap_title", chapter.chap_title);
      setValue("chap_wordcount", chapter.chap_wordcount);
      setValue("chap_first_name", chapter.chap_firstName);
      setValue("chap_last_name", chapter.chap_lastName);
      setValue("chap_corres_au_email", chapter.chap_corresAuEmail);
      setValue("chap_status", chapter.chap_status);
      setValue("chap_p_status", String(chapter.chap_pStatus));
      setValue("chap_pdue_date", formatDate(chapter.chap_pdueDate));
      setValue("chap_copy_editing", chapter.chap_copyEditing);
      setValue("chap_comments", chapter.chap_comments);
      
    } else {
      reset();
    }
  }, [chapter, setValue, reset]);

  const onSubmit = async (data) => {
    setLoading(true);
    try {
      const payload = {
        chap_bookTitle: parseInt(data.chap_book_title),
        chap_manuscriptID: data.chap_manuscript_id,
        chap_title: data.chap_title,
        chap_wordcount: Number(data.chap_wordcount) || null,
        chap_pre_wordcount: Number(data.chap_wordcount) || null,
        chap_firstName: data.chap_first_name,
        chap_lastName: data.chap_last_name,
        chap_corresAuEmail: data.chap_corres_au_email,
        chap_status: data.chap_status,
        chap_pStatus: Number(data.chap_p_status) || null,
        chap_pdueDate: data.chap_pdue_date,
        chap_copyEditing: data.chap_copy_editing,
        chap_comments: data.chap_comments,
        // chap_createdBy: user_pid,
        
      };
      if (isEditMode) {
        payload.chap_updatedBy = user_pid;
      } else {
        payload.chap_createdBy = user_pid;
      }
      if (isEditMode) {
        console.log(data, "data in update");
        const res = await updateChapterAction(chapterId, payload);
        if (res?.success) {
          router.push("/chapters/list");
          toast.success("Chapter updated successfully!");
        } else {
          toast.error(res?.message || "Failed to update book.");
        }

      } else {
        const res = await createChapterAction(payload);
        if (res?.success) {
          router.push("/chapters/list");
          toast.success("Chapter created successfully!");
        } else {
          toast.error(res?.message || "Failed to update book.");
        }

      }

    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const handleReset = () => {
    reset();
    toast.info("Form reset.");
  };

  const bookOptions = initialBooks.map((b) => ({
    value: String(b.book_pid),
    label: b.book_title.charAt(0).toUpperCase() + b.book_title.slice(1),
  }));

  const pstatusOptions = initialPStatus.map((p) => ({
    value: String(p.pstatus_pid),
    label: p.pstatus_name.charAt(0).toUpperCase() + p.pstatus_name.slice(1),
  }));

  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg  space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">
              {isEditMode ? "Edit Chapter" : "Create New Chapter"}
            </h2>
            <p className="text-gray-500 mt-1 text-sm">
              {isEditMode
                ? "Modify the details of the selected chapter."
                : "Fill in the details below to add a new chapter."}
            </p>
          </div>
          <BackButton />
        </div>
      </header>

      <form
        onSubmit={handleSubmit(onSubmit)}
        className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl mx-auto"
      >
        {/* Left Column */}
        <div className="space-y-4">
          {/* Book Title */}
          <div>
            <label
              htmlFor="chap_book_title"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Book Title <span className="text-red-500">*</span>
            </label>
            <Controller
              name="chap_book_title"
              control={control}
              render={({ field }) => (
                <Select
                  {...field}
                  instanceId="user-role-select"
                  options={bookOptions}
                  onChange={(selected) =>
                    field.onChange(selected ? selected.value : "")
                  }
                  value={
                    bookOptions.find(
                      (option) => option.value === field.value
                    ) || null
                  }
                  placeholder="Select a book title"
                  isClearable
                  inputId="chap_book_title"
                  styles={{
                    control: (base) => ({
                      ...base,
                      borderColor: errors.chap_book_title
                        ? "red"
                        : base.borderColor,
                    }),
                  }}
                />
              )}
            />
            {errors.chap_book_title && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_book_title.message}
              </p>
            )}
          </div>

          {/* Manuscript ID */}
          <div>
            <label
              htmlFor="chap_manuscript_id"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Manuscript ID <span className="text-red-500">*</span>
            </label>
            <input
              id="chap_manuscript_id"
              type="text"
              placeholder="Enter manuscript id"
              {...register("chap_manuscript_id")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_manuscript_id
                ? "border-red-500"
                : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.chap_manuscript_id && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_manuscript_id.message}
              </p>
            )}
          </div>

          {/* Chapter Title */}
          <div>
            <label
              htmlFor="chap_title"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Chapter Title <span className="text-red-500">*</span>
            </label>
            <input
              id="chap_title"
              type="text"
              placeholder="Enter chapter title"
              {...register("chap_title")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_title ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.chap_title && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_title.message}
              </p>
            )}
          </div>

          {/* Word Count */}
          <div>
            <label
              htmlFor="chap_wordcount"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Word Count <span className="text-red-500"></span>
            </label>
            <input
              id="chap_wordcount"
              type="number"
              placeholder="Enter word count"
              {...register("chap_wordcount")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_wordcount ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />

            {errors.chap_wordcount && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_wordcount.message}
              </p>
            )}
          </div>
          {/* First Name */}
          <div>
            <label
              htmlFor="chap_first_name"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              First Name <span className="text-red-500">*</span>
            </label>
            <input
              id="chap_first_name"
              type="text"
              placeholder="Enter First name"
              {...register("chap_first_name")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_first_name ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.chap_first_name && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_first_name.message}
              </p>
            )}
          </div>

          {/* Last Name */}
          <div>
            <label
              htmlFor="chap_last_name"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Last Name <span className="text-red-500"></span>
            </label>
            <input
              id="chap_last_name"
              type="text"
              placeholder="Enter last name"
              {...register("chap_last_name")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_last_name ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.chap_last_name && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_last_name.message}
              </p>
            )}
          </div>
        </div>

        {/* Right Column */}
        <div className="space-y-4">
          {/* Corresponding author email */}
          <div>
            <label
              htmlFor="chap_corres_au_email"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Corresponding Author Email{" "}
              <span className="text-red-500"></span>
            </label>
            <input
              id="chap_corres_au_email"
              type="email"
              placeholder="Enter corresponding author email"
              {...register("chap_corres_au_email")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_corres_au_email
                ? "border-red-500"
                : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.chap_corres_au_email && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_corres_au_email.message}
              </p>
            )}
          </div>
          {/* Chapter Status */}
          <div>
            <label
              htmlFor="chap_status"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Chapter Status <span className="text-red-500"></span>
            </label>
            <input
              id="chap_status"
              type="text"
              placeholder="Enter chapter status"
              {...register("chap_status")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_status ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.chap_status && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_status.message}
              </p>
            )}
          </div>

          {/* P+ Status */}
          {/* <div>
              <label
                htmlFor="chap_p_status"
                className="block text-sm font-medium text-gray-700 mb-1"
              >
                P+ Status <span className="text-red-500">*</span>
              </label>
              <input
                id="chap_p_status"
                type="text"
                placeholder="Enter p+ status"
                {...register("chap_p_status")}
                className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${
                  errors.chap_p_status ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
              />
              {errors.chap_p_status && (
                <p className="text-sm text-red-500 mt-1">
                  {errors.chap_p_status.message}
                </p>
              )}
            </div> */}
          {/* Book Title */}
          <div>
            <label
              htmlFor="chap_p_status"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              P+ Status <span className="text-red-500"></span>
            </label>
            <Controller
              name="chap_p_status"
              control={control}
              render={({ field }) => (
                <Select
                  {...field}
                  instanceId="user-role-select"
                  options={pstatusOptions}
                  onChange={(selected) =>
                    field.onChange(selected ? selected.value : "")
                  }
                  value={
                    pstatusOptions.find(
                      (option) => option.value === field.value
                    ) || null
                  }
                  placeholder="Select a pstatus"
                  isClearable
                  inputId="chap_p_status"
                  styles={{
                    control: (base) => ({
                      ...base,
                      borderColor: errors.chap_p_status
                        ? "red"
                        : base.borderColor,
                    }),
                  }}
                />
              )}
            />
            {errors.chap_p_status && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_p_status.message}
              </p>
            )}
          </div>

          {/* P+ Due Date */}
          <div>
            <label
              htmlFor="chap_pdue_date"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              P+ Due Date <span className="text-red-500"></span>
            </label>
            <input
              id="chap_pdue_date"
              type="date"
              placeholder="Enter p+ due date"
              {...register("chap_pdue_date")}
              min={new Date().toISOString().split("T")[0]}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_pdue_date ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.chap_pdue_date && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_pdue_date.message}
              </p>
            )}
          </div>

          {/* Copy-Editing */}
          {/* <div>
			   <label htmlFor="chap_copy_editing" className="block text-sm font-medium text-gray-700 mb-1">
                Copy-Editing <span className="text-red-500">*</span>
              </label>
			  <div className="flex items-center space-x-6">
				<label className="flex items-center">
				  <input
					type="radio"
					value="Yes"
					{...register('chap_copy_editing')}
					className="mr-2"
				  />
				  <span>Yes</span>
				</label>
				<label className="flex items-center">
				  <input
					type="radio"
					value="No"
					{...register('chap_copy_editing')}
					className="mr-2"
				  />
				  <span>No</span>
				</label>
			  </div>
			  {errors.chap_copy_editing && (
				<p className="text-sm text-red-500 mt-1">{errors.chap_copy_editing.message}</p>
			  )}
			</div> */}

          {/* Comments */}
          <div>
            <label
              htmlFor="chap_comments"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Comments <span className="text-red-500"></span>
            </label>
            <textarea
              id="chap_comments"
              placeholder="Enter comments"
              rows={4} // you can change the number of rows as needed
              {...register("chap_comments")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.chap_comments ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.chap_comments && (
              <p className="text-sm text-red-500 mt-1">
                {errors.chap_comments.message}
              </p>
            )}
          </div>
        </div>

        {/* Buttons */}
        <div className="col-span-1 md:col-span-2 flex flex-col sm:flex-row justify-end gap-4">
          {!isEditMode && (
            <button
              type="button"
              onClick={handleReset}
              className="w-full sm:w-auto px-5 py-2 border border-red-600 text-red-600 rounded-md hover:bg-red-100 transition cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
            >
              Reset
            </button>
          )}
          <button
            type="submit"
            className="w-full sm:w-auto px-5 py-2  text-white rounded-md bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2 d-flex justify-center align-center"
          >
            <span>
              {loading
                ? isEditMode
                  ? "Updating..."
                  : "Creating..."
                : isEditMode
                  ? "Update Chapter"
                  : "Create Chapter"}
            </span>
            {loading ? (
              <FiLoader size={20} className="animate-spin" />
            ) : (
              <FiArrowRightCircle size={20} />
            )}
          </button>
        </div>
      </form>
    </div>
  );
}
