"use client";

import { useEffect, useMemo, useState } from "react";
import Select from "react-select";
import { createBookReportAction,allBookReportList } from "@/lib/actions/entryActions"; // your API/service function
import * as XLSX from "xlsx";
import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";
import Papa from "papaparse";

export default function BooksReport({ initialReport = [], bookList = [], role }) {
  const [filteredGrouped, setFilteredGrouped] = useState([]);
  const [loading, setLoading] = useState(false);

  // filters
  const [startDate, setStartDate] = useState("");
  const [endDate, setEndDate] = useState("");
  const [selectedBook, setSelectedBook] = useState(null);
  const [selectcsvreport, setFilterCsvGrouped] = useState(null);
  const [reportTitle, setReportTitle] = useState("Book Wise Reports");

  const selectStyles = { menuPortal: (base) => ({ ...base, zIndex: 9999 }) };

  // react-select book options
  const bookOptions = useMemo(
    () =>
      (bookList || []).map((b) => ({
        value: String(b.book_pid),
        label: b.book_title?.charAt(0).toUpperCase() + b.book_title?.slice(1),
      })),
    [bookList]
  );

  const formatTimeDisplay = (t) => {
    if (!t && t !== 0) return "-";
    if (typeof t === "string") return t.split(".")[0];
    if (t instanceof Date) return t.toTimeString().split(" ")[0];
    return String(t);
  };

  // handle filter changes
  const handleChange = async (field, value) => {
    setLoading(true);

    // update local state
    if (field === "startDate") setStartDate(value);
    if (field === "endDate") setEndDate(value);
    if (field === "book") setSelectedBook(value);

    console.log(selectedBook?.value,"selectedBook?.value");
    

    // prepare payload for API call
    const payload = {
      book_pid: field === "book" ? value?.value : selectedBook?.value || null,
      start_date: field === "startDate" ? value : startDate || null,
      end_date: field === "endDate" ? value : endDate || null,
    };
      
    try {
      const resq = await createBookReportAction(payload); // call service/API
      const res  = resq?.bookreport;

      if (res && Array.isArray(res)) {
        // group data by book_title
        const groups = Object.values(
          res.reduce((acc, row) => {
            const book = row.book_title ?? "Unknown Book";
            if (!acc[book]) acc[book] = { book_title: book, tasks: [] };
            acc[book].tasks.push({
              task_name: row.task_name ?? "-",
              avg_time: row.avg_time_per_task ?? row.avgTime ?? "-",
              count: Number(row.task_count ?? 0),
            });
            return acc;
          }, {})
        );
        groups.forEach((g) => {
          g.total = g.tasks.reduce((s, t) => s + (Number(t.count) || 0), 0);
        });

        const reportlist = await allBookReportList(payload);
        console.log(reportlist?.bookreportlist,"reportlist?.bookreportlist");
        
        const csvReport = reportlist?.bookreportlist?.reduce((acc, item) => {
          let group = acc.find(g => g.book_title === item.book_title);
          if (!group) {
            group = { book_title: item.book_title, tasks: [] };
            acc.push(group);
          }
          group.tasks.push({
            task_name: item.task_name,
            entry_duration: item.entry_duration,
            count: item.task_count,
          });
          return acc;
        }, []);

        setFilteredGrouped(groups);
        setFilterCsvGrouped(csvReport);

        // ✅ Dynamically update title

        // Helper to format date as MM-DD-YYYY
        const formatDate = (dateString) => {
          const date = new Date(dateString);
          if (isNaN(date)) return "-"; // handle invalid date
          const mm = String(date.getMonth() + 1).padStart(2, "0");
          const dd = String(date.getDate()).padStart(2, "0");
          const yyyy = date.getFullYear();
          return `${dd}-${mm}-${yyyy}`;
        };
        
        let title = "Book Wise Reports";
         // If user selected a date range
        if (payload.start_date && payload.end_date) {
          title = `Book Wise Reports - ${formatDate(payload.start_date)} to ${formatDate(payload.end_date)}`;
        }
        // If user selected a book
        else  if (payload.book_pid && (field === "book" ? value?.label : selectedBook?.label)) {
          const bookLabel = field === "book" ? value?.label : selectedBook?.label;
          title = `Book Wise Report - ${bookLabel}`;
        }
       
        
        setReportTitle(title);
        
      } else {
        setFilteredGrouped([]);
        setFilterCsvGrouped()
      }
    } catch (err) {
      console.error("Error fetching book report:", err);
      setFilteredGrouped([]);
    } finally {
      setLoading(false);
    }
  };

  const grandTotal = filteredGrouped.reduce((s, g) => s + (g.total || 0), 0);

   // --- Export Handlers ---

  // Excel Export
  const handleExportExcel = () => {
    const ws_data = [
      ["Book Title", "Task", "Avg. Time", "Total"],
    ];
    filteredGrouped.forEach(group => {
      group.tasks.forEach((task, idx) => {
        ws_data.push([
          idx === 0 ? group.book_title : "",
          task.task_name,
          formatTimeDisplay(task.avg_time),
          task.count,
        ]);
      });
      ws_data.push([
        "", "Total", "", group.total
      ]);
    });
    ws_data.push(["Grand Total", "", "", grandTotal]);
    const ws = XLSX.utils.aoa_to_sheet(ws_data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, "Report");
    XLSX.writeFile(wb, "book_report.xlsx");
  };

// PDF Export
const handleExportPDF = async () => {
  // 🧠 Ensure we are in the browser
  if (typeof window === "undefined") return;

  const doc = new jsPDF();

  // Title
  const pageWidth = doc.internal.pageSize.getWidth();
  const imageurl = "/logo.png"; // ✅ correct path (not /public/logo.png)
  doc.addImage(imageurl, "PNG", 15, 5, 50, 15);
 
  doc.setFontSize(13);
  doc.text("Book Report", pageWidth / 2, 15, { align: "center" });

  const body = [];

  // 🔹 Loop through each book group
  filteredGrouped.forEach(group => {
    group.tasks.forEach((task, idx) => {
      body.push([
        idx === 0 ? group.book_title : "",
        task.task_name,
        formatTimeDisplay(task.avg_time),
        task.count.toString(),
      ]);
    });

    // ✅ Add per-book total row (styled)
    body.push([
      "",
      { content: "Total", styles: { fontStyle: "bold" } },
      "",
      { content: group.total.toString(), styles: { fontStyle: "bold" } },
    ]);
    // ✅ Add an empty row for spacing
    //body.push(["", "", "", ""]);
  });

  // ✅ Add grand total row
  body.push([
    { content: "Grand Total", styles: { fontStyle: "bold" } },
    "",
    "",
    { content: grandTotal.toString(), styles: { fontStyle: "bold" } },
  ]);

  // 🧾 Generate table
  autoTable(doc,{
    head: [["Book Title", "Task", "Avg. Time", "Total"]],
    body,
    startY: 20,
    theme: "grid",
    styles: {
      fontSize: 8,
      cellPadding: 1,
      halign: "center",
    },
    headStyles: {
      fillColor: [200, 200, 200],
      textColor: [0, 0, 0],
      fontStyle: "bold",
    },
  });

  // Save file
  const date = new Date().toISOString().split("T")[0];
  doc.save(`book_report_${date}.pdf`);
};

  // CSV Export
  const handleExportCSV = () => {
    const data = [];
    selectcsvreport.forEach(group => {
      group.tasks.forEach((task, idx) => {
        data.push({
          "Book Title": group.book_title,
          Task: task.task_name,
          "Avg. Time": formatTimeDisplay(task.entry_duration),
          Count: task.count,
        });
      });
    });

    const csv = Papa.unparse(data);
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.setAttribute("download", "book_report.csv");
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg space-y-4">
      {/* Filters */}
      <div className="bg-purple-50 border border-purple-200 rounded-lg p-4 shadow-sm flex flex-wrap gap-4 items-center justify-start">
        <div className="flex flex-col">
          <label className="text-xs text-gray-600 font-semibold mb-1">Start Date</label>
          <input
            type="date"
            className="w-40 border border-gray-300 rounded px-3 py-2 text-sm focus:ring focus:ring-purple-200"
            value={startDate}
            onChange={(e) => handleChange("startDate", e.target.value)}
            max={endDate}
          />
        </div>

        <div className="flex flex-col">
          <label className="text-xs text-gray-600 font-semibold mb-1">End Date</label>
          <input
            type="date"
            className="w-40 border border-gray-300 rounded px-3 py-2 text-sm focus:ring focus:ring-purple-200"
            value={endDate}
            onChange={(e) => handleChange("endDate", e.target.value)}
            min={startDate}
          />
        </div>

        <div className="flex flex-col">
          <label className="text-xs text-gray-600 font-semibold mb-1">Book</label>
          <Select
            instanceId="book-filter"
            options={bookOptions}
            value={selectedBook}
            onChange={(v) => handleChange("book", v)}
            placeholder="Select Book"
            isClearable
            className="w-48 text-sm"
            menuPortalTarget={typeof document !== "undefined" ? document.body : undefined}
            styles={selectStyles}
          />
        </div>

        <div className="pt-5">
          <button
            onClick={() => {
              setStartDate("");
              setEndDate("");
              setSelectedBook(null);
              setFilteredGrouped([]);
            }}
            className="text-sm px-4 py-2 bg-purple-100 hover:bg-purple-200 text-purple-700 rounded transition-all duration-200 cursor-pointer"
          >
            Clear Filters
          </button>
        </div>
      </div>
      <header>
        <h2 className="text-2xl font-semibold text-gray-800 text-center">{reportTitle}</h2>
      </header>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="min-w-full bg-white border border-gray-200 rounded-lg shadow-sm">
          <thead>
            <tr className="bg-gray-100 text-sm">
              <th className="py-2 px-4 border-b text-left w-1/3">Book Title</th>
              <th className="py-2 px-4 border-b text-left w-1/3">Task</th>
              <th className="py-2 px-4 border-b text-center w-1/6">Avg. Time</th>
              <th className="py-2 px-4 border-b text-center w-1/6">Total</th>
            </tr>
          </thead>
          <tbody>
            {loading ? (
              <tr>
                <td colSpan={4} className="text-center p-6">
                  Loading...
                </td>
              </tr>
            ) : filteredGrouped.length === 0 ? (
              <tr>
                <td colSpan={4} className="text-center p-6 text-gray-500">
                  No data found
                </td>
              </tr>
            ) : (
              filteredGrouped.map((group, gi) => {
                const rowSpan = group.tasks.length + 1;
                return group.tasks.map((task, ti) => (
                  <tr key={`${gi}-${ti}`} className={ti % 2 === 0 ? "bg-white" : "bg-gray-50"}>
                    {ti === 0 && (
                      <td
                        rowSpan={rowSpan}
                        className="py-2 px-4 border-l border-r border-b align-top font-medium text-gray-700 w-1/3"
                      >
                        {group.book_title}
                      </td>
                    )}
                    <td className="py-2 px-4 border-b border-r w-1/3">{task.task_name}</td>
                    <td className="py-2 px-4 border-b border-r text-center font-mono w-1/6">
                      {formatTimeDisplay(task.avg_time)}
                    </td>
                    <td className="py-2 px-4 border-b border-r text-center font-semibold w-1/6">{task.count}</td>
                  </tr>
                )).concat(
                  <tr key={`${gi}-total`} className="bg-gray-200/50">
                    <td className="py-1 px-4 border-b border-r text-left font-medium">Total</td>
                    <td className="py-1 px-4 border-b border-r text-center font-mono"></td>
                    <td className="py-1 px-4 border-b border-r text-center font-semibold">{group.total}</td>
                  </tr>
                );
              })
            )}
          </tbody>

          {filteredGrouped.length > 0 && (
            <tbody>
              <tr className="bg-gray-200/50">
                <td className="py-1 px-4 border-l border-b border-r text-left font-medium">Grand Total</td>
                <td className="py-1 px-4 border-b border-r text-center font-mono"></td>
                <td className="py-1 px-4 border-b border-r text-center font-mono"></td>
                <td className="py-1 px-4 border-b border-r text-center font-semibold">{grandTotal}</td>
              </tr>
            </tbody>
          )}
        </table>
      </div>
      {/* Export Buttons at Bottom */}
      {filteredGrouped.length > 0 && (
        <div className="flex gap-2 mt-6 justify-end">
          <button
            onClick={handleExportExcel}
            /*className="bg-green-100 hover:bg-green-200 text-green-700 px-4 py-2 rounded"  */className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
          >
            Excel
          </button>
          <button
            onClick={handleExportPDF}
            /* className="bg-blue-100 hover:bg-blue-200 text-blue-700 px-4 py-2 rounded" */
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
          >
            Export PDF
          </button>
          <button
            onClick={handleExportCSV}
            /* className="bg-yellow-100 hover:bg-yellow-200 text-yellow-700 px-4 py-2 rounded" */
            className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
          >
            Download CSV
          </button>
        </div>
      )}  
    </div>
  );
}
