'use client';
import Head from "next/head";
import { motion } from "framer-motion";
import { useForm } from 'react-hook-form'
import { zodResolver } from '@hookform/resolvers/zod'
import { loginSchema } from '@/validation/validation'
import { useRouter } from 'next/navigation'
import { toast } from 'react-toastify';
import { FiArrowRightCircle, FiLoader } from 'react-icons/fi';
import { useState } from 'react';
import { signIn } from "next-auth/react";

export default function Login() {
  const [loading, setLoading] = useState(false);
  const router = useRouter()
  const {
    register,
    handleSubmit,
    formState: { errors }
  } = useForm({ resolver: zodResolver(loginSchema) })

const onSubmit = async (data) => {
    setLoading(true);
    try {
      const res = await signIn("credentials", {
        redirect: false,
        email: data.email,
        password: data.password,
      });

      if (res.ok) {
        // toast.success('Login successful!');
          
          // window.location.href = "/dashboard";
          router.replace('/dashboard')
          toast.success("Login successful!");
      } else {
        toast.error(res.error || 'Login failed');
      }
    } catch (error) {
      console.error('Error submitting form:', error);
      toast.error('Something went wrong. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      <Head>
        <title>Login</title>
      </Head>

      <div className="min-h-screen flex flex-col md:flex-row bg-gray-100 overflow-hidden">

        {/* Left Side - SVG Image with animation */}
        <div className="relative w-full md:w-1/2 h-64 md:h-auto md:min-h-screen">
          <motion.img
            src="/reading.svg"
            alt="Person reading book"
            className="absolute inset-0 w-full h-full object-contain"
            animate={{
              y: ["0%", "2%", "0%"],
              rotate: [0, 1.5, 0],
            }}
            transition={{
              duration: 6,
              ease: "easeInOut",
              repeat: Infinity,
              repeatType: "mirror",
            }}
          />
        </div>

        {/* Right Side - Login Form */}
        <div className="w-full md:w-1/2 flex items-center justify-center p-8">
          <motion.div
            className="bg-white p-8 rounded-lg shadow-md w-full max-w-md z-10"
            initial={{ opacity: 0, x: 30 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.5, ease: "easeOut" }}
          >
            <h2 className="text-2xl font-bold mb-6 text-center">Sign in to Ebooks</h2>

            {/* Added pointer-events-none + opacity-70 to disable form during loading */}
            <form className={`space-y-4 ${loading ? 'pointer-events-none opacity-70' : ''}`} onSubmit={handleSubmit(onSubmit)}>
              <div>
                <label htmlFor="email" className="block text-sm font-medium text-gray-700"> Email address</label>
                <input type="email" id="email" {...register('email')} className="mt-1 block w-full px-4 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500" placeholder="you@example.com" />
                {errors.email && (
                  <p className="text-red-500 text-sm mt-1">{errors.email.message}</p>
                )}
              </div>

              <div>
                <label htmlFor="password" className="block text-sm font-medium text-gray-700"> Password </label>
                <input type="password" id="password" {...register('password')} className="mt-1 block w-full px-4 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500" placeholder="••••••"/>
                {errors.password && (
                  <p className="text-red-500 text-sm mt-1">{errors.password.message}</p>
                )}
              </div>

              <button type="submit" disabled={loading} className={`w-full py-2 px-4 bg-blue-600 text-white font-semibold rounded-md hover:bg-blue-700 transition duration-200 flex items-center justify-center gap-2 ${loading ? 'cursor-not-allowed opacity-70' : 'cursor-pointer'} `}>
                <span>{loading ? 'Signing in...' : 'Sign In'}</span>
                {loading ? (
                  <FiLoader
                    size={20}
                    className="animate-spin"
                    aria-label="Loading"
                  />
                ) : (
                  <FiArrowRightCircle size={20} />
                )}
              </button>
            </form>
          </motion.div>
        </div>
      </div>
    </>
  );
}
