import { z } from "zod";

export const loginSchema = z.object({
  email: z.string().min(1, "Email is required").email("Invalid email format"),
  password: z
    .string()
    .min(1, "Password is required")
    .min(6, "Password must have than 6 characters"),
});

export const userSchema = (isEditMode = false) =>
  z
    .object({
      user_empid: z.string().min(1, "Employee ID is required"),
      user_name: z.string().min(1, "Full Name is required"),
      user_email: z
        .string()
        .min(1, "Email is required")
        .email("Invalid email format"),
      user_role: z.string().min(1, "User Role is required"),
      user_password: isEditMode
        ? z.string().optional()
        : z.string().min(6, "Password must be at least 6 characters long"),
      user_con_pwd: isEditMode
        ? z.string().optional()
        : z
            .string()
            .min(6, "Confirm Password must be at least 6 characters long"),

      // ✅ Clean and simple user_profile rule
      user_profile: z.any().refine(
        (file) => {
          if (!(file instanceof FileList) || file.length === 0) return true; // optional
          const allowedTypes = ["image/png", "image/jpeg", "image/jpg"];
          return allowedTypes.includes(file[0].type);
        },
        {
          message: "Only .png, .jpg and .jpeg files are allowed",
        }
      ),
    })
    .refine(
      (data) => {
        if (isEditMode) return true;
        return data.user_password === data.user_con_pwd;
      },
      {
        message: "Passwords do not match",
        path: ["user_con_pwd"],
      }
    );


export const bookSchema = z.object({
  book_year: z.string().min(1, "Year of working title is required"),
  book_title: z.string().min(1, "Book title is required"),
  book_lead_editor_name: z.string().min(1, "Lead editor name is required"),
  book_id: z.string().min(1, "Book ID is required"),
  book_online_isbn: z.string().min(0, "Online ISBN is required").regex(/^[A-Za-z0-9\-:]+$/, "Only letters, numbers, hyphens, and colons are allowed"),
  book_author_invitations_sent: z.string().min(0, "This field is required"),
  book_au_invited: z.string().min(0, "AU Invited must be a number").regex(/^[0-9]+$/, "Only numbers are allowed"),
  book_au_accepted: z.string().min(0, "AU Accepted must be a number").regex(/^[0-9]+$/, "Only numbers are allowed"),
  book_chap_due_date: z.string().min(0, "Chapter Due Date is required"),
  book_drop_dead_date: z.string().min(0, "Drop dead date is required"),
  book_ch_sumitted: z.string().min(0, "Ch Submitted must be a number").regex(/^[0-9]+$/, "Only numbers are allowed"),
  book_ch_accepted: z.string().min(0, "Ch Accepted must be a number").regex(/^[0-9]+$/, "Only numbers are allowed"),
  book_last_due_production: z.string().min(0, "Last due to production is required"),
  book_project_manager: z.string().nonempty("Project manager is required"),
  // book_alter_project_manager: z.string().min(0, "Alternative Project manager is required"),
  book_note_comments: z.string().min(0, "Notes/comments are required"),
});

export const chapterSchema = z.object({
  chap_book_title: z.string().min(1, "Book title is required"),
  chap_manuscript_id: z.string().min(1, "Manuscript ID is required"),
  chap_title: z.string().min(1, "Chapter title is required"),
  //chap_wordcount: z.number().min(1, "Word Count is required"),
   chap_wordcount: z.string().min(1, "Word Count is required").regex(/^[0-9]+$/, "Only numbers are allowed"),
  // chap_wordcount: z.coerce.number({required_error: "Word Count is required"}).min(1, "Word Count must be at least 1"),
  chap_first_name: z.string().min(1, "First name is required"),
  chap_last_name: z.string().min(0, "Last name is required"),
  chap_corres_au_email: z
    .string()
    .min(0, "Corresponding author email is required"),

  chap_status: z.string().min(0, "Chapter status is required"),
  chap_p_status: z.string().min(0, "P+ status is required"),
  

  chap_pdue_date: z
    .string()
    .min(0, "P+ due date is required"),

  // chap_copy_editing: z
  //   .string()
  //   .nonempty("Copy-Editing is required")
  //   .refine((val) => val === "Yes" || val === "No", {
  //     message: "Select Copy-Editing as Yes or No",
  //   }),
  chap_comments: z.string().min(0, "Comments are required"),
});

export const knowledgeSchema = (isEditMode = false) =>
  z.object({
    knowledge_name: z.string().min(1, "Knowledge name is required"),
    knowledge_file: z.any().refine(
      (file) => {
        const allowedTypes = [
          "image/png",
          "image/jpeg",
          "application/pdf",
          "application/msword",
          "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
        ];

        if (!isEditMode) {
          return (
            file instanceof FileList &&
            file.length > 0 &&
            allowedTypes.includes(file[0].type)
          );
        }

        if (!(file instanceof FileList) || file.length === 0) {
          return true; // No file uploaded in edit mode is acceptable
        }

        return allowedTypes.includes(file[0].type);
      },
      {
        message: isEditMode
          ? "Only .png, .jpg, .jpeg, .pdf, .doc, and .docx files are allowed"
          : "Knowledge file is required and must be a valid file type",
      }
    ),
  });

  export const queriesAddSchema = (isEditMode = false) =>
  z.object({
    queries_name: z.string().min(1, "Query required"),
    queries_book_pid: z.string().min(1, "Book name is required"),
    queries_chap_pid: z.string().min(1, "Chapter name is required"),
    queries_task_pid: z.string().min(1, "Task name is required"),
    queries_name_reply: z.string().optional(),
    
  });

export const holidaySchema = z.object({
  holiday_date: z.string().min(1, "Holiday date is required"),
  holiday_name: z.string().min(1, "Holiday name is required"),
  holiday_type: z.string().min(1, "Holiday type is required"),
});

export const entryStartSchema = z.object({
  entry_book_pid: z.string().min(1, "Book Title is required"),
  entry_chapter_pid: z.string().min(1, "Chapter name is required"),
  entry_task_pid: z.string().min(1, "Task name is required"),
});

// export const entryEndSchema = entryStartSchema.extend({
//   entry_end_date: z.string().min(1, 'End Date is required'),
// });

export const entryEndSchema = z.object({
  entry_end_date: z.string().min(1, "End Date is required"),
});

// export const entryUpdateSchema = z.object({
//   entry_comments: z.string().min(1, "comments are required"),
//   entry_status: z.string().min(1, "Status is required"),
// });
// export const entryUpdateSchema = z
//     .object({
//     entry_comments: z
//       .any()
//       .refine(
//         (val) => val !== null && val !== undefined && String(val).trim() !== "",
//         { message: "Comments are required" }
//       ),
//     entry_status: z
//       .any()
//       .nullable()
//       .refine(
//         (val) => val != null && String(val).trim() !== "" && val !== "null",
//         { message: "Status is required" }
//       ),
//     entry_contact_email: z.string().optional(),
//     entry_upload_file: z.any().optional(),
//     })
//     .superRefine((data, ctx) => {

//     if (data.entry_status == 1 && (!data.entry_contact_email || data.entry_contact_email == "null")) {
//       ctx.addIssue({
//         path: ["entry_contact_email"],
//         message: "Contact Email is required",
//       });
//     }
//     if (
//       data.entry_status == 1 &&
//       (!data.entry_upload_file ||
//         !(data.entry_upload_file instanceof FileList) ||
//         data.entry_upload_file.length === 0)
//     ) {
//       ctx.addIssue({
//         path: ["entry_upload_file"],
//         message: "Upload File is required",
//       });
//     }
//   });
export const entryUpdateSchema = z
  .object({
    entry_task_pid: z.string().min(1, "Task ID is required"),
    entry_chapter_pid: z.string().min(1, "Chapter ID is required"),
    entry_wordcount: z.coerce.number().min(1, "Word count is required"),
    entry_comments: z
      .any()
      .refine(
        (val) => val !== null && val !== undefined && String(val).trim() !== "",
        { message: "Comments are required." }
      ),
    entry_status: z
      .any()
      .nullable()
      .refine(
        (val) => val != null && String(val).trim() !== "" && val !== "null",
        { message: "Status is required." }
      ),
    entry_contact_email: z.string().optional(),
    entry_upload_file: z.any().optional(),
  })
  .superRefine((data, ctx) => {
    // 🔹 Email required when status = 1
    if (
      (data.entry_status == 1 && (data.entry_task_pid == 1 || data.entry_task_pid == 2)) &&
      (!data.entry_contact_email || data.entry_contact_email === "null")
    ) {
      ctx.addIssue({
        path: ["entry_contact_email"],
        message: "Contact Email is required when status is Completed.",
      });
    }

    // 🔹 File required when status = 1
    if (
      (data.entry_status == 1 && (data.entry_task_pid == 1 || data.entry_task_pid == 2 || data.entry_task_pid == 3)) &&
      (!data.entry_upload_file ||
        !(data.entry_upload_file instanceof FileList) ||
        data.entry_upload_file.length === 0)
    ) {
      ctx.addIssue({
        path: ["entry_upload_file"],
        message: "Please upload a file (PDF, DOC, or DOCX) when status is Completed.",
      });
      return; // no need to continue checking file type
    }

    // 🔹 Validate file type if file exists
    if (
      (data.entry_task_pid == 1 || data.entry_task_pid == 2 || data.entry_task_pid == 3) &&
      data.entry_upload_file instanceof FileList &&
      data.entry_upload_file.length > 0
    ) {
      const allowedTypes = [
        "application/pdf",
        "application/msword",
        "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      ];

      const file = data.entry_upload_file[0];
      if (!allowedTypes.includes(file.type)) {
        ctx.addIssue({
          path: ["entry_upload_file"],
          message: "Only PDF, DOC, or DOCX files are allowed.",
        });
      }

      // 🔹 Optional: file size check (5 MB)
      const maxSizeMB = 25;
      const maxSizeBytes = maxSizeMB * 1024 * 1024;
      if (file.size > maxSizeBytes) {
        ctx.addIssue({
          path: ["entry_upload_file"],
          message: `File size must not exceed ${maxSizeMB} MB.`,
        });
      }
    }
  });



const acceptedFileTypes = [
  "image/png",
  "image/jpeg", // .jpg and .jpeg
  "application/pdf",
  "application/msword", // .doc
  "application/vnd.openxmlformats-officedocument.wordprocessingml.document", // .docx
];

// export const entryUpdateSchema = z.object({
//   entry_comments: z
//     .string()
//     .min(1, { message: "Comments are required" }),

//   // entry_contact_email: z
//   //   .string()
//   //   .min(1, { message: "Contact Email is required" })
//   //   .email({ message: "Invalid email address" }),
//   entry_contact_email: z.string().min(1, "Contact Email is required"),
//   entry_upload_file: z
//     .any()
//     .refine((files) => {
//       return files instanceof FileList && files.length > 0;
//     }, {
//       message: "Upload File is required",
//     })
//     .refine((files) => {
//       if (!(files instanceof FileList) || files.length === 0) return true; // skip this check if file isn't there (handled above)
//       const file = files[0];
//       return acceptedFileTypes.includes(file.type);
//     }, {
//       message: "Invalid file type. Allowed: png, jpg, jpeg, pdf, doc, docx",
//     }),

//   entry_status: z
//     .string()
//     .min(1, { message: "Status is required" }),
// });

export const pendingSchema = z.object({
  pending_reason: z.string().min(1, "pending reason is required"),
});
export const resumeSchema = z.object({
  resume_reason: z.string().min(1, "resume reason is required"),
});

export const breakSchema = z.object({
  entry_break_type: z.string().min(1, "Break type is required"),
});

export const edaSchema = z.object({
  entry_eda_comments: z.string().min(0, "comments is required"),  
  entry_eda_end_date: z.string().min(1, "Date is required"),
  entry_eda_status: z.string().min(1, "Status is required"),
  
});

export const rcbtrSchema = z.object({
  re_bt_review_file: z
    .any()
    .refine(
      (file) => file instanceof FileList && file.length > 0,
      "Please upload a valid file"
    ),
  re_bt_review_comments: z
    .string()
    .min(1, "Comments are required"),
  re_bt_review_date: z
    .string()
    .min(1, "Review date is required"),
  re_bt_review_status: z
    .string()
    .min(1, "Review status is required"),
});

export const rcarSchema = z.object({
  au_review_file: z
    .any()
    .refine(
      (file) => file instanceof FileList && file.length > 0,
      "Please upload a valid file"
    ),
  au_review_comments: z
    .string()
    .min(1, "Comments are required"),
  au_review_date: z
    .string()
    .min(1, "Review date is required"),
});

export const prSchema = z.object({
  entry_eda_comments: z.string().min(0, "comments is required"),  
  entry_eda_end_date: z.string().min(1, "Date is required"),
  entry_eda_status: z.string().min(1, "Status is required"),
  
});

export const mdSchema = z.object({
  entry_eda_comments: z.string().min(0, "comments is required"),  
  entry_eda_end_date: z.string().min(1, "Date is required"),
  entry_eda_status: z.string().min(1, "Status is required"),
  
});



// Zod schema for QC Review
// export const qcReviewSchema = z.object({
//   ce_required: z.enum(["yes", "no"], {
//     required_error: "Please select if CE is required",
//   }),
//   status: z.enum(["Unsubmit", "Completed"], {
//     required_error: "Please select a status",
//   }),
//   qc_notes: z.string().min(1, "QC Notes are required"),
//   qc_edited_file: z
//     .any()
//     .optional(), // <-- File validation can be tricky; z.any() is safest unless you want strict checking
// });


const acceptedFileTypesQC = [
  "image/png",
  "image/jpg",
  "image/jpeg",
  "application/pdf",
  "application/msword",
  "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
];

export const qcReviewSchema = z.object({
  ce_required: z.string().nonempty("CE is required"),
  status: z.string().nonempty("Status is required"),
  qc_notes: z.string().nonempty("QC Notes is required"),
  // currentStage: z.string().nonempty("Current stage is required"), 

  qc_edited_file: z
    .any()
    .refine((files) => {
      // check if 'files' exists, is an object, has length, and length > 0
      return (
        files &&
        typeof files === "object" &&
        "length" in files &&
        files.length > 0
      );
    }, {
      message: "QC Edited File is required",
    })
    .refine((files) => {
      if (
        !files ||
        typeof files !== "object" ||
        !("length" in files) ||
        files.length === 0
      )
        return true; // skip if not present
      const file = files[0];
      return acceptedFileTypesQC.includes(file.type);
    }, {
      message: "Invalid file type. Allowed: png, jpg, jpeg, pdf, doc, docx",
    }),
});

export const rcqcReviewSchema = z.object({
  rcqc_notes: z.string().nonempty("QC Notes is required"),
  // currentStage: z.string().nonempty("Current stage is required"),

  rcqc_edited_file: z
    .any()
    .refine((files) => {
      // check if 'files' exists, is an object, has length, and length > 0
      return (
        files &&
        typeof files === "object" &&
        "length" in files &&
        files.length > 0
      );
    }, {
      message: "RCQC Edited File is required",
    })
    .refine((files) => {
      if (
        !files ||
        typeof files !== "object" ||
        !("length" in files) ||
        files.length === 0
      )
        return true; // skip if not present
      const file = files[0];
      return acceptedFileTypesQC.includes(file.type);
    }, {
      message: "Invalid file type. Allowed: png, jpg, jpeg, pdf, doc, docx",
    }),
});

export const rcbtr_after_ac_ReviewSchema = z.object({
  rcbtr_after_ac_notes: z.string().nonempty("Notes is required"),
  // currentStage: z.string().nonempty("Current stage is required"),

  rcbtr_after_ac_edited_file: z
    .any()
    .refine((files) => {
      // check if 'files' exists, is an object, has length, and length > 0
      return (
        files &&
        typeof files === "object" &&
        "length" in files &&
        files.length > 0
      );
    }, {
      message: "Edited File is required",
    })
    .refine((files) => {
      if (
        !files ||
        typeof files !== "object" ||
        !("length" in files) ||
        files.length === 0
      )
        return true; // skip if not present
      const file = files[0];
      return acceptedFileTypesQC.includes(file.type);
    }, {
      message: "Invalid file type. Allowed: png, jpg, jpeg, pdf, doc, docx",
    }),
});



// export const userSchema = z
//   .object({
//     user_empid: z.string().min(1, 'Employee ID is required'),
//     user_name: z.string().min(1, 'Full Name is required'),
//     user_email: z.string().min(1, 'Email is required').email('Invalid email format'),
//     user_role: z.string().min(1, 'User Role is required'),
//     user_password: z.string().min(6, 'Password must be at least 6 characters long'),
//     user_con_pwd: z.string().min(6, 'Confirm Password must be at least 6 characters long'),
//     user_profile: z
//       .any()
//       .refine((file) => file instanceof FileList && file.length > 0, {
//         message: 'Profile picture is required',
//       })
//       .refine(
//         (file) => {
//           if (!(file instanceof FileList) || file.length === 0) return false;
//           const allowedTypes = ['image/png', 'image/jpeg', 'image/jpg'];
//           return allowedTypes.includes(file[0].type);
//         },
//         {
//           message: 'Only .png, .jpg and .jpeg files are allowed',
//         }
//       ),
//   })
//   .refine((data) => data.user_password === data.user_con_pwd, {
//     message: 'Passwords do not match',
//     path: ['user_con_pwd'],
//   });

export const entryAllUpdateSchema = z.object({
  entry_start_date: z.string().min(1, "Start Date is required"),
  entry_start_time: z.string().min(1, "Start Time is required"),
  entry_end_date: z.string().min(1, "End Date is required"),
  entry_end_time: z.string().min(1, "End Time is required"),
  entry_comments: z.string().min(1, "comments are required"),
  // entry_upload_file: z
  //   .any()
  //   .refine((files) => {
  //     // check if 'files' exists, is an object, has length, and length > 0
  //     return (
  //       files &&
  //       typeof files === "object" &&
  //       "length" in files &&
  //       files.length > 0
  //     );
  //   }, {
  //     message: "File is required",
  //   })
  //   .refine((files) => {
  //     if (
  //       !files ||
  //       typeof files !== "object" ||
  //       !("length" in files) ||
  //       files.length === 0
  //     )
  //       return true; // skip if not present
  //     const file = files[0];
  //     return acceptedFileTypes.includes(file.type);
  //   }, {
  //     message: "Invalid file type. Allowed: png, jpg, jpeg, pdf, doc, docx",
  //   }),
});

export const ceSchema = z.object({
  ce_emp_id: z.string().min(1, "Editor Name is required"),
  ce_due_date: z.string().min(1, "Due Date is required"),
  ce_notes: z.string().min(1, "Notes is required"),
  ce_comments: z.string().min(1, "Comments is required"),
});