import prisma from '@/lib/prismaconfig';

export const userperformanceReportDetails = async (payload) => {
  const { user_pid, start_date, end_date } = payload;

  // Build dynamic where conditions
  const where = { entry_isdelete: 'active' };

  if (user_pid) {
    where.entry_createdBy = Number(user_pid);
  }

  if (start_date && end_date) {
    where.entry_date = {
      gte: start_date,
      lte: end_date,
    };
  } else {
    return [];
  }

  // Fetch entries
  const entries = await prisma.acs_entry.findMany({
    where,
    select: {
      entry_createdBy: true,
      entry_taskId: true,
      entry_duration: true, // string field
    },
  });

   // Helper function to convert time string (HH:MM:SS) into seconds
  const convertTimeToSeconds = (time) => {
    const [hours, minutes, seconds] = time.split(':').map(Number);
    return (hours * 3600) + (minutes * 60) + (seconds || 0);
  };

  // Group by entry_createdBy and entry_taskId
  const grouped = {};

  for (const entry of entries) {
    const key = `${entry.entry_createdBy}-${entry.entry_taskId}`;
    if (!grouped[key]) {
      grouped[key] = {
        entry_createdBy: entry.entry_createdBy,
        entry_taskId: entry.entry_taskId,
        total_duration: 0,
        count: 0,
      };
    }

    const durationInSeconds = convertTimeToSeconds(entry.entry_duration);
    grouped[key].total_duration += durationInSeconds || 0; // Ensure valid number
    grouped[key].count += 1;
  }

  const groupedArray = Object.values(grouped);

  // Fetch user and task names
  const userIds = [...new Set(groupedArray.map((e) => e.entry_createdBy))];
  const taskIds = [...new Set(groupedArray.map((e) => e.entry_taskId))];

  const users = await prisma.acs_users.findMany({
    where: { user_pid: { in: userIds } },
    select: { user_pid: true, user_name: true },
  });

  const tasks = await prisma.acs_task.findMany({
    where: { task_pid: { in: taskIds } },
    select: { task_pid: true, task_name: true },
  });

  const userMap = Object.fromEntries(users.map((u) => [u.user_pid, u.user_name]));
  const taskMap = Object.fromEntries(tasks.map((t) => [t.task_pid, t.task_name]));

  // Format results
  const formatTime = (seconds) => {
    const h = Math.floor(seconds / 3600).toString().padStart(2, '0');
    const m = Math.floor((seconds % 3600) / 60).toString().padStart(2, '0');
    const s = Math.floor(seconds % 60).toString().padStart(2, '0');
    return `${h}:${m}:${s}`;
  };

  const result = groupedArray.map((g) => {
    return {
      task_name: taskMap[g.entry_taskId] || 'Unknown Task',
      user_name: userMap[g.entry_createdBy] || 'Unknown User',
      totalTime: formatTime(g.total_duration),
      task_count: g.count,
      avg_time: formatTime(g.total_duration / (g.count || 1)),
    };
  });
  return result;
};

export const userallperformanceDetails = async (payload) => {
  const { user_pid, start_date, end_date } = payload;

  const where = { entry_isdelete: 'active' };

  if (user_pid) {
    where.entry_createdBy = Number(user_pid);
  }

  if (start_date && end_date) {
    where.entry_date = {
      gte: start_date,
      lte: end_date,
    };
  } else {
    return [];
  }

  const entries = await prisma.acs_entry.findMany({
    where,
    select: {
      entry_duration: true,
      entry_createdBy: true,
      entry_taskId: true,
      user: {
        select: { user_name: true },
      },
      task: {
        select: { task_name: true },
      },
    },
  });

   // Directly format result per record
  const result = entries.map((entry) => ({
    user_name: entry.user?.user_name || 'Unknown User',
    task_name: entry.task?.task_name || 'Unknown Task',
    entry_duration: entry.entry_duration || '00:00:00',
    task_count: '1',
  }));

  return result;
};









