"use client";

import { useState, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiX, FiDownload, FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { rcqcReviewSchema } from "@/validation/validation";  // Your Zod schema
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";
import { fetchChapterDetailsforRCQCAction, updateRCQCEntryAction } from "@/lib/actions/entryActions";
import { useSession } from "next-auth/react";

export default function RCQCReviewModal({ isOpen, onClose, onCancel, chapterId, initialstatus }) {
  const { data: session, status } = useSession();
  const [loading, setLoading] = useState(false);
  const [chapterData, setChapterData] = useState(null);
  // console.log(chapterData, "chapterDatachapterData")
  // const [selectedFileOption, setSelectedFileOption] = useState('ce'); // 'ce' or 'upload'
  const router = useRouter();
  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm({
    resolver: zodResolver(rcqcReviewSchema),
    mode: "onBlur",
    defaultValues: {
      chapter_title: "",
      manuscript_id: "",
    },
  });

  useEffect(() => {
    if (!chapterId) return;

    async function fetchIndividualChapters() {
      try {
        const chapter_res = await fetchChapterDetailsforRCQCAction(chapterId); // ✅ directly call server action
        if (chapter_res?.success) {
          setChapterData(chapter_res.result);
        } else {
          toast.error('chapter fetching error')
        }
      } catch (err) {
        console.error(err);
      }
    }
    fetchIndividualChapters();
  }, [chapterId]);

  const handleDownload = (filePath) => {
    const filename = filePath.split("/").pop();

    const link = document.createElement("a");
    link.href = filePath; // Direct path to file (e.g. '/files/report.pdf')
    link.download = filename;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  const onSubmit = async (data) => {
    setLoading(true);
    try {
      const formData = new FormData();
      formData.append("status", "1");
      formData.append("rcqc_notes", data.rcqc_notes);
      formData.append("chapterId", chapterId);
      formData.append("user_pid", session.user.user_pid);
      formData.append("file_option", "upload");

      // Always append the uploaded file, renamed with _RCQC suffix
      if (data.rcqc_edited_file && data.rcqc_edited_file.length > 0) {
        const file = data.rcqc_edited_file[0];
        const originalName = file.name;
        const extension = originalName.split('.').pop();
        const baseName = originalName.replace('.' + extension, '');
        const newName = baseName + '_RCQC.' + extension;
        const renamedFile = new File([file], newName, { type: file.type });
        formData.append("rcqc_edited_file", renamedFile);
      }

      const QC_res = await updateRCQCEntryAction(formData);
      if (QC_res?.success) {
        toast.success(`RCQC Status updated successfully!`);
        // Success logic: close modal and reset form
        reset();
        onClose();
        router.refresh();
      } else {
        toast.error(QC_res?.message || "Failed to create entry.");
      }
    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const statusOptions = initialstatus.map((s) => ({
    value: String(s.status_pid),
    label: s.status_name.charAt(0).toUpperCase() + s.status_name.slice(1),
  }));

  return (
    <AnimatePresence>
      {isOpen && (
        <motion.div
          className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
        >
          <motion.div
            className="bg-white rounded-lg shadow-xl w-full max-w-2xl overflow-hidden"
            initial={{ scale: 0.9, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            exit={{ scale: 0.9, opacity: 0 }}
            transition={{ type: "spring", stiffness: 300, damping: 25 }}
          >
            {/* Header */}
            <div className="flex items-center justify-between px-5 py-4 border-b border-blue-200 bg-blue-50">
              <div className="flex flex-col">
                <h3 className="text-lg font-semibold text-blue-800">
                  RCQC Review
                </h3>
                <div className="text-sm text-gray-600">
                  <div>
                    Book Title:{" "}
                    <span className="font-semibold">
                      {chapterData?.book?.book_title}
                    </span>
                  </div>
                  <div>
                    Chapter Title:{" "}
                    <span className="font-semibold">
                      {chapterData?.chapter?.chap_title}
                    </span>
                  </div>
                  <div>
                    Manuscript ID:{" "}
                    <span className="font-semibold">
                      {chapterData?.chapter?.chap_manuscriptID}
                    </span>
                  </div>
                </div>
              </div>

              <button
                onClick={() => {
                  onClose();
                  reset();
                }}
                className="text-gray-500 hover:text-gray-700 transition cursor-pointer"
                aria-label="Close modal"
              >
                <FiX size={20} />
              </button>
            </div>

            {/* Form (Wraps body & footer) */}
            <form onSubmit={handleSubmit(onSubmit)}>
              {/* Body */}
              <div className="px-5 py-4 text-sm text-gray-700 space-y-3">
                <div className="relative overflow border border-gray-200 rounded-md z-10">
                  {/* Initial Check Notes */}
                  <table className="min-w-full table-auto border border-gray-100 text-sm">
                    <tbody className="text-gray-700">
                      <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                          CE Received Comments
                        </td>
                        <td className="px-4 py-2">
                          <div className="p-2 bg-gray-100 rounded-md text-sm text-gray-800 whitespace-pre-wrap flex-1 max-h-[6.25rem] overflow-y-auto">
                            {chapterData?.chapter?.ce_editing?.[0]?.ce_received_note || "No CE Notes available."}
                          </div>
                        </td>
                      </tr>
                      <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                          Download CE Received File
                        </td>
                        <td className="px-4 py-2">
                          {(() => {
                            const chapter = chapterData?.chapter;
                            const stage = chapterData;

                            let fileName;

                            // condition
                            if (stage?.stage_from === "RCBTR" && stage?.stage_to === "RCQC") {
                              fileName = stage?.stage_uploadFile;
                            } else {
                              fileName = chapter?.ce_editing?.[0]?.ce_received_file;
                            }

                            if (!fileName) return null;

                            // download path
                           const downloadPath =
                              stage?.stage_from === "RCBTR" && stage?.stage_to === "RCQC"
                                ? `/${fileName}`  // when condition is TRUE
                                : `/CE/Recieve_Files/${fileName}`;  // fallback

                            // console.log(downloadPath, "downloadPath");

                            return (
                              <button
                                type="button"
                                onClick={() => handleDownload(downloadPath)}
                                className="px-2 py-1 text-xs bg-blue-600 text-white rounded-md hover:bg-blue-700 transition flex items-center gap-1"
                              >
                                <FiDownload size={14} />
                                Download
                              </button>
                            );
                          })()}
                        </td>

                      </tr>
                      <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                          Upload RCQC File
                        </td>
                        <td className="px-4 py-2">
                          <input
                            type="file"
                            id="rcqc_edited_file"
                            {...register('rcqc_edited_file')}
                            accept=".pdf,.docx,application/pdf,application/vnd.openxmlformats-officedocument.wordprocessingml.document"
                            className={`block w-full text-sm text-gray-700
                        file:bg-blue-100 file:text-blue-700 file:rounded file:px-4 file:py-2 file:border-0
                        cursor-pointer
                        border rounded-md
                        ${errors.rcqc_edited_file ? 'border-red-500' : 'border-gray-300'}
                        focus:outline-none focus:ring-2 focus:ring-blue-500
                      `}
                          />
                          {errors.rcqc_edited_file && (
                            <p className="text-sm text-red-500 mt-1">{errors.rcqc_edited_file.message}</p>
                          )}
                        </td>
                      </tr>
                      <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                          Upload RCQC Notes
                        </td>
                        <td className="px-4 py-2">
                          <textarea
                            id="rcqc_notes"
                            placeholder="Enter RCQC notes"
                            rows={4}
                            {...register("rcqc_notes")}
                            className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.rcqc_notes
                              ? "border-red-500"
                              : "border-gray-300"
                              } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                          />
                          {errors.rcqc_notes && (
                            <p className="text-sm text-red-500 mt-1">
                              {errors.rcqc_notes.message}
                            </p>
                          )}
                        </td>
                      </tr>

                    </tbody>
                  </table>
                </div>
              </div>

              {/* Footer */}
              <div className="flex justify-end gap-2 px-5 py-3 bg-gray-50 border-t border-gray-200">
                <button
                  type="button"
                  onClick={() => {
                    onClose();
                    reset();
                  }}
                  className="w-full sm:w-auto px-5 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  className="w-full sm:w-auto px-5 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2"
                  disabled={loading}
                >
                  <span>{loading ? "Updating..." : "Submit"}</span>
                  {loading ? (
                    <FiLoader size={20} className="animate-spin" />
                  ) : (
                    <FiArrowRightCircle size={20} />
                  )}
                </button>
              </div>
            </form>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
}
