"use client";

import { useState, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiX, FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { rcbtrSchema } from "@/validation/validation"; // Your Zod schema
import { toast } from "react-toastify";
import {
  fetchChapterDetailsforQCAction,
  RCBTRUpdateAction,
} from "@/lib/actions/entryActions";
import { useSession } from "next-auth/react";
import { useRouter } from "next/navigation";
import Select from "react-select";

export default function RCBTRReviewModal({
  isOpen,
  onClose,
  onCancel,
  chapterId,
  initialstatus,
}) {
  const { data: session } = useSession();
  const [chapterData, setChapterData] = useState(null);
  const [loading, setLoading] = useState(false);
  const router = useRouter();

  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
  } = useForm({
    resolver: zodResolver(rcbtrSchema),
    mode: "onBlur",
    defaultValues: {
      re_bt_review_file: "",
      re_bt_review_comments: "",
      re_bt_review_date: "",
      re_bt_review_status: "",
    },
  });

  useEffect(() => {
    if (!chapterId) return;
    async function fetchIndividualChapters() {
      try {
        const chapter_res = await fetchChapterDetailsforQCAction(chapterId);
        if (chapter_res?.success) {
          setChapterData(chapter_res.result);
        } else {
          toast.error("Error fetching chapter details");
        }
      } catch (err) {
        console.error(err);
      }
    }
    fetchIndividualChapters();
  }, [chapterId]);

  const onSubmit = async (data) => {
    try {
      setLoading(true);
      const formData = new FormData();
      formData.append("re_bt_review_comments", data.re_bt_review_comments);
      formData.append("re_bt_review_date", data.re_bt_review_date);
      formData.append("re_bt_review_status", data.re_bt_review_status);
      formData.append("chapterId", chapterId);
      formData.append("user_pid", session.user.user_pid);

      if (data.re_bt_review_file && data.re_bt_review_file.length > 0) {
        formData.append("re_bt_review_file", data.re_bt_review_file[0]);
      }

      const response = await RCBTRUpdateAction(formData);
      if (response?.success) {
        toast.success("Revision Check Review updated successfully!");
        reset();
        onClose();
        router.refresh();
      } else {
        toast.error(response?.message || "Failed to update entry.");
      }
    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  // const statusOptions = initialstatus.map((s) => ({
  //   value: String(s.status_pid),
  //   label: s.status_name.charAt(0).toUpperCase() + s.status_name.slice(1),
  // }));

  const statusOptions = initialstatus.map((s) => ({
    value: String(s.status_pid),
    label:
      s.status_pid === 1
        ? "Send to ER"
        : s.status_pid === 4
          ? "Send to Author"
          : s.status_name.charAt(0).toUpperCase() + s.status_name.slice(1),
  }));

  return (
    <AnimatePresence>
      {isOpen && (
        <motion.div
          // Overlay
          className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
          onClick={(e) => {
            // prevent modal close on overlay click while loading
            if (e.target === e.currentTarget && !loading) {
              onClose();
              reset();
            }
          }}
        >
          {/* Modal Box */}
          <motion.div
            className="bg-white rounded-lg shadow-xl w-full max-w-xl overflow-hidden"
            initial={{ scale: 0.9, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            exit={{ scale: 0.9, opacity: 0 }}
            transition={{ type: "spring", stiffness: 300, damping: 25 }}
          >
            {/* Header */}
            <div className="flex items-center justify-between px-5 py-4 border-b border-blue-200 bg-blue-50">
              <div className="flex flex-col">
                <h3 className="text-lg font-semibold text-blue-800">
                  Revision Check Review
                </h3>
                <div className="text-sm text-gray-600">
                  <div>
                    Book Title:{" "}
                    <span className="font-semibold">
                      {chapterData?.book?.book_title}
                    </span>
                  </div>
                  <div>
                    Chapter Title:{" "}
                    <span className="font-semibold">
                      {chapterData?.chapter?.chap_title}
                    </span>
                  </div>
                  <div>
                    Manuscript ID:{" "}
                    <span className="font-semibold">
                      {chapterData?.chapter?.chap_manuscriptID}
                    </span>
                  </div>
                </div>
              </div>

              {/* Close Button */}
              <button
                onClick={() => {
                  if (!loading) {
                    onClose();
                    reset();
                  }
                }}
                disabled={loading}
                className={`text-gray-500 transition cursor-pointer ${loading
                    ? "opacity-50 cursor-not-allowed"
                    : "hover:text-gray-700"
                  }`}
                aria-label="Close modal"
              >
                <FiX size={20} />
              </button>
            </div>

            {/* Form */}
            <form onSubmit={handleSubmit(onSubmit)}>
              <div className="px-5 py-4 text-sm text-gray-700 space-y-3">
                <div className="relative overflow border border-gray-200 rounded-md z-10">
                  <table className="min-w-full table-auto border border-gray-100 text-sm">
                    <tbody className="text-gray-700">
                      {/* File Upload */}
                      <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                          Upload Review File <span className="text-red-500">*</span>
                        </td>
                        <td className="px-4 py-2">
                          <input
                            type="file"
                            id="re_bt_review_file"
                            {...register("re_bt_review_file")}
                            accept=".pdf,.docx,application/pdf,application/vnd.openxmlformats-officedocument.wordprocessingml.document"
                            disabled={loading}
                            className={`block w-full text-sm text-gray-700 file:bg-blue-100 file:text-blue-700 file:rounded file:px-4 file:py-2 file:border-0 cursor-pointer border rounded-md ${errors.re_bt_review_file
                                ? "border-red-500"
                                : "border-gray-300"
                              } focus:outline-none focus:ring-2 focus:ring-blue-500 ${loading ? "opacity-60 cursor-not-allowed" : ""
                              }`}
                          />
                          {errors.re_bt_review_file && (
                            <p className="text-sm text-red-500 mt-1">
                              {errors.re_bt_review_file.message}
                            </p>
                          )}
                        </td>
                      </tr>

                      {/* Comments */}
                      <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                          Comments <span className="text-red-500">*</span>
                        </td>
                        <td className="px-4 py-2">
                          <textarea
                            id="re_bt_review_comments"
                            placeholder="Enter comments"
                            rows={4}
                            {...register("re_bt_review_comments")}
                            disabled={loading}
                            className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.re_bt_review_comments
                                ? "border-red-500"
                                : "border-gray-300"
                              } focus:outline-none focus:ring-2 focus:ring-blue-500 ${loading ? "opacity-60 cursor-not-allowed" : ""
                              }`}
                          />
                          {errors.re_bt_review_comments && (
                            <p className="text-sm text-red-500 mt-1">
                              {errors.re_bt_review_comments.message}
                            </p>
                          )}
                        </td>
                      </tr>

                      {/* Date */}
                      <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                          Date <span className="text-red-500">*</span>
                        </td>
                        <td className="px-4 py-2">
                          <input
                            id="re_bt_review_date"
                            type="date"
                            {...register("re_bt_review_date")}
                            disabled={loading}
                            className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm bg-white text-gray-700 ${errors.re_bt_review_date
                                ? "border-red-500"
                                : "border-gray-300"
                              } focus:outline-none focus:ring-2 focus:ring-blue-500 ${loading ? "opacity-60 cursor-not-allowed" : ""
                              }`}
                          />
                          {errors.re_bt_review_date && (
                            <p className="text-sm text-red-500 mt-1">
                              {errors.re_bt_review_date.message}
                            </p>
                          )}
                        </td>
                      </tr>

                      {/* Status */}
                      <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                          Status <span className="text-red-500">*</span>
                        </td>
                        <td className="px-4 py-2">
                          <Controller
                            name="re_bt_review_status"
                            control={control}
                            render={({ field }) => (
                              <Select
                                {...field}
                                className="z-50 relative"
                                instanceId="re-bt-review-status"
                                options={statusOptions}
                                onChange={(selected) =>
                                  field.onChange(selected ? selected.value : "")
                                }
                                value={
                                  statusOptions.find(
                                    (opt) => opt.value === field.value
                                  ) || null
                                }
                                placeholder="Select a status"
                                isClearable
                                isDisabled={loading}
                                inputId="re_bt_review_status"
                                menuPortalTarget={document.body}
                                styles={{
                                  control: (base) => ({
                                    ...base,
                                    borderColor: errors.re_bt_review_status
                                      ? "red"
                                      : base.borderColor,
                                    opacity: loading ? 0.6 : 1,
                                    cursor: loading
                                      ? "not-allowed"
                                      : base.cursor,
                                  }),
                                  menuPortal: (base) => ({
                                    ...base,
                                    zIndex: 9999,
                                  }),
                                }}
                              />
                            )}
                          />
                          {errors.re_bt_review_status && (
                            <p className="text-sm text-red-500 mt-1">
                              {errors.re_bt_review_status.message}
                            </p>
                          )}
                        </td>
                      </tr>
                    </tbody>
                  </table>
                </div>
              </div>

              {/* Footer Buttons */}
              <div className="flex justify-end gap-2 px-5 py-3 bg-gray-50 border-t border-gray-200">
                <button
                  type="button"
                  onClick={() => {
                    if (!loading) {
                      onClose();
                      reset();
                    }
                  }}
                  disabled={loading}
                  className={`w-full sm:w-auto px-5 py-2 rounded-md text-white shadow-sm flex items-center gap-2 text-sm font-medium transform focus:outline-none focus:ring-2 focus:ring-offset-2 ${loading
                      ? "bg-red-400 cursor-not-allowed opacity-70"
                      : "bg-red-600 hover:bg-red-700 focus:ring-red-500 hover:scale-105"
                    }`}
                >
                  Cancel
                </button>

                <button
                  type="submit"
                  disabled={loading}
                  className={`w-full sm:w-auto px-5 py-2 rounded-md text-white shadow-sm flex items-center gap-2 text-sm font-medium transform focus:outline-none focus:ring-2 focus:ring-offset-2 ${loading
                      ? "bg-green-400 cursor-not-allowed opacity-70"
                      : "bg-green-600 hover:bg-green-700 focus:ring-green-500 hover:scale-105"
                    }`}
                >
                  <span>{loading ? "Updating..." : "Submit"}</span>
                  {loading ? (
                    <FiLoader size={20} className="animate-spin" />
                  ) : (
                    <FiArrowRightCircle size={20} />
                  )}
                </button>
              </div>
            </form>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
}
