"use client";

import { useState, useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiX, FiDownload, FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { qcReviewSchema } from "@/validation/validation";  // Your Zod schema
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";
import { fetchChapterDetailsforQCAction,updateQCEntryAction } from "@/lib/actions/entryActions";
import Select from "react-select";
import { useSession } from "next-auth/react";
export default function QCReviewModal({ isOpen, onClose, onCancel, chapterId, initialstatus }) {
     const { data: session, status } = useSession(); 
  const [loading, setLoading] = useState(false);
  const [chapterData, setChapterData] = useState(null);
   const router = useRouter();
  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
  } = useForm({
    resolver: zodResolver(qcReviewSchema),
    mode: "onBlur",
    defaultValues: {
      ce_required: "",
      status: "",
      chapter_title: "",
      manuscript_id: "",
    },
  });
  useEffect(() => {
    if (!chapterId) return;

    async function fetchIndividualChapters() {
      try {
        const chapter_res = await fetchChapterDetailsforQCAction(chapterId); // ✅ directly call server action
        if (chapter_res?.success) {
          setChapterData(chapter_res.result);
        } else {
          toast.error('chapter fetching error')
        }
      } catch (err) {
        console.error(err);
      }
    }
    fetchIndividualChapters();
  }, [chapterId]);

   const handleDownload = (filePath) => {
    const link = document.createElement("a");
    link.href = `/${filePath}`;
    link.download = filePath.split("/").pop(); // Extract filename from path
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

 const onSubmit = async (data) => {
     setLoading(true);
     try {
         const formData = new FormData();
         formData.append("ce_required", data.ce_required);
         formData.append("status", data.status);
         formData.append("qc_notes", data.qc_notes);
         formData.append("chapterId", chapterId);
         formData.append("user_pid", session.user.user_pid);
 
         // Append file - data.qc_edited_file is a FileList, so append first file
        if (data.qc_edited_file && data.qc_edited_file.length > 0) {
          formData.append("qc_edited_file", data.qc_edited_file[0]);
        }
        const QC_res = await updateQCEntryAction(formData);
             if (QC_res?.success ) {
               toast.success(`QC Status updated successfully!`);
               // Success logic: close modal and reset form
               reset();
               onClose();
               router.refresh();
             } else {
               toast.error(QC_res?.message || "Failed to create entry.");
             }
     } catch (error) {
       toast.error(error.message || "Unexpected error occurred");
       console.error(error);
     } finally {
       setLoading(false);
     }
   };
 const statusOptions = initialstatus.map((s) => ({
        value: String(s.status_pid),
        label: s.status_name.charAt(0).toUpperCase() + s.status_name.slice(1),
    }));
  return (
    <AnimatePresence>
      {isOpen && (
        <motion.div
          className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
        >
          <motion.div
            className="bg-white rounded-lg shadow-xl w-full max-w-2xl overflow-hidden"
            initial={{ scale: 0.9, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            exit={{ scale: 0.9, opacity: 0 }}
            transition={{ type: "spring", stiffness: 300, damping: 25 }}
          >
            {/* Header */}
            <div className="flex items-center justify-between px-5 py-4 border-b border-blue-200 bg-blue-50">
              <div className="flex flex-col">
                <h3 className="text-lg font-semibold text-blue-800">
                  QC Review
                </h3>
                <div className="text-sm text-gray-600">
                  <div>
                    Book Title:{" "}
                    <span className="font-semibold">
                      {chapterData?.book?.book_title}
                    </span>
                  </div>
                  <div>
                    Chapter Title:{" "}
                    <span className="font-semibold">
                      {chapterData?.chapter?.chap_title}
                    </span>
                  </div>
                  <div>
                    Manuscript ID:{" "}
                    <span className="font-semibold">
                      {chapterData?.chapter?.chap_manuscriptID}
                    </span>
                  </div>
                </div>
              </div>

              <button
                onClick={() => {
                  onClose();
                  reset();
                }}
                className="text-gray-500 hover:text-gray-700 transition cursor-pointer"
                aria-label="Close modal"
              >
                <FiX size={20} />
              </button>
            </div>

            {/* Form (Wraps body & footer) */}
            <form onSubmit={handleSubmit(onSubmit)}>
              {/* Body */}
              <div className="px-5 py-4 text-sm text-gray-700 space-y-3">
                 <div className="relative overflow border border-gray-200 rounded-md z-10">
                     {/* Initial Check Notes */}
                    <table className="min-w-full table-auto border border-gray-100 text-sm">
                      <tbody className="text-gray-700">
                        <tr className="border-b border-gray-100">
                          <td className="font-medium px-4 py-1 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                            Initial Check Notes
                          </td>
                          <td className="px-4 py-2">
                            <div className="p-2 bg-gray-100 rounded-md text-sm text-gray-800 whitespace-pre-wrap flex-1 max-h-[6.25rem] overflow-y-auto">
                              {chapterData?.stage_notes || "No notes available."}
                            </div>
                          </td>
                        </tr>
                        <tr className="border-b border-gray-100">
                          <td className="font-medium px-4 py-1 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                            Download Initial Check File
                          </td>
                          <td className="px-4 py-1">
                            {chapterData?.stage_uploadFile && (
                              <button
                                type="button"
                                onClick={() =>
                                  handleDownload(chapterData.stage_uploadFile)
                                }
                                className="px-2 py-1 text-xs bg-blue-600 text-white rounded-md hover:bg-blue-700 transition flex items-center gap-1"
                              >
                                <FiDownload size={14} />
                                Download
                              </button>
                            )}
                          </td>
                        </tr>
                        <tr className="border-b border-gray-100">
                          <td className="font-medium px-4 py-1 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                            CE Required
                          </td>
                          <td className="px-4 py-1">
                            <div className="flex flex-col gap-1">
                              <div className="flex items-center gap-3">
                                <input
                                  type="radio"
                                  value="yes"
                                  {...register("ce_required")}
                                  className="text-blue-600"
                                />
                                <label className="text-sm">Yes</label>

                                <input
                                  type="radio"
                                  value="no"
                                  {...register("ce_required")}
                                  className="text-blue-600"
                                />
                                <label className="text-sm">No</label>
                              </div>
                              {errors.ce_required && (
                                <p className="text-xs text-red-500">
                                  {errors.ce_required.message}
                                </p>
                              )}
                            </div>
                          </td>
                        </tr>
                        <tr className="border-b border-gray-100">
                          <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                            Upload QC Edited File
                          </td>
                          <td className="px-4 py-1">
                            <input
                              type="file"
                              id="qc_edited_file"
                              {...register('qc_edited_file')}
                              accept=".pdf,.docx,application/pdf,application/vnd.openxmlformats-officedocument.wordprocessingml.document"
                              className={`block w-full text-sm text-gray-700
                        file:bg-blue-100 file:text-blue-700 file:rounded file:px-4 file:py-1 file:border-0
                        cursor-pointer
                        border rounded-md
                        ${errors.qc_edited_file ? 'border-red-500' : 'border-gray-300'}
                        focus:outline-none focus:ring-2 focus:ring-blue-500
                      `}
                            />
                            {errors.qc_edited_file && (
                              <p className="text-sm text-red-500 mt-1">{errors.qc_edited_file.message}</p>
                            )}
                          </td>
                        </tr>
                        <tr className="border-b border-gray-100">
                          <td className="font-medium px-4 py-1 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                            Upload QC Notes
                          </td>
                          <td className="px-4 py-1">
                            <textarea
                              id="qc_notes"
                              placeholder="Enter QC notes"
                              rows={4}
                              {...register("qc_notes")}
                              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.qc_notes
                                  ? "border-red-500"
                                  : "border-gray-300"
                                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                            />
                            {errors.qc_notes && (
                              <p className="text-sm text-red-500 mt-1">
                                {errors.qc_notes.message}
                              </p>
                            )}
                          </td>
                        </tr>
                        <tr className="border-b border-gray-100">
                        <td className="font-medium px-4 py-1 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                            Status
                          </td>
                          <td className="px-4 py-1">
                            <Controller
                              name="status"
                              control={control}
                              render={({ field }) => (
                                <Select
                                  {...field}
                                  className="z-50 relative"
                                  instanceId="entry-book-pid-select"
                                  options={statusOptions}
                                  onChange={(selected) =>
                                    field.onChange(selected ? selected.value : "")
                                  }
                                  value={
                                    statusOptions.find((option) => option.value === field.value) || null
                                  }
                                  placeholder="Select a status"
                                  isClearable
                                  inputId="status"
                                  menuPortalTarget={document.body}
                                  styles={{
                                    control: (base) => ({
                                      ...base,
                                      borderColor: errors.status ? "red" : base.borderColor,
                                    }),
                                    menuPortal: (base) => ({
                                      ...base,
                                      zIndex: 9999,
                                    }),
                                    menu: (base) => ({
                                      ...base,
                                      padding: 0, // Remove any padding around the menu
                                    }),
                                    option: (base) => ({
                                      ...base,
                                      padding: '8px 12px', // Adjust padding between options
                                      fontSize: '14px', // Optional: Adjust font size if needed
                                    }),
                                  }}
                                />
                              )}
                            />
                            {errors.status && (
                              <p className="text-sm text-red-500 mt-1">
                                {errors.status.message}
                              </p>
                            )}
                          </td>
                        </tr>

                      </tbody>
                    </table>
                 </div>
              </div>

              {/* Footer */}
              <div className="flex justify-end gap-2 px-5 py-2 bg-gray-50 border-t border-gray-200">
                <button
                  type="button"
                  onClick={() => {
                    onClose();
                    reset();
                  }}
                  className="w-full sm:w-auto px-5 py-1 bg-red-600 text-white rounded-md hover:bg-red-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  className="w-full sm:w-auto px-5 py-1 bg-green-600 text-white rounded-md hover:bg-green-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2"
                  disabled={loading}
                >
                  <span>{loading ? "Updating..." : "Submit"}</span>
                  {loading ? (
                    <FiLoader size={20} className="animate-spin" />
                  ) : (
                    <FiArrowRightCircle size={20} />
                  )}
                </button>
              </div>
            </form>
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  );
}
