"use client";

import { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { FiX } from "react-icons/fi";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { prSchema } from "@/validation/validation"; // Your Zod schema
import { toast } from "react-toastify";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { PRUpdate } from "@/lib/actions/entryActions";
import { useSession } from "next-auth/react";
import { useRouter } from "next/navigation";
import Select from "react-select";
export default function PRReviewModal({ isOpen, onClose, onCancel, chapterId, initialstatus }) {
    const { data: session, status } = useSession();
    const [loading, setLoading] = useState(false);
    const router = useRouter();
    const {
        register,
        handleSubmit,
        control,
        formState: { errors },
        reset,
    } = useForm({
        resolver: zodResolver(prSchema),
        mode: "onBlur",
        defaultValues: {
            entry_eda_comments : "",            
            entry_eda_end_date: "",
            entry_eda_status: "",
        },
    });

    const onSubmit = async (data) => {
        setLoading(true);
        try {
            const payload = {
                user_id: session.user.user_pid,
                chap_id: chapterId,
                status: data.entry_eda_status,
                end_date: data.entry_eda_end_date,
                comments : data.entry_eda_comments
            };
            const eda_res = await PRUpdate(payload);
            if (eda_res?.success) {
                toast.success(`PR status updated successfully!`);
                // Success logic: close modal and reset form
                reset();
                onClose();
                router.refresh();
            } else {
                toast.error(eda_res?.message || "Failed to PR updation.");
            }
        } catch (error) {
            toast.error(error.message || "Unexpected error occurred");
            console.error(error);
        } finally {
            setLoading(false);
        }
    };
    const statusOptions = initialstatus.map((s) => ({
        value: String(s.status_pid),
        label: s.status_name.charAt(0).toUpperCase() + s.status_name.slice(1),
    }));
    return (
        <AnimatePresence>
            {isOpen && (
                <motion.div
                    className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm"
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    exit={{ opacity: 0 }}
                >
                    <motion.div
                        className="bg-white rounded-lg shadow-xl w-full max-w-md overflow-hidden"
                        initial={{ scale: 0.9, opacity: 0 }}
                        animate={{ scale: 1, opacity: 1 }}
                        exit={{ scale: 0.9, opacity: 0 }}
                        transition={{ type: "spring", stiffness: 300, damping: 25 }}
                    >
                        {/* Header */}
                        <div className="flex items-center justify-between px-5 py-4 border-b border-blue-200 bg-blue-50">
                            <h3 className="text-lg font-semibold text-blue-800">
                                PR Status Update
                            </h3>
                            <button
                                onClick={() => {
                                    onClose();
                                    reset();
                                }}
                                className="text-gray-500 hover:text-gray-700 transition cursor-pointer"
                                aria-label="Close modal"
                            >
                                <FiX size={20} />
                            </button>
                        </div>

                        {/* Form (Wraps body & footer) */}
                        <form onSubmit={handleSubmit(onSubmit)}>
                            {/* Body */}
                            <div className="px-5 py-4 text-sm text-gray-700 space-y-3">
                                <div className="relative overflow border border-gray-200 rounded-md z-10">
                                    <table className="min-w-full table-auto border border-gray-100 text-sm">
                                        <tbody className="text-gray-700">
                                            <tr className="border-b border-gray-100">
                                                <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                                                    Comments
                                                </td>
                                                <td className="px-4 py-2">
                                                    <textarea
                                                        id="entry_eda_comments"
                                                        placeholder="Enter comments"
                                                        rows={4}
                                                        {...register("entry_eda_comments")}
                                                        className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.entry_eda_comments
                                                            ? "border-red-500"
                                                            : "border-gray-300"
                                                            } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                                                    />
                                                    {errors.entry_eda_comments && (
                                                        <p className="text-sm text-red-500 mt-1">
                                                            {errors.entry_eda_comments.message}
                                                        </p>
                                                    )}
                                                </td>
                                            </tr>
                                            <tr className="border-b border-gray-100">
                                                <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                                                    Date  <span className="text-red-500">*</span>
                                                </td>
                                                <td className="px-4 py-2">
                                                    <input
                                            id="entry_eda_end_date"
                                            type="date"
                                            // min={getTodayDate()}
                                            // max={getTodayDate()}
                                            placeholder="Select end date"
                                            {...register("entry_eda_end_date")}
                                            className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm bg-white text-gray-700 ${errors.entry_eda_end_date
                                                ? "border-red-500"
                                                : "border-gray-300"
                                                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                                        />
                                        {errors.entry_eda_end_date && (
                                            <p className="text-sm text-red-500 mt-1">
                                                {errors.entry_eda_end_date.message}
                                            </p>
                                        )}
                                                </td>
                                            </tr>
                                            <tr className="border-b border-gray-100">
                                                <td className="font-medium px-4 py-2 bg-gray-50 w-1/3 border-r border-gray-100 font-semibold">
                                                    Status  <span className="text-red-500">*</span>
                                                </td>
                                                <td className="px-4 py-2">
                                                    <Controller
                                                        name="entry_eda_status"
                                                        control={control}
                                                        render={({ field }) => (
                                                            <Select
                                                                {...field}
                                                                className="z-50 relative"
                                                                instanceId="entry-book-pid-select"
                                                                options={statusOptions}
                                                                onChange={(selected) =>
                                                                    field.onChange(selected ? selected.value : "")
                                                                }
                                                                value={
                                                                    statusOptions.find(
                                                                        (option) => option.value === field.value
                                                                    ) || null
                                                                }
                                                                placeholder="Select a staus"
                                                                isClearable
                                                                inputId="entry_eda_status"
                                                                menuPortalTarget={document.body}
                                                                styles={{
                                                                    control: (base) => ({
                                                                        ...base,
                                                                        borderColor: errors.entry_eda_status
                                                                            ? "red"
                                                                            : base.borderColor,
                                                                    }),
                                                                    menuPortal: (base) => ({
                                                                        ...base,
                                                                        zIndex: 9999, 
                                                                    }),
                                                                }}
                                                            />
                                                        )}
                                                    />
                                                    {errors.entry_eda_status && (
                                                        <p className="text-sm text-red-500 mt-1">
                                                            {errors.entry_eda_status.message}
                                                        </p>
                                                    )}
                                                </td>
                                            </tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            {/* Footer */}
                            <div className="flex justify-end gap-2 px-5 py-3 bg-gray-50 border-t border-gray-200">
                                <button
                                    type="button"
                                    onClick={() => {
                                        onClose();
                                        reset();
                                    }}
                                    className="w-full sm:w-auto px-5 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
                                >
                                    Cancel
                                </button>
                                <button
                                    type="submit"
                                    className="w-full sm:w-auto px-5 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2"
                                    disabled={loading}
                                >
                                    <span>{loading ? "Updating..." : "Submit"}</span>
                                    {loading ? (
                                        <FiLoader size={20} className="animate-spin" />
                                    ) : (
                                        <FiArrowRightCircle size={20} />
                                    )}
                                </button>
                            </div>
                        </form>
                    </motion.div>
                </motion.div>
            )}
        </AnimatePresence>
    );
}
