"use client"

import { useState, useEffect, useRef } from "react"
import Link from "next/link"
import { usePathname } from "next/navigation"
import { useSession, signOut } from "next-auth/react"
import Image from "next/image"
import { toast } from "react-toastify"
import { FiHome, FiBook, FiUser, FiBookOpen, FiCalendar, FiSettings, FiLock, FiBriefcase, FiEdit, FiFileText, FiMenu, FiLogOut, FiChevronUp, FiClock, FiUsers, FiChevronDown, FiEdit2, FiUserCheck, FiSearch, FiEdit3 ,FiCopy} from "react-icons/fi" // Added FiChevronDown for clarity
import { PiListMagnifyingGlass } from "react-icons/pi";
import {MdOutlineSettings} from "react-icons/md";

import { motion, AnimatePresence } from "framer-motion"

// Define the icon map once
const iconMap = {
    FiHome, FiBook, FiUser, FiBookOpen, FiCalendar, FiSettings, FiLock, FiBriefcase, FiEdit, FiFileText, FiMenu, FiClock, FiUsers, FiChevronDown, FiEdit2, FiUserCheck, FiSearch, FiEdit3, FiCopy,	
PiListMagnifyingGlass,MdOutlineSettings

};

// Recursive component for handling Submenus
const SubmenuItem = ({ subItem, pathname, isExpanded, level = 2 }) => {
    // const hasSubSubmenu = subItem.submenu && subItem.submenu.length > 0;
    // const [isSubExpanded, setIsSubExpanded] = useState(
    //     hasSubSubmenu && subItem.submenu.some(item => pathname.startsWith(item.href))
    // );
    // const isActive = hasSubSubmenu
    //     ? subItem.submenu.some(item => pathname.startsWith(item.href))
    //     : pathname.startsWith(subItem.href) && pathname !== subItem.href.replace('#', '');

    // // Classes for styling
    // const linkClasses = `block w-full px-3 py-1.5 text-sm rounded-md transition-colors ${isActive ? "text-white font-semibold bg-white/10" : "text-gray-300 hover:text-white hover:bg-white/5"
    //     }`;
     const hasSubSubmenu = subItem.submenu && subItem.submenu.length > 0;
    
        // THIRD-LEVEL: Active only when EXACT path matches
        const directActive = pathname === subItem.href;
    
        // SECOND-LEVEL: Active when any child matches exactly
        const childActive =
            hasSubSubmenu &&
            subItem.submenu.some(child => pathname === child.href);
    
        // Mark menu active (second OR third level)
        const isActive = directActive || childActive;
    
        // Auto-open only when a child is active
        const [isSubExpanded, setIsSubExpanded] = useState(childActive);
    
        const linkClasses = `
        block w-full px-3 py-1.5 text-sm rounded-md transition-colors
        ${isActive
                ? "text-white font-semibold bg-white/10"
                : "text-gray-300 hover:text-white hover:bg-white/5"
            }
    `;
    
    const padding = level === 2 ? 'pl-1' : 'pl-1'; // Adjust padding for deeper levels

        const content = (
            <span className="flex items-center justify-between">
                <span className="whitespace-nowrap">{subItem.label}</span>

                {hasSubSubmenu && (
                    isSubExpanded ? (
                        <FiChevronUp className="transition-transform duration-200" size={16} />
                    ) : (
                        <FiChevronDown className="transition-transform duration-200" size={16} />
                    )
                )}
            </span>
        );


    return (
        <li className={`py-1 ${padding}`}>
            {hasSubSubmenu ? (
                <button
                    type="button"
                    onClick={() => setIsSubExpanded(p => !p)}
                    className={`text-left ${linkClasses}`}
                >
                    {content}
                </button>
            ) : (
                // Only render Link if there's no submenu or it's just a placeholder '#'
                <Link href={subItem.href === '#' ? '#' : subItem.href} className={linkClasses}>
                    {content}
                </Link>
            )}

            <AnimatePresence>
                {isSubExpanded && hasSubSubmenu && isExpanded && (
                    <motion.ul
                        initial={{ height: 0, opacity: 0 }}
                        animate={{ height: "auto", opacity: 1 }}
                        exit={{ height: 0, opacity: 0 }}
                        transition={{ duration: 0.3, ease: "easeInOut" }}
                        className="ml-4 pl-3 border-l border-white/20 overflow-hidden"
                    >
                        {subItem.submenu.map(nestedItem => (
                            <SubmenuItem
                                key={nestedItem.id || nestedItem.href || nestedItem.label}
                                // key={nestedItem.href}
                                subItem={nestedItem}
                                pathname={pathname}
                                isExpanded={isExpanded}
                                level={level + 1} // Increase level for deeper nesting
                            />
                        ))}
                    </motion.ul>
                )}
            </AnimatePresence>
        </li>
    );
};

export default function SidebarClient({ links = [] }) {
    const [isExpanded, setIsExpanded] = useState(false);
    const [openMenu, setOpenMenu] = useState(null);
    const [profileDropdownOpen, setProfileDropdownOpen] = useState(false);
    const sidebarRef = useRef(null);
    const pathname = usePathname();
    const { data: session } = useSession();
    const user = session?.user;

    // Set CSS variable for sidebar width
    useEffect(() => {
        const sidebarWidth = isExpanded ? 240 : 80;
        document.body.style.setProperty('--sidebar-width', `${sidebarWidth}px`);
    }, [isExpanded]);

    // Open submenus if a child link is active
    useEffect(() => {
        links.forEach(({ label, submenu }) => {
            if (submenu) {
                // Check for activity in submenu or nested submenus
                const isAnySubItemActive = submenu.some((item) => {
                    const primaryMatch = pathname.startsWith(item.href) && item.href !== '#';
                    const tertiaryMatch = item.submenu?.some((subItem) => pathname.startsWith(subItem.href) && subItem.href !== '#');
                    return primaryMatch || tertiaryMatch;
                });

                if (isAnySubItemActive) {
                    setOpenMenu(label);
                }
            }
        });

        const handleClickOutside = (event) => {
            if (sidebarRef.current && !sidebarRef.current.contains(event.target)) {
                setProfileDropdownOpen(false);
            }
        };

        document.addEventListener("mousedown", handleClickOutside);
        return () => {
            document.removeEventListener("mousedown", handleClickOutside);
        };

    }, [pathname, links]);

    const toggleMenu = (label) => {
        setOpenMenu(prev => (prev === label ? null : label));
    };

    const handleLogout = async () => {
        try {
            fetch('/api/auth/logout', { method: 'POST' }).catch(console.error);
            toast.success('You have been logged out.');
            await signOut({ callbackUrl: '/' });
        } catch (error) {
            console.error("Logout error:", error);
            toast.error("An error occurred during logout.");
        }
    };

    const NavItem = ({ href, label, icon, submenu, isExpanded, onToggle, isOpen }) => {

        const Icon = icon ? iconMap[icon.trim()] : null;

        // Determine if the parent item itself is active (either directly or via a submenu)
        const isParentActive = submenu?.some(item => {
            const primaryMatch = pathname.startsWith(item.href) && item.href !== '#';
            const tertiaryMatch = item.submenu?.some(subItem => pathname.startsWith(subItem.href) && subItem.href !== '#');
            return primaryMatch || tertiaryMatch;
        });

        // Determine if a non-submenu link is active
        const isLinkActive = pathname === href && !submenu;

        const isActive = isLinkActive || isParentActive;

        const commonClasses = "flex items-center h-12 w-full rounded-lg transition-all duration-300 cursor-pointer overflow-hidden";
        const activeClasses = "bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] text-white shadow-lg shadow-pink-500/20";
        const inactiveClasses = "text-gray-200 hover:bg-white/10 hover:text-white";

        const content = (
            <>
                {/* {console.log(Icon, label, "sdfsd")} */}
                <div className="flex-shrink-0 w-12 h-12 flex items-center justify-center">
                    {Icon && <Icon size={22} />}
                </div>
                <AnimatePresence>
                    {isExpanded && (
                        <motion.span
                            initial={{ opacity: 0, x: -10 }}
                            animate={{ opacity: 1, x: 0 }}
                            exit={{ opacity: 0, x: -10 }}
                            transition={{ duration: 0.2, delay: 0.1 }}
                            className="flex-1 flex items-center justify-between whitespace-nowrap font-medium text-sm pr-4"
                        >
                            {label}
                            {submenu && <FiChevronDown className={`transition-transform duration-300 ${isOpen ? 'rotate-180' : ''}`} size={18} />}
                        </motion.span>
                    )}
                </AnimatePresence>
            </>
        );

        const Wrapper = submenu ? 'button' : Link;
        const wrapperProps = submenu
            ? { type: "button", onClick: onToggle }
            : { href };

        return (
            <div className="px-1">
                <Wrapper
                    {...wrapperProps}
                    className={`${commonClasses} ${isActive ? activeClasses : inactiveClasses}`}
                    style={{ height: "40px" }}
                >
                    {content}
                </Wrapper>
            </div>
        );
    };

    return (
        <motion.aside
            onMouseEnter={() => setIsExpanded(true)}
            onMouseLeave={() => setIsExpanded(false)}
            ref={sidebarRef}
            className="fixed top-1/2 left-3 -translate-y-1/2 flex flex-col h-[97vh] bg-gray-900/80 backdrop-blur-lg rounded-2xl shadow-2xl z-40 shadow"
            animate={{ width: isExpanded ? 240 : 80 }}
            transition={{ type: "spring", stiffness: 400, damping: 40 }}
        >
            {/* Logo */}
            <div className="flex-shrink-0 w-full h-18 flex items-center justify-center">
                <Image src={"/logo.png"} alt="Logo" width={120} height={36} className="object-cover p-3" />
            </div>

            {/* Navigation */}
            <nav className="flex-1 overflow-y-auto overflow-x-hidden hide-scrollbar">
                <ul className="space-y-1">
                    {links.map((link) => (
                        <li key={link.id}>
                            <NavItem
                                {...link}
                                isExpanded={isExpanded}
                                onToggle={() => toggleMenu(link.label)}
                                isOpen={openMenu === link.label} // Pass open state for chevron rotation
                            />
                            <AnimatePresence>
                                {isExpanded && openMenu === link.label && link.submenu && (
                                    <motion.ul
                                        initial={{ height: 0, opacity: 0 }}
                                        animate={{ height: "auto", opacity: 1 }}
                                        exit={{ height: 0, opacity: 0 }}
                                        transition={{ duration: 0.3, ease: "easeInOut" }}
                                        className="ml-8 pl-3 border-l border-white/20 overflow-hidden"
                                    >
                                        {link.submenu.map(subItem => (
                                            <SubmenuItem
                                                key={subItem.id || subItem.href || subItem.label}
                                                // key={subItem.href || subItem.label} 
                                                subItem={subItem}
                                                pathname={pathname}
                                                isExpanded={isExpanded}
                                            />
                                        ))}
                                    </motion.ul>
                                )}
                            </AnimatePresence>
                        </li>
                    ))}
                </ul>
            </nav>

            {/* User Profile */}
            <div className="flex-shrink-0 w-full h-20 flex items-center justify-center px-1">
                <div className="relative w-full group">
                    <AnimatePresence>
                        {isExpanded && profileDropdownOpen && (
                            <motion.div
                                initial={{ opacity: 0, y: 10 }}
                                animate={{ opacity: 1, y: 0 }}
                                exit={{ opacity: 0, y: 10 }}
                                transition={{ duration: 0.2 }}
                                className="absolute bottom-14 left-0 w-full p-2"
                            >
                                <div className="bg-gray-800 rounded-lg shadow-lg overflow-hidden">
                                    <Link href="/users/profile" onClick={() => setProfileDropdownOpen(false)} className="flex items-center gap-3 w-full text-left px-4 py-2.5 text-sm text-gray-200 hover:bg-white/10 cursor-pointer">
                                        <FiUser />
                                        <span>View Profile</span>
                                    </Link>
                                    <button onClick={handleLogout} className="flex items-center gap-3 w-full text-left px-4 py-2.5 text-sm text-gray-200 hover:bg-white/10 cursor-pointer">
                                        <FiLogOut />
                                        <span>Logout</span>
                                    </button>
                                </div>
                            </motion.div>
                        )}
                    </AnimatePresence>
                    <button
                        onClick={() => isExpanded && setProfileDropdownOpen(p => !p)}
                        className="flex items-center h-12 w-full rounded-lg text-gray-200 hover:bg-white/10 hover:text-white transition-colors cursor-pointer overflow-hidden"
                    >
                        <div className="flex-shrink-0 w-16 h-12 flex items-center justify-center">
                            {user?.profile ? (
                                <Image src={`/emp_profile/${user.profile}`} alt="Profile" width={36} height={36} className="rounded-full object-cover w-9 h-9" />
                            ) : (
                                <div className="w-9 h-9 bg-gray-500 rounded-full flex items-center justify-center">
                                    <span className="font-bold text-sm">{user?.name?.[0]?.toUpperCase() || "U"}</span>
                                </div>
                            )}
                        </div>
                        <AnimatePresence>
                            {isExpanded && (
                                <motion.div
                                    initial={{ opacity: 0, x: -10 }}
                                    animate={{ opacity: 1, x: 0 }}
                                    exit={{ opacity: 0, x: -10 }}
                                    transition={{ duration: 0.2, delay: 0.1 }}
                                    className="flex-1 flex items-center justify-between pr-4"
                                >
                                    <span className="whitespace-nowrap font-medium text-sm">
                                        {user?.name || "User"}
                                    </span>
                                    <FiChevronDown className={`transition-transform duration-300 ${profileDropdownOpen ? 'rotate-180' : ''}`} />
                                </motion.div>
                            )}
                        </AnimatePresence>
                    </button>
                </div>
            </div>
        </motion.aside>
    );
}