"use client";

import React, { useEffect, useState } from "react";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { userSchema } from "@/validation/validation";
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";
import Select from "react-select";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { createUserAction, updateUserAction } from "@/lib/actions/userActions";
import { useSession } from "next-auth/react";
import BackButton from "@/app/components/BackButton";
export default function CreateUserForm({ roles, user, userId }) {
  const { data: session, status } = useSession();
  const [loading, setLoading] = useState(false);
  const isEditMode = !!userId;
  const router = useRouter();
  const [profileImageUrl, setProfileImageUrl] = useState(null);
  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
    setValue,
  } = useForm({
    resolver: zodResolver(userSchema(isEditMode)),
    mode: "onBlur",
    defaultValues: {
      user_empid: "",
      user_password: "",
      user_name: "",
      user_con_pwd: "",
      user_email: "",
      user_role: "",
      user_profile: null,
    },
  });
  useEffect(() => {
    if (user) {
      setValue("user_empid", user.user_empid);
      setValue("user_name", user.user_name);
      setValue("user_email", user.user_email);
      setValue("user_role", String(user.user_role));
      if (user.user_profile) {
        setProfileImageUrl(`/emp_profile/${user.user_profile}`);
      }
    } else {
      reset();
    }
  }, [user, setValue, reset]);
  const onSubmit = async (data) => {
    setLoading(true);
    try {
      const formData = new FormData();
      formData.append("user_empid", data.user_empid);
      formData.append("user_name", data.user_name);
      formData.append("user_email", data.user_email);
      formData.append("user_role", data.user_role);
      formData.append("user_pid", session.user.user_pid);
      if (!isEditMode) {
        formData.append("user_password", data.user_password);
        formData.append("user_con_pwd", data.user_con_pwd);
      }
      if (data.user_profile && data.user_profile.length > 0) {
        formData.append("user_profile", data.user_profile[0]);
      }
      if (isEditMode) {
        console.log(data, "data in update");
        const res = await updateUserAction(userId, formData);
        if (res?.success) {
          router.push("/users/list");
          toast.success("User updated successfully!");
        } else {
          toast.error(res?.message || "Failed to create user.");
        }
      } else {
        const res = await createUserAction(formData);
        if (res?.success) {
          router.push("/users/list");
          toast.success("User created successfully!");
        } else {
          toast.error(res?.message || "Failed to create user.");
        }
      }
    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };
  const handleReset = () => {
    reset();
    const fileInput = document.getElementById("user_profile");
    if (fileInput) fileInput.value = "";
    toast.info("Form reset.");
  };
  const roleOptions = roles.map((role) => ({
    value: String(role.role_pid),
    label: role.role_name.charAt(0).toUpperCase() + role.role_name.slice(1),
  }));
  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg  space-y-4">
      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">
              {isEditMode ? "Edit User" : "Create New User"}
            </h2>
            <p className="text-gray-500 mt-1 text-sm">
              {isEditMode
                ? "Modify the details of the selected user."
                : "Fill in the details below to add a new user."}
            </p>
          </div>
          <BackButton/>
        </div>
      </header>
      <form
        onSubmit={handleSubmit(onSubmit)}
        className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl mx-auto"
      >
        {/* Left Column */}
        <div className="space-y-4 ">
          {/* Employee ID */}
          <div>
            <label
              htmlFor="user_empid"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              User ID <span className="text-red-500">*</span>
            </label>
            <input
              id="user_empid"
              type="text"
              placeholder="Enter user ID"
              {...register("user_empid")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.user_empid ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.user_empid && (
              <p className="text-sm text-red-500 mt-1">
                {errors.user_empid.message}
              </p>
            )}
          </div>
          {/* Full Name */}
          <div>
            <label
              htmlFor="user_name"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Full Name <span className="text-red-500">*</span>
            </label>
            <input
              id="user_name"
              type="text"
              placeholder="Enter full name"
              {...register("user_name")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.user_name ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.user_name && (
              <p className="text-sm text-red-500 mt-1">
                {errors.user_name.message}
              </p>
            )}
          </div>
          {/* Email */}
          <div>
            <label
              htmlFor="user_email"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Email Address <span className="text-red-500">*</span>
            </label>
            <input
              id="user_email"
              type="email"
              placeholder="Enter email address"
              {...register("user_email")}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.user_email ? "border-red-500" : "border-gray-300"
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.user_email && (
              <p className="text-sm text-red-500 mt-1">
                {errors.user_email.message}
              </p>
            )}
          </div>
          {/* User Role */}
          <div>
            <label
              htmlFor="user_role"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              User Role <span className="text-red-500">*</span>
            </label>
            <Controller
              name="user_role"
              control={control}
              render={({ field }) => (
                <Select
                  {...field}
                  instanceId="user-role-select"
                  options={roleOptions}
                  onChange={(selected) =>
                    field.onChange(selected ? selected.value : "")
                  }
                  value={
                    roleOptions.find(
                      (option) => option.value === field.value
                    ) || null
                  }
                  placeholder="Select a role"
                  isClearable
                  inputId="user_role"
                  styles={{
                    control: (base) => ({
                      ...base,
                      borderColor: errors.user_role
                        ? "red"
                        : base.borderColor,
                    }),
                  }}
                />
              )}
            />
            {errors.user_role && (
              <p className="text-sm text-red-500 mt-1">
                {errors.user_role.message}
              </p>
            )}
          </div>
        </div>
        {/* Right Column */}
        <div className="space-y-4">
          {/* Password & Confirm Password */}
          {!isEditMode && (
            <>
              <div>
                <label
                  htmlFor="user_password"
                  className="block text-sm font-medium text-gray-700 mb-1"
                >
                  Password <span className="text-red-500">*</span>
                </label>
                <input
                  id="user_password"
                  type="password"
                  placeholder="Enter password"
                  {...register("user_password")}
                  className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.user_password
                      ? "border-red-500"
                      : "border-gray-300"
                    } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                />
                {errors.user_password && (
                  <p className="text-sm text-red-500 mt-1">
                    {errors.user_password.message}
                  </p>
                )}
              </div>
              <div>
                <label
                  htmlFor="user_con_pwd"
                  className="block text-sm font-medium text-gray-700 mb-1"
                >
                  Confirm Password <span className="text-red-500">*</span>
                </label>
                <input
                  id="user_con_pwd"
                  type="password"
                  placeholder="Confirm password"
                  {...register("user_con_pwd")}
                  className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.user_con_pwd ? "border-red-500" : "border-gray-300"
                    } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                />
                {errors.user_con_pwd && (
                  <p className="text-sm text-red-500 mt-1">
                    {errors.user_con_pwd.message}
                  </p>
                )}
              </div>
            </>
          )}
          {/* Profile Picture */}
          <div>
            <label
              htmlFor="user_profile"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Profile Picture <span className="text-red-500"></span>{" "}
              (Allowed: png, jpg, jpeg)
            </label>
            {/* Styled file input */}
            <input
              type="file"
              id="user_profile"
              {...register("user_profile")}
              accept="image/png, image/jpeg, image/jpg"
              className={`block w-full text-sm text-gray-700
                  file:bg-blue-100 file:text-blue-700 file:rounded file:px-4 file:py-2 file:border-0
                  cursor-pointer
                  border rounded-md
                  ${errors.user_profile ? "border-red-500" : "border-gray-300"}
                  focus:outline-none focus:ring-2 focus:ring-blue-500
                `}
            />
            {errors.user_profile && (
              <p className="text-sm text-red-500 mt-1">
                {errors.user_profile.message}
              </p>
            )}
            {isEditMode && profileImageUrl && (
              <div className="mb-2">
                <img
                  src={profileImageUrl}
                  alt="Profile"
                  className="w-24 h-24 object-cover rounded-full border border-gray-300"
                />
              </div>
            )}
          </div>
        </div>
        {/* Buttons */}
        <div className="col-span-1 md:col-span-2 flex flex-col sm:flex-row justify-end gap-4">
          {!isEditMode && (
            <button
              type="button"
              onClick={handleReset}
              className="w-full sm:w-auto px-5 py-2 border border-red-600 text-red-600 rounded-md hover:bg-red-100 transition cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
            >
              Reset
            </button>
          )}
          <button
            type="submit"
            className="w-full sm:w-auto px-5 py-2  text-white rounded-md bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B]  transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2 d-flex justify-center"
          >
            <span>
              {loading
                ? isEditMode
                  ? "Updating..."
                  : "Creating..."
                : isEditMode
                  ? "Update User"
                  : "Create User"}
            </span>
            {loading ? (
              <FiLoader size={20} className="animate-spin" />
            ) : (
              <FiArrowRightCircle size={20} />
            )}
          </button>
        </div>
      </form>
    </div>
  );
}
