"use client";

import { useMemo, useState, useRef } from "react";
import Select from "react-select";
import { createUserPerformanceReportAction, alluserperformanceReportList } from "@/lib/actions/entryActions"; 
import * as XLSX from "xlsx";
import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";
import Papa from "papaparse";
import Swal from "sweetalert2";
import { Bar, Doughnut   } from "react-chartjs-2";
import { Chart as ChartJS, CategoryScale, LinearScale, BarElement, ArcElement, Title,  Tooltip, Legend } from "chart.js";
import ChartDataLabels from 'chartjs-plugin-datalabels';
ChartJS.register(ChartDataLabels);
ChartJS.register(CategoryScale, LinearScale, BarElement, ArcElement, Title, Tooltip, Legend);

export default function UserPerformanceList({ initialUsers = [], role, username }) {
    const [filteredGrouped, setFilteredGrouped] = useState([]);
    const [loading, setLoading] = useState(false);

    // filters
    const [startDate, setStartDate] = useState("");
    const [endDate, setEndDate] = useState("");
    const [selectedUser, setSelectedUser] = useState(null);
    const [selectcsvreport, setFilterCsvGrouped] = useState(null);
    const [reportTitle, setReportTitle] = useState("User Performance Reports");
    const [selectedOption, setSelectedOption] = useState(null);
    const [selectedDonutOption, setSelectedDonutOption] = useState(null);
    const chartRef = useRef(null);
    const donutRef = useRef(null);
    

    const UserOptions = useMemo(
    () =>
      (initialUsers || []).map((u) => ({
        value: String(u.user_pid),
        label: u.user_name?.charAt(0).toUpperCase() + u.user_name?.slice(1),
      })),
        [initialUsers]
    );

    const formatTimeDisplay = (t) => {
        if (!t && t !== 0) return "-";
        if (typeof t === "string") return t.split(".")[0];
        if (t instanceof Date) return t.toTimeString().split(" ")[0];
        return String(t);
    };

    const options = [
        { value: "png", label: "Download PNG" },
        { value: "jpg", label: "Download JPG" },
    ];

    const handleImageChange = (option) => {
        setSelectedOption(option);
        handleDownloadChartImage(option.value);
    };

    const handleDonutImageChange = (option) => {
        setSelectedDonutOption(option);
        handleDownloadDonutChartImage(option.value);
    };

    const handleChange = async (field, value) => {
        setLoading(true);

        // update local state
        if (field === "startDate") setStartDate(value);
        if (field === "endDate") setEndDate(value);
        if (field === "user") setSelectedUser(value);
        
        let UserName = username;
        if(role == 1 || role == 2)
        {
            UserName = null;
        }
        // prepare payload for API call
        const payload = {
            user_pid: field === "user" ? value?.value : selectedUser?.value || UserName,
            start_date: field === "startDate" ? value : startDate || null,
            end_date: field === "endDate" ? value : endDate || null,
        };

        if (payload.start_date && payload.end_date) {
            const startDate = new Date(payload.start_date);
            const endDate = new Date(payload.end_date);
    
            const diffTime = Math.abs(endDate - startDate);
            const totalDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    
            if (totalDays > 31) {
                Swal.fire({
                //icon: "warning",
                title: "⚠️ Date Range Too Long",
                text: "The selected date range exceeds 31 days. Please choose a shorter range.",
                confirmButtonText: "Got it",
                confirmButtonColor: "#7C3AED", // purple accent
                background: "#F9FAFB", // light gray background
                color: "#1F2937", // dark text
                width: 500,
                padding: "1.5rem",
                showClass: {
                    popup: "animate__animated animate__fadeInDown"
                },
                hideClass: {
                    popup: "animate__animated animate__fadeOutUp"
                }
                });
                setLoading(false);
                setEndDate('');
                return; // stop further execution
            }
        } else if (!payload.start_date && payload.end_date) {
            Swal.fire({
                title: "Start Date Empty",
                text: "Please select a start date before choosing an end date.",
                confirmButtonText: "OK"
            });
            setLoading(false);
            setEndDate(''); // clear invalid end date
            return; // stop further execution
        } 

        try {
            const resq = await createUserPerformanceReportAction(payload);
            const res  = resq?.userperformancereport;
            if (res && Array.isArray(res)) {
                // group data by user_name
                const groups = Object.values(
                res.reduce((acc, row) => {
                    const user = row.user_name ?? "Unknown user";
                    if (!acc[user]) acc[user] = { user_name: user, tasks: [] };
                    acc[user].tasks.push({
                    task_name: row.task_name ?? "-",
                    avg_time: row.avg_time ?? row.avg_time ?? "-",
                    count: Number(row.task_count ?? 0),
                    });
                    return acc;
                }, {})
                );

                groups.forEach((g) => {
                g.total = g.tasks.reduce((s, t) => s + (Number(t.count) || 0), 0);
                });

                const reportlist = await alluserperformanceReportList(payload);
                const csvReport = reportlist?.userperformancereportlist?.reduce((acc, item) => {
                    let group = acc.find(g => g.user_name === item.user_name);
                    if (!group) {
                        group = { user_name: item.user_name, tasks: [] };
                        acc.push(group);
                    }
                    group.tasks.push({
                        task_name: item.task_name,
                        entry_duration: item.entry_duration,
                        count: item.task_count,
                    });
                    return acc;
                }, []);

                // Helper to format date as MM-DD-YYYY
                const formatDate = (dateString) => {
                const date = new Date(dateString);
                if (isNaN(date)) return "-"; // handle invalid date
                    const mm = String(date.getMonth() + 1).padStart(2, "0");
                    const dd = String(date.getDate()).padStart(2, "0");
                    const yyyy = date.getFullYear();
                    return `${dd}-${mm}-${yyyy}`;
                };
                
                let title = "User Performance Reports";

                if (payload.start_date && payload.end_date) {
                    const startDate = new Date(payload.start_date);
                    const endDate = new Date(payload.end_date);

                    // Calculate difference in days
                    const diffTime = Math.abs(endDate - startDate);
                    const totalDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

                    // If days > 25 → show Month, Year format
                    if (totalDays > 25) {
                        const monthName = endDate.toLocaleString("default", { month: "long" });
                        const year = endDate.getFullYear();
                        title = `User Performance Reports - ${monthName}, ${year}`;
                    } else {
                        // Otherwise show full range
                        title = `User Performance Reports - ${formatDate(payload.start_date)} to ${formatDate(payload.end_date)}`;
                    }
                } 
                else if (payload.user_pid && (field === "user" ? value?.label : selectedUser?.label)) {
                    const userLabel = field === "user" ? value?.label : selectedUser?.label;
                    title = `User Performance Report - ${userLabel}`;
                }

                setFilteredGrouped(groups);
                setFilterCsvGrouped(csvReport);
                setReportTitle(title);
            } else {
                setFilteredGrouped([]);
                setFilterCsvGrouped();
                setReportTitle();
            }
        } catch (err) {
            console.error("Error fetching User Performance report:", err);
            setFilteredGrouped([]);
        } finally {
            setLoading(false);
        }
    }
    
    const grandTotal = filteredGrouped.reduce((s, g) => s + (g.total || 0), 0);

    const formatDate = (dateString) => {
        const date = new Date(dateString);
        if (isNaN(date)) return "-"; // handle invalid date
        const mm = String(date.getMonth() + 1).padStart(2, "0");
        const dd = String(date.getDate()).padStart(2, "0");
        const yyyy = date.getFullYear();
        return `${dd}-${mm}-${yyyy}`;
    };

    const handleDownloadChartImage = (format = "png") => {
        const chartCanvas = chartRef.current?.canvas;
        if (!chartCanvas) return;

        const scale = 3;
        const tempCanvas = document.createElement("canvas");
        tempCanvas.width = chartCanvas.width * scale;
        tempCanvas.height = chartCanvas.height * scale;

        const ctx = tempCanvas.getContext("2d");

        if (format === "png" || format === "jpg") {
            ctx.fillStyle = "#ffffff";
            ctx.fillRect(0, 0, tempCanvas.width, tempCanvas.height);
        }

        ctx.scale(scale, scale);
        ctx.drawImage(chartCanvas, 0, 0);

        let image, filename;

        if (format === "png") {
            image = tempCanvas.toDataURL("image/png", 1.0);
            filename = "UserPerformanceChart.png";
        } else if (format === "jpg") {
            image = tempCanvas.toDataURL("image/jpeg", 1.0);
            filename = "UserPerformanceChart.jpg";
        } else if (format === "svg") {
            const chart = chartRef.current;
            if (chart && chart.toBase64Image) {
                image = chart.toBase64Image("image/svg+xml");
                filename = "UserPerformanceChart.svg";
            }
        }

        if (image) {
            const link = document.createElement("a");
            link.href = image;
            link.download = filename;
            link.click();
        }

    }

    const handleDownloadDonutChartImage = (format = "png") => {
        const donutCanvas = donutRef.current?.canvas;
        if (!donutCanvas) return;

        const scale = 3;
        const tempCanvas = document.createElement("canvas");
        tempCanvas.width = donutCanvas.width * scale;
        tempCanvas.height = donutCanvas.height * scale;

        const ctx = tempCanvas.getContext("2d");

        if (format === "png" || format === "jpg") {
            ctx.fillStyle = "#ffffff";
            ctx.fillRect(0, 0, tempCanvas.width, tempCanvas.height);
        }

        ctx.scale(scale, scale);
        ctx.drawImage(donutCanvas, 0, 0);

        let image, filename;

        if (format === "png") {
            image = tempCanvas.toDataURL("image/png", 1.0);
            filename = "UserPerformanceDonutChart.png";
        } else if (format === "jpg") {
            image = tempCanvas.toDataURL("image/jpeg", 1.0);
            filename = "UserPerformanceDonutChart.jpg";
        } else if (format === "svg") {
            const chart = donutRef.current;
            if (chart && chart.toBase64Image) {
                image = chart.toBase64Image("image/svg+xml");
                filename = "UserPerformanceDonutChart.svg";
            }
        }

        if (image) {
            const link = document.createElement("a");
            link.href = image;
            link.download = filename;
            link.click();
        }
    }

    // Excel Export
    const handleExportExcel = () => {
        const ws_data = [["", "", `User Performance Reports`],
                ["Owned By", "Task Name", "Avg. Time", "Total"],
        ];
        filteredGrouped.forEach(group => {
            group.tasks.forEach((task, idx) => {
            ws_data.push([
                idx === 0 ? group.user_name.charAt(0).toUpperCase() + group.user_name.slice(1) : "",
                task.task_name,
                formatTimeDisplay(task.avg_time),
                `${task.count}`,
            ]);
            });
            ws_data.push([
                "", "Total", "", `${group.total}`
            ]);
        });
        ws_data.push(["Grand Total", "", "", `${grandTotal}`]);
        const ws = XLSX.utils.aoa_to_sheet(ws_data);
        const wb = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(wb, ws, "Report");
        XLSX.writeFile(wb, "User Performance Report.xlsx");
    };

    const handleExportPDF = async () => {
        if (typeof window === "undefined") return;

        const doc = new jsPDF();
        const pageWidth = doc.internal.pageSize.getWidth();
        const imageurl = "/logo-new.png";
        doc.addImage(imageurl, "PNG", 12, 5, 40, 12);
        
        doc.setFontSize(13);
        const titleY = 12;

        let title = "User Performance Reports";

       const drawHeader = (docInstance) => {
            docInstance.addImage(imageurl, "PNG", 15, 2, 35, 15);
            docInstance.setFontSize(13);
            docInstance.setTextColor(40);

            if (startDate && endDate) {
                const start = new Date(startDate);
                const end = new Date(endDate);
                const diffTime = Math.abs(end - start);
                const totalDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

                const startMonth= start.getMonth();
                const endMonth  = end.getMonth();
                const startYear = start.getFullYear();
                const endYear   = end.getFullYear();

                if (startMonth === endMonth && startYear === endYear && totalDays >= 25) {
                    const monthName = endDate.toLocaleString("default", { month: "long" });
                    title = `User Performance Reports - ${monthName}, ${endYear}`;
                } 
                else {
                    title = `User Performance Reports - ${formatDate(start)} to ${formatDate(end)}`;
                }
            } else if (selectedBook?.label) {
            title = `User Performance Report - ${selectedBook.label}`;
            }

            docInstance.text(title, pageWidth / 2, titleY, { align: "center" });
            docInstance.setLineWidth(0.1);
            docInstance.line(10, 18, pageWidth - 10, 18);
        };

        drawHeader(doc);

        const tableStartY = titleY + 8;
        const body = [];

        filteredGrouped.forEach(group => {
            group.tasks.forEach((task, idx) => {
                body.push([
                    idx === 0 ? group.user_name.charAt(0).toUpperCase() + group.user_name.slice(1) : "",
                    task.task_name,
                    formatTimeDisplay(task.avg_time),
                    task.count.toString(),
                ]);
            });

            body.push([
                "",
                { content: "Total", styles: { fontStyle: "bold", halign: "left" } },
                "",
                { content: group.total.toString(), styles: { fontStyle: "bold", halign: "left" } },
            ]);
        });

        body.push([
            { content: "Grand Total", colSpan: 3, styles: { fontStyle: "bold", halign: "left" } },
            { content: grandTotal.toString(), styles: { fontStyle: "bold", halign: "left" } },
        ]);

        autoTable(doc, {
            head: [["Owned By", "Task Name", "Avg. Time", "Total"]],
            body,
            startY: tableStartY,
            theme: "grid",
            styles: {
                fontSize: 8,
                cellPadding: 1,
                halign: "left",
            },
            headStyles: {
                fillColor: [200, 200, 200],
                textColor: [0, 0, 0],
                fontStyle: "bold",
            },
            didParseCell: function (data) {
                if (data.column.index === 0 && data.cell.raw === "") {
                    data.cell.styles.lineWidth = { top: 0, bottom: 0, left: 0.1, right: 0.1 };
                }
            },
            didDrawPage: (data) => {
                drawHeader(doc);
            }
        });

        doc.addPage();
        drawHeader(doc);

        doc.setFontSize(12);
        const chartCanvas = document.querySelector("#barChartContainer canvas");
        if (chartCanvas) {
            const scale = 3; 

            const tempCanvas = document.createElement("canvas");
            tempCanvas.width = chartCanvas.width * scale;
            tempCanvas.height = chartCanvas.height * scale;
            const ctx = tempCanvas.getContext("2d");
            ctx.scale(scale, scale);
            ctx.drawImage(chartCanvas, 0, 0);

            const chartImg = tempCanvas.toDataURL("image/png", 1.0);
            const chartWidth = pageWidth - 20;
            const chartHeight = (chartCanvas.height * chartWidth) / chartCanvas.width;

            doc.addImage(chartImg, "PNG", 10, 30, chartWidth, chartHeight);

        }  
        const date = new Date().toISOString().split("T")[0];
        doc.save(`user_performance_${date}.pdf`);
    };


    // CSV Export
    const handleExportCSV = () => {
        const data = [];
        selectcsvreport.forEach(group => {
            group.tasks.forEach((task, idx) => {
            data.push({
                "Owned By":  group.user_name.charAt(0).toUpperCase() + group.user_name.slice(1),
                "Task Name": task.task_name,
                "Avg. Time": `${formatTimeDisplay(task.entry_duration)}`,
                Count: `${task.count}`,
            });
            });
        });

        const csv = Papa.unparse(data);
        const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
        const link = document.createElement("a");
        link.href = URL.createObjectURL(blob);
        link.setAttribute("download", "user_perforance_report.csv");
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    };

    const allTaskNames = Array.from(
        new Set(filteredGrouped.flatMap((group) => group.tasks.map((t) => t.task_name)))
    );

    const totalBars = filteredGrouped.length * allTaskNames.length;
    const dynamicThickness = totalBars > 30 ? 10 : totalBars > 20 ? 20 : 30;

    const datasets = allTaskNames.map((taskName, idx) => {
        const colors = [
            "#e03c4cff",
            "#c23057ff",
            "#c28131ff",
            "#658336ff",
            "#229f71ff",
            "#5e1e9eff",
            "#e3bb39ff",
        ];

        const avgTimeData = filteredGrouped.map((group) => {
            const task = group.tasks.find((t) => t.task_name === taskName);
            return task ? task.avg_time : ""; // formatter
        });
        return {
            label: taskName,
            data: filteredGrouped.map((group) => {
                const task = group.tasks.find((t) => t.task_name === taskName);
                return task ? task.count : 0;
            }),
            avgTime: avgTimeData,
            backgroundColor: colors[idx % colors.length],
            borderRadius: 6,
            barThickness: dynamicThickness,   // ✅ dynamic size
            maxBarThickness: dynamicThickness,
        };
    });

    const barData = {
        labels: filteredGrouped.map((group) => group.user_name.charAt(0).toUpperCase() + group.user_name.slice(1)),
        datasets: datasets,
    };

   
    const allUserNames = Array.from(
        new Set(filteredGrouped.flatMap((group) =>  group.user_name))
    );

    const timeToSeconds = (timeStr) => {
        if (!timeStr || timeStr.split(':').length !== 3) return 0;
        const parts = timeStr.split(':').map(Number);
        return parts[0] * 3600 + parts[1] * 60 + parts[2];
    };
    const secondsToTime = (totalSeconds) => {
        if (totalSeconds < 0) totalSeconds = 0;
        const hours = Math.floor(totalSeconds / 3600);
        const minutes = Math.floor((totalSeconds % 3600) / 60);
        const seconds = Math.floor(totalSeconds % 60);
        const pad = (num) => String(num).padStart(2, '0');
        return `${pad(hours)}:${pad(minutes)}:${pad(seconds)}`;
    };

    const Donutdatasets = allUserNames.map((userName) => {
        const userTotals = filteredGrouped.reduce((currentTotals, group) => {
            if (group.user_name === userName) {
                const groupTaskTotals = group.tasks.reduce((taskSum, taskItem) => {
                    const count = taskItem?.count || 0;
                    const timeStr = taskItem?.avg_time || "00:00:00";
                    const time = timeToSeconds(timeStr);
                    
                    taskSum.totalCount += count;
                    taskSum.totalTime += time; 

                    return taskSum;
                }, { totalCount: 0, totalTime: 0 }); 

                currentTotals.totalCount += groupTaskTotals.totalCount;
                currentTotals.totalTime += groupTaskTotals.totalTime;

                return currentTotals;
            }
            return currentTotals;
        }, { totalCount: 0, totalTime: 0 });

        const grandAvgTime = userTotals.totalCount > 0
            ? userTotals.totalTime
            : 0;

        const grandAvgTimeString = secondsToTime(grandAvgTime);
        return { 
            label: userName, 
            data: [userTotals.totalCount],
            averageTime: grandAvgTimeString,
        };
    });

    const dataValues = Donutdatasets.map(dataset => dataset.data[0]);
    const dataAvgValues = Donutdatasets.map(dataset => dataset.averageTime);
    const labels = Donutdatasets.map(dataset => 
        dataset.label.charAt(0).toUpperCase() + dataset.label.slice(1)
    );

    const backgroundColor = [
        "#ff6384",
        "#36a2eb",
        "#ffce56",
        "#4bc0c0",
    ];

    const DonutData = {
        labels: labels, 
        datasets: [
            {
                label: "Task Count",
                data: dataValues, 
                averageTimes: dataAvgValues,
                backgroundColor: backgroundColor.slice(0, dataValues.length), 
                borderWidth: 2,
            }
        ]
    };

    const barOptions = {
        responsive: true,
        maintainAspectRatio: false,
        devicePixelRatio: 3,
        plugins: {
            legend: {
                position: "bottom",
            },
            title: {
                display: true,
                text: "Graphical Representation of User Performance Report",
                font: { size: 16, weight: "bold" },
            },
            datalabels: {
                display: true,
                color: "#000",
                anchor: "end",
                align: "top",
                offset: 4,
                font: { size: 11, weight: "bold" },
                formatter: (value) => (value > 0 ? value : ""), // Show only non-zero values
            },
            tooltip: { 
                callbacks: {
                    label: function(context) {
                        let lines = [];
        
                        lines.push(`${context.dataset.label || ''}: ${context.parsed.y}`);

                        const avgTimeArray = context.dataset.avgTime;
                        const avgTime = avgTimeArray ? avgTimeArray[context.dataIndex] : null;

                        if (avgTime !== null) {
                            lines.push(`Avg Time: ${avgTime}`); 
                        }
            
                        return lines; 
                    },
                }
            },
        },
        scales: {
            x: {
                title: {
                display: true,
                text: "Employee Name",
                font: { size: 14, weight: "bold"},
                },
            },
            y: {
                beginAtZero: true,
                title: {
                display: true,
                text: "Task Count",
                font: { size: 14, weight: "bold" },
                },
                ticks: { stepSize: 1 },
            },
        },
    };


 const donutOptions = {
    cutout: "60%",     // donut hole size
    responsive: true,
    maintainAspectRatio: false,
    devicePixelRatio: 3,
    plugins: {
        legend: {
            position: "bottom",
        },
        title: {
            display: true,
            text: "Graphical Representation of User Performance Report",
            font: { size: 16, weight: "bold" },
        },
        datalabels: {
            display: true,
            color: "#000",
            font: { size: 11, weight: "bold" },
            formatter: (value) => (value > 0 ? value : ""), // Show only non-zero values
        },
        tooltip: {
            callbacks: {
                label: function(context) {
                    let lines = [];

                    let label = context.dataset.label || '';
                    if (label) {
                        label += ': ';
                    }
                    if (context.parsed !== null) {
                        label += context.parsed;
                    }
                    lines.push(label);

                    const averageTimes = context.dataset.averageTimes;
                    const dataIndex = context.dataIndex;

                    if (averageTimes && averageTimes[dataIndex]) {
                        lines.push(`Avg Time: ${averageTimes[dataIndex]}`); 
                    }

                    return lines;
                },
            },
        },
    },
    
  };


    const selectStyles = { menuPortal: (base) => ({ ...base, zIndex: 9999 }) }
    return (
       <div className="max-w-full px-2 py-2 mx-auto rounded-lg space-y-4">
            <header>
                <div className="flex justify-between items-center border-b border-purple-500">
                    <div className="mb-1">
                        <h2 className="text-2xl font-semibold text-gray-800">User Performance Report</h2>
                        <p className="text-gray-500 mt-1 text-sm">
                            Below is a list of all user performance metrics.
                        </p>
                    </div>
                </div>
            </header>

            <div className="bg-purple-50 border border-purple-200 rounded-lg p-2 shadow-sm">
                <div className="flex flex-wrap gap-4 items-center justify-start">
                    {/* Start Date */}
                    <div className="flex flex-col">
                        <label className="text-xs text-gray-600 font-semibold mb-1">
                        Start Date
                        </label>
                        <input
                            type="date"
                            className="w-33 border border-gray-300 rounded px-3 py-2 text-sm focus:ring focus:ring-purple-200"
                            value={startDate}
                            onChange={(e) => handleChange("startDate", e.target.value)}
                            max={endDate}
                        />
                    </div>

                    {/* End Date */}
                    <div className="flex flex-col">
                        <label className="text-xs text-gray-600 font-semibold mb-1">
                        End Date
                        </label>
                        <input
                            type="date"
                            className="w-33 border border-gray-300 rounded px-3 py-2 text-sm focus:ring focus:ring-purple-200"
                            value={endDate}
                            onChange={(e) => handleChange("endDate", e.target.value)}
                            min={startDate}
                        />
                    </div>
                    {(role === 1 || role === 2) && (
                        <div className="flex flex-col">
                            {/* User Filter */}
                            <label className="text-xs text-gray-600 font-semibold mb-1">
                            User
                            </label>
                            <Select
                            instanceId="user-filter"
                            options={UserOptions}
                            value={selectedUser}
                            onChange={(v) => handleChange("user", v)}
                            placeholder="Select User"
                            isClearable
                            className="w-33 text-sm"
                            menuPortalTarget={document.body}
                            styles={selectStyles}
                            />
                        </div>
                    )}

                    {/* Clear Filters Button */}
                    <div className="pt-5">
                        <button
                            onClick={() => {
                                setSelectedUser(null);
                                setStartDate("");
                                setEndDate("");
                                setFilteredGrouped([]);
                            }}
                        className="text-sm px-4 py-2 bg-purple-100 hover:bg-purple-200 text-purple-700 rounded transition-all duration-200 cursor-pointer"
                        >
                        Clear Filters
                        </button>
                    </div>
                </div>
            </div>
            {filteredGrouped.length > 0 && (
                <header>
                    <h2 className="text-2xl font-semibold text-gray-800 text-center">{reportTitle}</h2>
                </header>
            )}
            {/* Table */}
            {filteredGrouped.length > 0 && (
                <div className="overflow-x-auto">
                    <table className="min-w-full bg-white border border-gray-200 rounded-lg shadow-sm">
                        <thead>
                            <tr className="bg-gray-100 text-sm">
                            <th className="py-2 px-4 border-b text-left w-1/3">Owned By</th>
                            <th className="py-2 px-4 border-b text-left w-1/3">Task Name</th>
                            <th className="py-2 px-4 border-b text-left w-1/6">Avg. Time</th>
                            <th className="py-2 px-4 border-b text-left w-1/6">Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            {loading ? (
                            <tr>
                                <td colSpan={4} className="text-center p-6">
                                Loading...
                                </td>
                            </tr>
                            ) : filteredGrouped.length === 0 ? (
                            <tr>
                                <td colSpan={4} className="text-center p-6 text-gray-500">
                                No data found
                                </td>
                            </tr>
                            ) : (
                            filteredGrouped.map((group, gi) => {
                                const rowSpan = group.tasks.length + 1;
                                return group.tasks.map((task, ti) => (
                                <tr key={`${gi}-${ti}`} className={ti % 2 === 0 ? "bg-white" : "bg-gray-50"}>
                                    {ti === 0 && (
                                    <td
                                        rowSpan={rowSpan}
                                        className="py-2 px-4 border-l border-r border-b align-top font-medium text-gray-700 w-1/3"
                                    >
                                        {group.user_name.charAt(0).toUpperCase() + group.user_name.slice(1)}
                                    </td>
                                    )}
                                    <td className="py-2 px-4 border-b border-r w-1/3">{task.task_name}</td>
                                    <td className="py-2 px-4 border-b border-r text-left font-mono w-1/6">
                                    {task.avg_time}
                                    </td>
                                    <td className="py-2 px-4 border-b border-r text-left font-semibold w-1/6">{task.count}</td>
                                </tr>
                                )).concat(
                                <tr key={`${gi}-total`} className="bg-gray-200/50">
                                    <td className="py-1 px-4 border-b border-r text-left font-medium">Total</td>
                                    <td className="py-1 px-4 border-b border-r text-left font-mono"></td>
                                    <td className="py-1 px-4 border-b border-r text-left font-semibold">{group.total}</td>
                                </tr>
                                );
                            })
                            )}
                        </tbody>
                        {filteredGrouped.length > 0 && (
                            <tbody>
                                <tr className="bg-gray-200/50">
                                    <td className="py-1 px-4 border-l border-b border-r text-left font-medium">Grand Total</td>
                                    <td className="py-1 px-4 border-b border-r text-center font-mono"></td>
                                    <td className="py-1 px-4 border-b border-r text-center font-mono"></td>
                                    <td className="py-1 px-4 border-b border-r text-left font-semibold">{grandTotal}</td>
                                </tr>
                            </tbody>
                        )}
                    </table>
                </div>
            )}
            {/* Export Buttons at Bottom */}
            {filteredGrouped.length > 0 && (
                <div className="flex gap-2 mt-6 justify-end">
                <button
                    onClick={handleExportExcel}
                    className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
                >
                    Excel
                </button>
                <button
                    onClick={handleExportPDF}
                    className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
                >
                    Export PDF
                </button>
                <button
                   onClick={handleExportCSV}
                    className="bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white px-5 py-1.5 cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
                >
                    Download CSV
                </button>
                </div>
            )} 
            {/* Pie Chart */}
            {filteredGrouped.length > 0 && (
                <>
                    <div
                        id="barChartContainer"  // 👈 required for pdf capture
                        className="bg-purple-50 border border-purple-200 rounded-lg p-4 shadow-sm flex flex-wrap gap-4 items-center justify-start"
                    >
                        <div className="w-full max-w-4xl mx-auto">
                            <div className="relative h-[400px]">
                                <Bar ref={chartRef} data={barData} options={barOptions} />
                            </div>

                            {/* ⬇️ Download Button (centered below the chart) */}
                            <div className="relative flex justify-end float-right  mt-2 mb-5 w-50">
                                <Select
                                    value={selectedOption}
                                    onChange={handleImageChange}
                                    options={options}
                                    placeholder="Download Image"
                                    menuPortalTarget={document.body} // ensures dropdown overlays correctly
                                    styles={{
                                        control: (base, state) => ({
                                            ...base,
                                            cursor: "pointer",
                                            boxShadow: "none",
                                            borderColor: state.isFocused ? "#4CAF50" : "#2196F3", 
                                            "&:hover": { borderColor: "#4CAF50" },
                                            background: "linear-gradient(to right, #cd1a41ff, #e1e448ff)", // green → blue gradient
                                            color: "#fff",
                                        }),
                                        menuPortal: (base) => ({
                                        ...base,
                                        zIndex: 9999, // ensures dropdown is on top
                                        }),
                                        placeholder: (base) => ({
                                        ...base,
                                        color: "white",
                                        fontWeight: 500,
                                        }),
                                        singleValue: (base) => ({
                                        ...base,
                                        color: "white",
                                        }),
                                    }}
                                    className="w-full"
                                    classNamePrefix="react-select"
                                />
                            </div>
                        </div>
                    </div>
                    <div id="donutChartContainer" className="bg-purple-50 border border-purple-200 rounded-lg p-4 shadow-sm flex flex-wrap gap-4 items-center justify-start">
                        <div className="w-full max-w-4xl mx-auto">
                            <div className="relative h-[400px]">
                                <Doughnut ref={donutRef} data={DonutData} options={donutOptions} /> 
                            </div>
                        
                            {/* ⬇️ Download Button (centered below the chart) */}
                            <div className="relative flex justify-end float-right  mt-2 mb-5 w-50">
                                <Select
                                    value={selectedDonutOption}
                                    onChange={handleDonutImageChange}
                                    options={options}
                                    placeholder="Download Image"
                                    menuPortalTarget={document.body} // ensures dropdown overlays correctly
                                    styles={{
                                        control: (base, state) => ({
                                            ...base,
                                            cursor: "pointer",
                                            boxShadow: "none",
                                            borderColor: state.isFocused ? "#4CAF50" : "#2196F3", 
                                            "&:hover": { borderColor: "#4CAF50" },
                                            background: "linear-gradient(to right, #cd1a41ff, #e1e448ff)", // green → blue gradient
                                            color: "#fff",
                                        }),
                                        menuPortal: (base) => ({
                                        ...base,
                                        zIndex: 9999, // ensures dropdown is on top
                                        }),
                                        placeholder: (base) => ({
                                        ...base,
                                        color: "white",
                                        fontWeight: 500,
                                        }),
                                        singleValue: (base) => ({
                                        ...base,
                                        color: "white",
                                        }),
                                    }}
                                    className="w-full"
                                    classNamePrefix="react-select"
                                />
                            </div>
                        </div>
                    </div>
        
                </>
            )} 
        </div>
    );
};