"use client";

import React, { useEffect, useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { knowledgeSchema } from '@/validation/validation';
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";
import { FiArrowRightCircle, FiDownload, FiLoader } from "react-icons/fi";
import { createKnowledgeAction, updateKnowledgeAction } from "@/lib/actions/knowledgeActions";
import BackButton from "@/app/components/BackButton";

export default function CreateUserForm({ knowledge, knowledgeId, user_pid }) {
 
  const [loading, setLoading] = useState(false);
  const [knowledgeFileUrl, setKnowledgeFileUrl] = useState(null);
  const isEditMode = !!knowledgeId;
  const router = useRouter();

  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
    setValue,
  } = useForm({
    resolver: zodResolver(knowledgeSchema(isEditMode)),
    mode: 'onBlur',
    defaultValues: {
      knowledge_name: '',
      knowledge_file: null,
    },
  });

  useEffect(() => {
    if (knowledge) {
      setValue('knowledge_name', knowledge.knowledge_name);

      if (knowledge.knowledge_file) {
        setKnowledgeFileUrl(`/Knowledge/${knowledge.knowledge_file}`);
      }
    } else {
      reset();
    }
  }, [knowledge, setValue, reset]);


  const onSubmit = async (data) => {
    setLoading(true);
    try {
      const formData = new FormData();
      formData.append('knowledge_name', data.knowledge_name);
      formData.append('knowledge_createdBy',user_pid);

      if (data.knowledge_file && data.knowledge_file.length > 0) {
        formData.append('knowledge_file', data.knowledge_file[0]);
      }
      if (isEditMode) {
        console.log(data, "data in update");
        const res = await updateKnowledgeAction(knowledgeId, formData);
        if (res?.success) {
          router.push("/knowledge/list");
          toast.success("Knowledge updated successfully!");
        } else {
          toast.error(res?.message || "Failed to update Knowledge.");
        }

      } else {
        const res = await createKnowledgeAction(formData);
        if (res?.success) {
          router.push("/knowledge/list");
          toast.success("Knowledge created successfully!");
        } else {
          toast.error(res?.message || "Failed to create Knowledge.");
        }

      }

    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const handleReset = () => {
    reset();
    const fileInput = document.getElementById('knowledge_file');
    if (fileInput) fileInput.value = '';
    toast.info('Form reset.');
  };


  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg  space-y-4">
     
       <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
             <h2 className="text-2xl font-semibold text-gray-800">
          {isEditMode ? 'Edit Knowledge Base' : 'Create New Knowledge Base'}
        </h2>
        <p className="text-gray-500 mt-1 text-sm">
          {isEditMode
            ? 'Modify the details of the selected knowledge base.'
            : 'Fill in the details below to add a new knowledge base.'}
        </p>
          </div>
          <BackButton/>
        </div>
      </header>

      <form onSubmit={handleSubmit(onSubmit)} className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl mx-auto">
        {/* Left Column */}
        <div className="space-y-4">
          {/* Knowledge Name */}
          <div>
            <label htmlFor="knowledge_name" className="block text-sm font-medium text-gray-700 mb-1">
              Knowledge Name <span className="text-red-500">*</span>
            </label>
            <input
              id="knowledge_name"
              type="text"
              placeholder="Enter knowledge name"
              {...register('knowledge_name')}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.knowledge_name ? 'border-red-500' : 'border-gray-300'
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.knowledge_name && (
              <p className="text-sm text-red-500 mt-1">{errors.knowledge_name.message}</p>
            )}
          </div>


        </div>

        {/* Right Column */}
        <div className="space-y-4">


          {/* Knowledge File */}

          <div>
            <label
              htmlFor="knowledge_file"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
               Knowledge File <span className="text-red-500">*</span>
               <span className="text-xs text-gray-500 ml-1">(png, jpg, pdf, doc, docx)</span>
            </label>
             <div className="flex items-center gap-4">
            <input
              type="file"
              id="knowledge_file"
              {...register('knowledge_file')}
                 accept="image/png, image/jpeg, image/jpg, application/pdf, application/msword, application/vnd.openxmlformats-officedocument.wordprocessingml.document"
              className={`block w-full text-sm text-gray-700
                    file:bg-blue-100 file:text-blue-700 file:rounded file:px-4 file:py-2 file:border-0
                    cursor-pointer
                    border rounded-md
                    ${errors.knowledge_file ? 'border-red-500' : 'border-gray-300'}
                   focus:outline-none focus:ring-2 focus:ring-purple-500
                 `}
               />
               {isEditMode && knowledgeFileUrl && (
                 <a
                   href={knowledgeFileUrl}
                   download
                   target="_blank"
                   rel="noopener noreferrer"
                   className="flex-shrink-0 inline-flex items-center px-3 py-2 bg-indigo-600 text-white font-semibold rounded-lg shadow-md hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-400 focus:ring-opacity-75 transition duration-300"
                   aria-label="Download file"
                 >
                   <FiDownload className="w-5 h-5" />
                 </a>
               )}
             </div>
            {errors.knowledge_file && (
              <p className="text-sm text-red-500 mt-1">{errors.knowledge_file.message}</p>
            )}           
          </div>

        </div>

        {/* Buttons */}
        <div className="col-span-1 md:col-span-2 flex flex-col sm:flex-row justify-end gap-4">
          {!isEditMode && (
            <button
              type="button"
              onClick={handleReset}
              className="w-full sm:w-auto px-5 py-2 border border-red-600 text-red-600 rounded-md hover:bg-red-100 transition cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
            >
              Reset
            </button>
          )}
          <button
            type="submit"
            className="w-full sm:w-auto px-5 py-2 bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white rounded-md  transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2 d-flex justify-center"
          >
            <span>{loading ? (isEditMode ? 'Updating...' : 'Creating...') : isEditMode ? 'Update Knowledge ' : 'Create Knowledge'}</span>
            {loading ? (
              <FiLoader size={20} className="animate-spin" />
            ) : (
              <FiArrowRightCircle size={20} />
            )}
          </button>
        </div>
      </form>
    </div>
  );
}
