"use client";

import React, { useEffect, useState } from "react";
import { useForm, Controller } from 'react-hook-form';
import { zodResolver } from "@hookform/resolvers/zod";
import { holidaySchema } from '@/validation/validation';
import { toast } from "react-toastify";
import Select from 'react-select';
import { useRouter } from "next/navigation";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { createHolidayAction, updateHolidayAction } from "@/lib/actions/holidayActions";
import BackButton from "@/app/components/BackButton";

export default function CreateHolidayForm({ holiday, holidayId, user_pid }) {
  const [loading, setLoading] = useState(false);
  const isEditMode = !!holidayId;
  const router = useRouter();

  const {
    register,
    handleSubmit,
    control,
    formState: { errors },
    reset,
    setValue,
  } = useForm({
    resolver: zodResolver(holidaySchema),
    mode: 'onBlur',
    defaultValues: {
      holiday_date: '',
      holiday_name: '',
      holiday_type: '',
    },
  });

  const formatDate = (dateString) => {
    if (!dateString) return "";
    return new Date(dateString).toISOString().split("T")[0];
  };

  // Hydrate form with book data if editing
  useEffect(() => {
    if (holiday) {
      setValue('holiday_date', formatDate(holiday.holiday_date));
      setValue('holiday_name', holiday.holiday_name);
      setValue('holiday_type', holiday.holiday_OfficeType);
    } else {
      reset();
    }
  }, [holiday, setValue, reset]);

  const onSubmit = async (data) => {
    setLoading(true);
    try {
      const payload = {
        holiday_date: data.holiday_date,
        holiday_name: data.holiday_name,
        holiday_OfficeType: data.holiday_type,
        holiday_createdBy: user_pid
      };
      if (isEditMode) {
        console.log(data, "data in update");
        const res = await updateHolidayAction(holidayId, payload);
        if (res?.success) {
          router.push("/holidays/list");
          toast.success("Holiday updated successfully!");
        } else {
          toast.error(res?.message || "Failed to update holiday.");
        }

      } else {
        const res = await createHolidayAction(payload);
        if (res?.success) {
          router.push("/holidays/list");
          toast.success("Holiday created successfully!");
        } else {
          toast.error(res?.message || "Failed to create holiday.");
        }

      }

    } catch (error) {
      toast.error(error.message || "Unexpected error occurred");
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const handleReset = () => {
    reset();
    toast.info("Form reset.");
  };

  const holidayTypeOptions = [
    { value: 'PDMR', label: 'PDMR' },
    { value: 'Compuscript', label: 'Compuscript' },
  ];

  return (
    <div className="max-w-full px-2 py-2 mx-auto rounded-lg  space-y-4">

      <header>
        <div className="flex justify-between items-center border-b border-purple-500">
          <div className="mb-1">
            <h2 className="text-2xl font-semibold text-gray-800">
              {isEditMode ? 'Edit Holiday' : 'Create New Holiday'}
            </h2>
            <p className="text-gray-500 mt-1 text-sm">
              {isEditMode
                ? 'Modify the details of the selected holiday.'
                : 'Fill in the details below to add a new holiday.'}
            </p>
          </div>
          <BackButton/>
        </div>
      </header>

      <form onSubmit={handleSubmit(onSubmit)} className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl mx-auto">
        {/* Left Column */}
        <div className="space-y-4">
          {/* Holiday Date */}
          <div>
            <label htmlFor="holiday_date" className="block text-sm font-medium text-gray-700 mb-1">
              Holiday Date <span className="text-red-500">*</span>
            </label>
            <input
              id="holiday_date"
              type="date"
              placeholder="Enter holiday date"
              {...register('holiday_date')}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.holiday_date ? 'border-red-500' : 'border-gray-300'
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.holiday_date && (
              <p className="text-sm text-red-500 mt-1">{errors.holiday_date.message}</p>
            )}
          </div>

          {/* Holiday Name */}
          <div>
            <label htmlFor="holiday_name" className="block text-sm font-medium text-gray-700 mb-1">
              Holiday Name <span className="text-red-500">*</span>
            </label>
            <input
              id="holiday_name"
              type="text"
              placeholder="Enter holiday name"
              {...register('holiday_name')}
              className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.holiday_name ? 'border-red-500' : 'border-gray-300'
                } focus:outline-none focus:ring-2 focus:ring-blue-500`}
            />
            {errors.holiday_name && (
              <p className="text-sm text-red-500 mt-1">{errors.holiday_name.message}</p>
            )}
          </div>
        </div>

        {/* Right Column */}
        <div className="space-y-4">
          {/* Holiday Type Dropdown */}
          <div>
            <label htmlFor="holiday_type" className="block text-sm font-medium text-gray-700 mb-1">
              Holiday Type <span className="text-red-500">*</span>
            </label>
            <Controller
              name="holiday_type"
              control={control}
              render={({ field }) => (
                <Select
                  {...field}
                  options={holidayTypeOptions}
                  placeholder="Select holiday type"
                  isClearable
                  classNamePrefix="react-select"
                  value={holidayTypeOptions.find((opt) => opt.value === field.value) || null}
                  onChange={(selected) => field.onChange(selected ? selected.value : '')}
                />
              )}
            />
            {errors.holiday_type && (
              <p className="text-sm text-red-500 mt-1">{errors.holiday_type.message}</p>
            )}
          </div>
        </div>

        {/* Buttons */}
        <div className="col-span-1 md:col-span-2 flex flex-col sm:flex-row justify-end gap-4">
          {!isEditMode && (
            <button
              type="button"
              onClick={handleReset}
              className="w-full sm:w-auto px-5 py-2 border border-red-600 text-red-600 rounded-md hover:bg-red-100 transition cursor-pointer rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
            >
              Reset
            </button>
          )}
          <button
            type="submit"
            className="w-full sm:w-auto px-5 py-2 bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B] text-white rounded-md  transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2 d-flex justify-center"
          >
            <span>
              {loading ? (isEditMode ? 'Updating...' : 'Creating...') : isEditMode ? 'Update Holiday' : 'Create Holiday'}
            </span>
            {loading ? (
              <FiLoader size={20} className="animate-spin" />
            ) : (
              <FiArrowRightCircle size={20} />
            )}
          </button>
        </div>
      </form>
    </div>
  );
}

