"use client";

import React, { useState, useEffect } from "react";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useRouter } from "next/navigation";
import { entryAllUpdateSchema } from "@/validation/validation";
import Select from "react-select";
import { FiArrowRightCircle, FiLoader } from "react-icons/fi";
import { toast } from "react-toastify";
import { updateFinalAllEntryAction } from "@/lib/actions/entryActions";


export default function CreateEntryEditForm({ initialBooks, initialchapters, initialtasks, initialstatus, initialemails, entry }) {
    const [loading, setLoading] = useState(false);
    const [entryId, setEntryId] = useState(null);
    const router = useRouter();
    
    function getSchemaByButtonType() {
        return entryAllUpdateSchema;
    }
    
    const {
        register,
        handleSubmit,
        control,
        formState: { errors },
        reset,
        watch,
        setValue,
    } = useForm({
        resolver: zodResolver(getSchemaByButtonType()),
        mode: "onBlur",
        defaultValues: {
            entry_book_pid: "",
            entry_chapter_pid: "",
            entry_task_pid: "",
            entry_start_date: "",
            entry_start_time: "",
            entry_end_date: "",
            entry_end_time: "",
            entry_comments: "",
            entry_contact_email: "",
            entry_status: "",
            entry_upload_file: null,
        },
    });

    const formatDate = (dateString) => {
        if (!dateString) return "";
        return new Date(dateString).toISOString().split("T")[0];
    };

    useEffect(() => {
        setValue("entry_book_pid", String(entry.entry_bookId));
        setValue("entry_chapter_pid", String(entry.entry_chapterId));
        setValue("entry_task_pid", String(entry.entry_taskId));
        setValue("entry_comments", entry.entry_comments);
        setValue("entry_status", String(entry.entry_status));
        setValue("entry_start_date", formatDate(entry.entry_date));
        setValue("entry_start_time", String(entry.entry_start_time));
        setValue("entry_end_date", formatDate(entry.entry_end_date));
        setValue("entry_end_time", String(entry.entry_end_time));
        setValue("entry_contact_email", String(entry.entry_email));
        setEntryId(entry.entry_pid);
    })

    const bookOptions = initialBooks.map((b) => ({
        value: String(b.book_pid),
        label: b.book_title.charAt(0).toUpperCase() + b.book_title.slice(1),
    }));

    const chapterOptions = initialchapters.map((c) => ({
        value: String(c.chap_pid),
        label: c.chap_title.charAt(0).toUpperCase() + c.chap_title.slice(1),
    }));

    const taskOptions = initialtasks.map((t) => ({
        value: String(t.task_pid),
        label: t.task_name.charAt(0).toUpperCase() + t.task_name.slice(1),
    }));

    const statusOptions = initialstatus.map((s) => ({
        value: String(s.status_pid),
        label: s.status_name.charAt(0).toUpperCase() + s.status_name.slice(1),
    }));

    const emailOptions = initialemails.map((e) => ({
        value: String(e.user_pid),
        label: e.user_email.charAt(0).toUpperCase() + e.user_email.slice(1),
    }));

    const onSubmit = async (data) => {
        setLoading(true);
        try {
            const formData = new FormData();
            formData.append("entry_comments", data.entry_comments);
            formData.append("entry_start_date", data.entry_start_date);
            formData.append("entry_start_time", data.entry_start_time);
            formData.append("entry_end_date", data.entry_end_date);
            formData.append("entry_end_time", data.entry_end_time);
            formData.append("entry_contact_email", data.entry_contact_email);
            if (data.entry_upload_file && data.entry_upload_file.length > 0) {
                formData.append("entry_upload_file", data.entry_upload_file[0]);
            }

            const Final_result = await updateFinalAllEntryAction(entryId, formData);
            console.log(Final_result?.success, "Final_result");
            if (Final_result?.success) {
                toast.success("Work Entry updated successfully!");
                router.push("/edit_entrys/lists");
            } else {
                toast.error(Final_result?.message || "Failed to update entry.");
            }

        } catch (error) {
            toast.error(error.message || "Unexpected error occurred");
            console.error(error);
        } finally {
            setLoading(false);
        }
    }

    return (
        <div>
            <div className="max-w-full h-screen px-4 py-4 mx-auto bg-white rounded-lg shadow space-y-4 overflow-x-auto" style={{ height: '100%' }}>
                <header>
                    <div className="flex justify-between items-center border-b border-purple-500">
                        <div className="mb-1">
                            <h2 className="text-xl font-semibold text-gray-800"> Edit Entry </h2>
                            <p className="text-gray-500 mt-1 text-sm"> Modify the details of the selected entry. </p>
                        </div>
                    </div>
                </header>

                <div style={{ position: "relative" }}>
                    <form  onSubmit={handleSubmit(onSubmit)} className="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-4xl mx-auto">
                        <div className="space-y-4">
                            {/* Book Title*/}
                            <div>
                                <label htmlFor="entry_book_pid" className="block text-sm font-medium text-gray-700 mb-1">
                                Book Title <span className="text-red-500">*</span>
                                </label>
                                <Controller
                                    name="entry_book_pid"
                                    control={control}
                                    render={({ field }) => (
                                        <Select
                                            {...field}
                                            instanceId="entry-book-pid-select"
                                            options={bookOptions}
                                            onChange={(selected) =>
                                                field.onChange(selected ? selected.value : "")
                                            }
                                            value={
                                                bookOptions.find(
                                                (option) => option.value === field.value
                                                ) || null
                                            }
                                            isDisabled={true}  
                                            placeholder="Select a Book"
                                            isClearable
                                            inputId="entry_book_pid"
                                            styles={{
                                                control: (base) => ({
                                                ...base,
                                                borderColor: errors.entry_book_pid
                                                    ? "red"
                                                    : base.borderColor,
                                                }),
                                            }}
                                        />
                                    )}
                                />
                                {errors.entry_book_pid && (
                                <p className="text-sm text-red-500 mt-1">
                                    {errors.entry_book_pid.message}
                                </p>
                                )}
                            </div>

                            {/* Chapter Name*/}
                            <div>
                                <label htmlFor="entry_chapter_pid" className="block text-sm font-medium text-gray-700 mb-1" >
                                    Chapter Name <span className="text-red-500">*</span>
                                </label>
                                <Controller
                                    name="entry_chapter_pid"
                                    control={control}
                                    render={({ field }) => (
                                        <Select
                                            {...field}
                                            instanceId="entry-chapter-pid-select"
                                            options={chapterOptions}
                                            onChange={(selected) =>
                                                field.onChange(selected ? selected.value : "")
                                            }
                                            value={
                                                chapterOptions.find(
                                                (option) => option.value === field.value
                                                ) || null
                                            }
                                            isDisabled={true} 
                                            placeholder="Select a chapter"
                                            isClearable
                                            inputId="entry_chapter_pid"
                                            styles={{
                                                control: (base) => ({
                                                ...base,
                                                borderColor: errors.entry_chapter_pid
                                                    ? "red"
                                                    : base.borderColor,
                                                }),
                                            }}
                                        />
                                    )}
                                />
                                {errors.entry_chapter_pid && (
                                <p className="text-sm text-red-500 mt-1">
                                    {errors.entry_chapter_pid.message}
                                </p>
                                )}
                            </div>

                            {/* Task Name*/}
                            <div>
                                <label htmlFor="entry_task_pid" className="block text-sm font-medium text-gray-700 mb-1" >
                                    Task Name <span className="text-red-500">*</span>
                                </label>
                                <Controller
                                    name="entry_task_pid"
                                    control={control}
                                    render={({ field }) => (
                                        <Select
                                            {...field}
                                            instanceId="entry-task-pid-select"
                                            options={taskOptions}
                                            onChange={(selected) =>
                                                field.onChange(selected ? selected.value : "")
                                            }
                                            value={
                                                taskOptions.find(
                                                (option) => option.value === field.value
                                                ) || null
                                            }
                                            isDisabled={true} 
                                            placeholder="Select a task"
                                            isClearable
                                            inputId="entry_task_pid"
                                            styles={{
                                                control: (base) => ({
                                                ...base,
                                                borderColor: errors.entry_task_pid
                                                    ? "red"
                                                    : base.borderColor,
                                                }),
                                            }}
                                        />
                                    )}
                                />
                                {errors.entry_task_pid && (
                                    <p className="text-sm text-red-500 mt-1">
                                        {errors.entry_task_pid.message}
                                    </p>
                                )}
                            </div>
                            <div>
                                <label htmlFor="entry_comments" className="block text-sm font-medium text-gray-700 mb-1"
                                >
                                    Comments <span className="text-red-500">*</span>
                                </label>
                                <textarea
                                    id="entry_comments"
                                    placeholder="Enter comments"
                                    rows={4} // you can change the number of rows as needed
                                    {...register("entry_comments")}
                                    className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm ${errors.entry_comments ? "border-red-500" : "border-gray-300"
                                    } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                                />
                                {errors.entry_comments && (
                                    <p className="text-sm text-red-500 mt-1">
                                        {errors.entry_comments.message}
                                    </p>
                                )}
                            </div>
                            <div>
                            <label htmlFor="entry_status" className="block text-sm font-medium text-gray-700 mb-1">
                                Status<span className="text-red-500">*</span>
                            </label>
                            <Controller
                                name="entry_status"
                                control={control}
                                render={({ field }) => (
                                    <Select
                                        {...field}
                                        instanceId="entry-task-pid-select"
                                        options={statusOptions}
                                        onChange={(selected) =>
                                        field.onChange(selected ? selected.value : "")
                                        }
                                        value={
                                        statusOptions.find(
                                            (option) => option.value === field.value
                                        ) || null
                                        }
                                        isDisabled={true} 
                                        placeholder="Select a Status"
                                        isClearable
                                        inputId="entry_status"
                                        styles={{
                                        control: (base) => ({
                                            ...base,
                                            borderColor: errors.entry_status
                                            ? "red"
                                            : base.borderColor,
                                        }),
                                        }}
                                    />
                                )}
                            />
                            {errors.entry_status && (
                                <p className="text-sm text-red-500 mt-1">
                                    {errors.entry_status.message}
                                </p>
                            )}
                            </div>
                        </div>


                        <div className="space-y-4">
                            {/* Start Time and Start Button in Same Row */}
                            <div>
                                <label htmlFor="entry_start_date" className="block text-sm font-medium text-gray-700 mb-1">
                                    Start Date <span className="text-red-500">*</span>
                                </label>
                                <input
                                    id="entry_start_date"
                                    type="date"
                                    placeholder="Select start date"
                                    {...register("entry_start_date")}
                                    className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm bg-white text-gray-700 ${errors.entry_start_date
                                        ? "border-red-500"
                                        : "border-gray-300"
                                        } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                                />
                                {errors.entry_start_date && (
                                <p className="text-sm text-red-500 mt-1">
                                    {errors.entry_start_date.message}
                                </p>
                                )}
                            </div>
                            <div className="flex flex-col md:flex-row items-start md:items-end gap-4 w-full">
                                <div className="flex-1">
                                    <label htmlFor="entry_start_time" className="block text-sm font-medium text-gray-700 mb-1">
                                        Start Time <span className="text-red-500">*</span>
                                    </label>
                                    <input
                                        id="entry_start_time"
                                        type="text"
                                        placeholder="Select start time"
                                        {...register("entry_start_time")}
                                        className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm text-gray-700 ${errors.entry_start_time
                                            ? "border-red-500"
                                            : "border-gray-300"
                                            } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                                    />
                                    {errors.entry_start_time && (
                                        <p className="text-sm text-red-500 mt-1">
                                            {errors.entry_start_time.message}
                                        </p>
                                    )}
                                </div>    
                            </div>
                            <div>
                                <label htmlFor="entry_end_date" className="block text-sm font-medium text-gray-700 mb-1">
                                    End Date <span className="text-red-500">*</span>
                                </label>
                                <input
                                    id="entry_end_date"
                                    type="date"
                                    placeholder="Select end date"
                                    {...register("entry_end_date")}
                                    className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm bg-white text-gray-700 ${errors.entry_end_date
                                        ? "border-red-500"
                                        : "border-gray-300"
                                        } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                                />
                                {errors.entry_end_date && (
                                <p className="text-sm text-red-500 mt-1">
                                    {errors.entry_end_date.message}
                                </p>
                                )}
                            </div>
                            <div className="flex flex-col md:flex-row items-start md:items-end gap-4 w-full">
                                <div className="flex-1">
                                    <label htmlFor="entry_end_time" className="block text-sm font-medium text-gray-700 mb-1">
                                        End Time <span className="text-red-500">*</span>
                                    </label>
                                    <input
                                        id="entry_end_time"
                                        type="text"
                                        placeholder="Select end time"
                                        {...register("entry_end_time")}
                                        className={`w-full px-4 py-1.5 rounded-md border placeholder:text-sm text-gray-700 ${errors.entry_end_time
                                        ? "border-red-500"
                                        : "border-gray-300"
                                        } focus:outline-none focus:ring-2 focus:ring-blue-500`}
                                    />
                                    {errors.entry_end_time && (
                                        <p className="text-sm text-red-500 mt-1">
                                            {errors.entry_end_time.message}
                                        </p>
                                    )}
                                </div>
                            </div>
                            <div>
                                <label htmlFor="entry_contact_email" className="block text-sm font-medium text-gray-700 mb-1">
                                    Contact Email <span className="text-red-500"></span>
                                </label>
                                <Controller
                                    name="entry_contact_email"
                                    control={control}
                                    render={({ field }) => (
                                        <Select
                                            {...field}
                                            instanceId="entry-book-pid-select"
                                            options={emailOptions}
                                            onChange={(selected) =>
                                                field.onChange(selected ? selected.value : "")
                                            }
                                            value={
                                                emailOptions.find(
                                                (option) => option.value === field.value
                                                ) || null
                                            }
                                            isDisabled={true} 
                                            placeholder="Select a email"
                                            isClearable
                                            inputId="entry_contact_email"
                                            styles={{
                                                control: (base) => ({
                                                ...base,
                                                borderColor: errors.entry_contact_email
                                                    ? "red"
                                                    : base.borderColor,
                                                }),
                                            }}
                                        />
                                    )}
                                />
                                {errors.entry_contact_email && (
                                    <p className="text-sm text-red-500 mt-1">
                                        {errors.entry_contact_email.message}
                                    </p>
                                )}
                            </div>
                            <div>
                                <label htmlFor="entry_upload_file" className="block text-sm font-medium text-gray-700 mb-1" >
                                    Upload File <span className="text-red-500"></span> (Allowed:
                                     pdf, doc, docx)
                                </label>
                                <input
                                    type="file"
                                    id="entry_upload_file"
                                    disabled={true}
                                    {...register("entry_upload_file")}
                                    accept=".pdf,.docx,application/pdf,application/vnd.openxmlformats-officedocument.wordprocessingml.document"
                                    className={`block w-full text-sm text-gray-700                        file:bg-blue-100 file:text-blue-700 file:rounded file:px-4 file:py-2 file:border-0 cursor-pointer border rounded-md ${errors.entry_upload_file ? "border-red-500" : "border-gray-300" } focus:outline-none focus:ring-2 focus:ring-blue-50`}
                                />
                                {errors.entry_upload_file && (
                                    <p className="text-sm text-red-500 mt-1">
                                        {errors.entry_upload_file.message}
                                    </p>
                                )}
                            </div>
                        </div>
                        <div className="col-span-1 md:col-span-2 flex flex-col sm:flex-row justify-end gap-4">
                            <button
                                type="submit"
                                className="w-full sm:w-auto px-5 py-2 text-white rounded-md bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] hover:from-[#6B3DC6] hover:to-[#C64A9B]transition shadow-sm cursor-pointer flex items-center gap-2 rounded-md shadow-md text-sm font-medium transition-transform transform hover:scale-105 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2"
                                disabled={loading}
                            >
                                <span>{loading ? "Updating..." : "Update"}</span>
                                {loading ? (
                                <FiLoader size={20} className="animate-spin" />
                                ) : (
                                <FiArrowRightCircle size={20} />
                                )}
                            </button>
                        </div>
                    </form>
                </div>
            </div>   
        </div>
    );
}