import { useMemo, useState } from "react";
import { FiEdit, FiSave, FiX } from "react-icons/fi";
import { Tooltip } from "react-tooltip";
import { updateFinalAllPendingEntryAction } from "@/lib/actions/entryActions";
import { toast } from "react-toastify";
import { useRouter } from "next/navigation";

export default function ChildPendingDataTable({ pendings = [] }) {
    const [editRowIndex, setEditRowIndex] = useState(null);
    const [editedData, setEditedData] = useState({});
    const router = useRouter();

    const handleEditClick = (index, row) => {
        setEditRowIndex(index);
        setEditedData({ ...row });
    };

    const handleCancel = () => {
        setEditRowIndex(null);
        setEditedData({});
    };

    const handleSave = async () => {
        const pendingId = editedData.pr_pid;
        const pendingEntryId = editedData.pr_entry_pid;
        const pendingPayload = {
            Pending_date: editedData.pr_pendingDate,
            Pending_time: editedData.pr_pendingTime,
            Resume_date: editedData.pr_resumeDate,
            Resume_time: editedData.pr_resumeTime,
        };

        const Final_result = await updateFinalAllPendingEntryAction(pendingId, pendingEntryId, pendingPayload);
        if (Final_result?.success) {
            toast.success("Pending Entry updated successfully!");
            router.push("/edit_entrys/lists");
        } else {
            toast.error(Final_result?.message || "Failed to update entry.");
        }
        
        // Add your save logic here, like an API call or state update.
        setEditRowIndex(null);
    };

    const handleChange = (field, value) => {
        setEditedData((prev) => ({
            ...prev,
            [field]: value,
        }));
    };

    const columns = useMemo(
        () => [
            {
                label: 'S.No',
                selector: (_row, index) => index + 1,
            },
            {
                label: "Pending Date",
                key: "pr_pendingDate",
            },
            {
                label: "Pending Time",
                key: "pr_pendingTime",
            },
            {
                label: "Resume Date",
                key: "pr_resumeDate",
            },
            {
                label: "Resume Time",
                key: "pr_resumeTime",
            },
            {
                label: "Duration",
                key: "pr_duration",
            },
            {
                label: "Reason",
                key: "pr_pendingReason",
            },
            {
                label: "Action",
                cell: (row, index) => {
                    const isEditing = editRowIndex === index;
                    return (
                        <div className="flex space-x-2 text-blue-600 items-center">
                            {isEditing ? (
                                <>
                                    <button
                                        className="text-green-600 hover:text-green-800"
                                        onClick={handleSave}
                                        type="button"
                                    >
                                        <FiSave size={18} />
                                    </button>
                                    <button
                                        className="text-red-600 hover:text-red-800"
                                        onClick={handleCancel}
                                        type="button"
                                    >
                                        <FiX size={18} />
                                    </button>
                                </>
                            ) : (
                                <>
                                    <button
                                        data-tooltip-id={`edit-tooltip-${row?.pr_pid}`}
                                        data-tooltip-content="Edit"
                                        className="hover:text-blue-800"
                                        onClick={() => handleEditClick(index, row)}
                                        type="button"
                                    >
                                        <FiEdit size={18} />
                                    </button>
                                    <Tooltip
                                        id={`edit-tooltip-${row?.pr_pid}`}
                                        place="top"
                                        style={{
                                            backgroundColor: "#3b82f6",
                                            color: "#fff",
                                            fontSize: "12px",
                                            borderRadius: "4px",
                                            padding: "6px 10px",
                                             zIndex: "1000",
                                        }}
                                    />
                                </>
                            )}
                        </div>
                    );
                },
            },
        ],
        [editRowIndex, editedData, handleSave, router ]
    );

    if (!pendings.length) {
        return (
            <div className="p-4 text-center text-gray-500">
                No Pendings found for this book.
            </div>
        );
    }

    return (
        <div className="bg-blue-50 rounded-lg shadow-inner mt-4 border border-blue-100 overflow-x-auto">
            <table className="min-w-full">
                <thead>
                    <tr className="bg-pink-50">
                        {columns.map((col, i) => (
                            <th
                                key={i}
                                className="px-6 py-3 text-left text-xs font-bold text-black uppercase tracking-wider"
                            >
                                {col.label}
                            </th>
                        ))}
                    </tr>
                </thead>
                <tbody>
                    {pendings.map((row, rowIndex) => (
                        <tr key={rowIndex} className="even:bg-gray-50">
                            {columns.map((col, colIndex) => {
                                // If it's a custom cell like Action
                                if (col.cell) {
                                    return (
                                        <td key={colIndex} className="px-6 py-3 text-sm">
                                            {col.cell(row, rowIndex)}
                                        </td>
                                    );
                                }

                                const isEditing = editRowIndex === rowIndex;
                                const fieldKey = col.key;
                                // const value = isEditing
                                //     ? editedData[fieldKey]
                                //     : row[fieldKey];

                                let value;
                                if (col.selector) {
                                    value = col.selector(row, rowIndex);
                                } else {
                                    value = isEditing ? editedData[fieldKey] : row[fieldKey];
                                }

                                return (
                                    <td
                                        key={colIndex}
                                        className="px-6 py-3 text-sm text-gray-800 whitespace-pre-wrap"
                                    >
                                        {isEditing && fieldKey && fieldKey !== "pr_duration" && fieldKey !== "pr_pendingReason" ? (
                                            <input
                                                type="text"
                                                value={value || ""}
                                                onChange={(e) =>
                                                    handleChange(fieldKey, e.target.value)
                                                }
                                                className="border rounded px-2 py-1 text-sm w-full"
                                            />
                                        ) : (
                                            value
                                        )}
                                    </td>
                                );
                            })}
                        </tr>
                    ))}
                </tbody>
            </table>
        </div>
    );
}
