"use client";

import { useEffect, useMemo, useState, Suspense } from "react";
import dynamic from "next/dynamic";
import Link from "next/link";
import { getBookColumns } from "./columns";
import { getCopyeditingColumns } from "./columnsCE";
import { FiBookOpen, FiCheckCircle, FiFileText, FiCalendar, FiEdit } from 'react-icons/fi';
import { motion } from 'framer-motion';
import Select from "react-select";
import selectStyles from "@/app/components/selectStyles";

import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  BarElement,
  ArcElement,
  Title,
  Tooltip,
  Legend,
} from 'chart.js';
import { Bar } from "react-chartjs-2";
import CustomDataTable from "../dataTable/customDataTable";

ChartJS.register(
  CategoryScale,
  LinearScale,
  BarElement,
  ArcElement,
  Title,
  Tooltip,
  Legend
);

const Loader_component = () => {
return <div className="max-w-full px-2 py-2 mx-auto rounded-lg space-y-4 animate-pulse">
      {/* Header */}
      {/* <header>
        <div className="flex justify-between items-center border-b border-purple-500 pb-2">
          <div className="mb-1 space-y-2">
            <div className="h-6 bg-gray-300 rounded w-40"></div>
            <div className="h-4 bg-gray-200 rounded w-64"></div>
          </div>
          <div className="h-8 w-28 bg-gradient-to-r from-gray-200 via-gray-300 to-gray-200 rounded-md"></div>
        </div>
      </header> */}

      {/* Filters */}
      {/* <div className="bg-purple-50 border border-purple-200 rounded-lg p-3 shadow-sm">
        <div className="flex flex-wrap gap-4 items-center justify-start">
        
          <div className="flex flex-col space-y-1">
            <div className="h-3 bg-gray-200 rounded w-10"></div>
            <div className="h-8 bg-gray-200 rounded-md w-40"></div>
          </div>

        
          <div className="pt-5">
            <div className="h-8 w-24 bg-gray-200 rounded-md"></div>
          </div>
        </div>
      </div> */}

      {/* Table Shimmer */}
      <div className="w-full mt-4 border border-gray-200 rounded-lg overflow-hidden">
        {/* Table Header */}
        <div className="bg-gray-100 flex items-center border-b border-gray-200 px-4 py-3">
          {[...Array(5)].map((_, i) => (
            <div
              key={i}
              className="h-4 bg-gray-300 rounded mx-2"
              style={{ width: `${12 + i * 5}%` }}
            ></div>
          ))}
        </div>

        {/* Table Body */}
        <div className="divide-y divide-gray-100">
          {[...Array(6)].map((_, i) => (
            <div
              key={i}
              className="flex items-center px-4 py-3 bg-white"
            >
              {[...Array(5)].map((_, j) => (
                <div
                  key={j}
                  className="h-4 bg-gray-200 rounded mx-2"
                  style={{ width: `${12 + j * 5}%` }}
                ></div>
              ))}
            </div>
          ))}
        </div>
      </div>
    </div>
};
export default function BooksList({ initialBooks, initialCopyediting, role }) {
  const [data, setData] = useState(initialBooks || []);
  const [copyeditingData, setCopyeditingData] = useState(initialCopyediting || []);
  const [loading, setLoading] = useState(true);
  const [filteredData, setFilteredData] = useState([]);
  const [filteredCopyeditingData, setFilteredCopyeditingData] = useState([]);
  const [yearOptions, setYearOptions] = useState([]);
  const [activeTab, setActiveTab] = useState('yearWise');
  const [selectedYear, setSelectedYear] = useState(null);

  // console.log(filteredData, "filteredData")

  useEffect(() => {
    setLoading(true);
    const timer = setTimeout(() => {
      setData(initialBooks || []);
      setFilteredData(initialBooks || []);
      setLoading(false);
    }, 800); // delay for smooth shimmer view
    return () => clearTimeout(timer);
  }, [initialBooks]);

  useEffect(() => {
    const timer = setTimeout(() => {
      setCopyeditingData(initialCopyediting || []);
      setFilteredCopyeditingData(initialCopyediting || []);
    }, 800); // delay for smooth shimmer view
    return () => clearTimeout(timer);
  }, [initialCopyediting]);

  // useEffect(() => {
  //   const currentYear = new Date().getFullYear();
  //   const years = [];

  //   for (let i = 0; i < 10; i++) {
  //     const year = currentYear + i;
  //     years.push({ value: year.toString(), label: `${year}_Working Title` });
  //   }

  //   setYearOptions(years);
  //   setSelectedYear({ value: currentYear.toString(), label: `${currentYear}_Working Title` });
  // }, []);

  // useEffect(() => {
  //   const filtered = data.filter((item) => {
  //     const itemDate = new Date(item.book_createdAt).getFullYear().toString();
  //     const matchesYear = selectedYear ? itemDate === selectedYear.value : true;

  //     return (
  //       matchesYear
  //     );
  //   });
  //   setFilteredData(filtered);
  // }, [selectedYear, data]);


  useEffect(() => {
    const filtered = data.filter((item) => {
      const matchesBookYear = selectedYear
        ? item.book_year?.trim() === selectedYear.value
        : true;
      return matchesBookYear;
    });
    setFilteredData(filtered);
  }, [selectedYear, data]);

  useEffect(() => {
    const filtered = copyeditingData.filter((item) => {
      const matchesBookYear = selectedYear
        ? item.book?.book_year?.trim() === selectedYear.value
        : true;
      return matchesBookYear;
    });
    setFilteredCopyeditingData(filtered);
  }, [selectedYear, copyeditingData]);


  const columns = useMemo(() => getBookColumns(), []);
  const copyeditingColumns = useMemo(() => getCopyeditingColumns(), []);
  // console.log('copyeditingColumns in BookList:', copyeditingColumns);
  // const selectStyles = {
  //   control: (base) => ({ ...base, minWidth: 220 }),
  //   menuPortal: (base) => ({ ...base, zIndex: 9999 }),
  // };

  // Chart Data Preparation

  // 

 
const getStatusLabel = (status) => {
  const statusMap = {
    label: "Sent",
    label: "Accepted",
    label: "Rejected",
    label: "Timeout",
    label: "Received",
  }
  console.log('status in getStatusLabel:', status);
  return statusMap[status]?.label || 'Unknown';
};

const copyeditingStatusCounts = useMemo(() => {
  return filteredCopyeditingData.reduce((acc, item) => {
    console.log('item in copyeditingStatusCounts:', item);
    const status = item.ce_editing?.[0]?.ce_status;
    if (status) {
      const label = getStatusLabel(status);
      acc[label] = (acc[label] || 0) + 1;
    }
    return acc;
  }, {});
}, [filteredCopyeditingData]);


return (
  <div className="relative max-w-full px-1 py-1 mx-auto space-y-4 h-full">
    {/* New Header Section */}
    <header className="relative z-10 bg-white rounded-2xl shadow-lg p-2 overflow-hidden border border-gray-200/50">
      {/* Header Layout */}
      <div className="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-6">

        {/* Left Section - Heading */}
        <motion.h2
          initial={{ opacity: 0, x: -20 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.5 }}
          className="text-xl font-bold text-gray-800 w-full lg:w-auto"
        >
          Dashboard
          {/* <p className="text-gray-500 mt-1 text-sm lg:text-base font-medium">
            Welcome back  Here's your book data overview.
          </p> */}
          <p className="text-gray-500 text-sm lg:text-base font-medium">Welcome back!</p>
          <p className="text-gray-500 text-sm lg:text-base font-medium">Here's your book data</p>

        </motion.h2>

        {/* Center Section - Stat Cards */}
        <motion.div
          initial="hidden"
          animate="visible"
          variants={{ visible: { transition: { staggerChildren: 0.2, delayChildren: 0.2 } } }}
          className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-2 gap-4 w-full max-w-2xl"
        >
          <StatCard
            icon={
              <motion.div
                animate={{ rotateY: [0, 15, 0] }}
                transition={{
                  repeat: Infinity,
                  repeatType: "reverse",
                  duration: 3,
                  ease: "easeInOut",
                }}
                className="inline-block"
                style={{ transformOrigin: "center" }}
              >
                <FiBookOpen size={28} className="text-blue-500" />
              </motion.div>
            }
            label="Total Books"
            value={filteredData.length}
            color="blue"
            href="/books/list" 
          />

          <StatCard
            icon={
              <motion.div
                animate={{ rotateZ: [0, -5, 5, 0] }}
                transition={{
                  repeat: Infinity,
                  duration: 4,
                  ease: "easeInOut",
                }}
                className="inline-block"
              >
                <FiFileText size={28} className="text-green-500" />
              </motion.div>
            }
            label="Total Chapters"
            value={filteredData.reduce(
              (acc, book) => acc + (book.chapters?.length || 0),
              0
            )}
            color="green"
            href="/chapters/list"
          />

        </motion.div>

        {/* Right Section - Animated Icons */}
        <motion.div
          initial={{ opacity: 0, x: 20 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.6, delay: 0.3 }}
          className="hidden lg:flex items-center space-x-4"
        >
          {[
            { Icon: FiBookOpen, colors: "from-purple-500 to-indigo-600" },
            { Icon: FiCheckCircle, colors: "from-green-500 to-teal-600" },
            { Icon: FiFileText, colors: "from-pink-500 to-rose-600" },
          ].map(({ Icon, colors }, i) => (
            <motion.div
              key={i}
              className={`w-11 h-11 rounded-full bg-gradient-to-br ${colors} flex items-center justify-center text-white shadow-md border-2 border-white`}
              animate={{ y: [0, -4, 0] }}
              transition={{
                repeat: Infinity,
                duration: 2.5 + i,
                ease: "easeInOut",
              }}
              whileHover={{ scale: 1.2, rotate: 15, y: -5 }}
            >
              <Icon size={20} />
            </motion.div>
          ))}
        </motion.div>
      </div>

      {/* Gradient Glows */}
      <div className="absolute top-0 right-0 h-24 w-64 bg-gradient-to-l from-purple-500/20 to-transparent -z-10 blur-2xl"></div>
      <div className="absolute bottom-0 left-0 h-24 w-64 bg-gradient-to-r from-pink-500/20 to-transparent -z-10 blur-2xl"></div>
    </header>


    {/* Tabs UI */}
    <div className="bg-white/30 backdrop-blur-sm rounded-xl shadow-lg border border-gray-200/70 overflow-hidden">
      <div className="relative border-b border-gray-200/80 px-6">

        <div className="absolute inset-y-0 left-0 w-1/4 bg-gradient-to-r from-sky-400/20 to-transparent -z-10 blur-2xl"></div>
        <div className="absolute inset-y-0 right-0 w-1/4 bg-gradient-to-l from-sky-400/20 to-transparent -z-10 blur-2xl"></div>


        <nav className="relative flex space-x-1" aria-label="Tabs">
          <button
            onClick={() => setActiveTab('yearWise')}
            className={`relative whitespace-nowrap py-4 px-4 font-semibold text-sm transition-colors focus:outline-none flex items-center gap-2 ${activeTab === 'yearWise' ? 'text-[#8C3FE1]' : 'text-gray-500 hover:text-gray-700 cursor-pointer'
              }`}
          >
            <FiCalendar />
            <span className={activeTab === 'yearWise' ? 'bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] text-transparent bg-clip-text' : ''}>Year Wise</span>
            {activeTab === 'yearWise' && (
              <motion.div layoutId="active-tab-underline" className="absolute bottom-0 left-0 right-0 h-0.5 bg-gradient-to-r from-[#8C3FE1] to-[#E557AE]" />
            )}
          </button>
          {role !== 3 && (
            <button
              onClick={() => setActiveTab('copyediting')}
              className={`relative whitespace-nowrap py-4 px-4 font-semibold text-sm transition-colors focus:outline-none flex items-center gap-2 ${activeTab === 'copyediting' ? 'text-[#8C3FE1]' : 'text-gray-500 hover:text-gray-700 cursor-pointer'
                }`}
            >
              <FiEdit />
              <span className={activeTab === 'copyediting' ? 'bg-gradient-to-r from-[#8C3FE1] to-[#E557AE] text-transparent bg-clip-text' : ''}>Copyediting</span>
              {activeTab === 'copyediting' && (
                <motion.div layoutId="active-tab-underline" className="absolute bottom-0 left-0 right-0 h-0.5 bg-gradient-to-r from-[#8C3FE1] to-[#E557AE]" />
              )}
            </button>
          )}
        </nav>
      </div>

      {activeTab === 'yearWise' && (
          <Suspense fallback={<Loader_component />}>
            {loading ? (
              <Loader_component />
            ) : (
            <section>
              <div className="px-10 py-2 flex flex-col sm:flex-row justify-between items-center gap-3">
                <h2 className="text-xl font-semibold text-gray-800">&nbsp;</h2>
                <div className="flex items-center space-x-3">
                  <Select
                    instanceId="book-year-filter"
                    id="book-year-filter"
                    options={Array.from(
                      new Set(data.map((item) => item.book_year?.trim()))
                    )
                      .filter(Boolean)
                      .map((year) => ({ value: year, label: year }))}
                    value={
                      selectedYear
                        ? { value: selectedYear.value, label: selectedYear.label }
                        : null
                    }
                    onChange={setSelectedYear}
                    placeholder="Select Book Year"
                    isClearable
                    className="w-60 text-sm"
                    menuPortalTarget={document.body}
                    styles={selectStyles}
                  />
                </div>
              </div>
              <CustomDataTable
                columns={columns}
                data={filteredData}
                  progressPending={false}
                defaultSort={{ key: 'Working Title', direction: 'ascending' }}
              />
            </section>
            )}
          </Suspense>
      )}

      {activeTab === 'copyediting' && (
          <Suspense fallback={<Loader_component />}>
            {loading ? (
              <Loader_component />
            ) : (
            <section>
              <div className="px-10 py-2 flex flex-col sm:flex-row justify-between items-center gap-3">
                <h2 className="text-xl font-semibold text-gray-800">&nbsp;</h2>
                <div className="flex items-center space-x-3">
                  <Select
                    instanceId="book-year-filter"
                    id="book-year-filter"
                    options={Array.from(
                      new Set([
                        ...data.map((item) => item.book_year?.trim()),
                        ...copyeditingData.map((item) => item.book?.book_year?.trim())
                      ])
                    )
                      .filter(Boolean)
                      .map((year) => ({ value: year, label: year }))}
                    value={
                      selectedYear
                        ? { value: selectedYear.value, label: selectedYear.label }
                        : null
                    }
                    onChange={setSelectedYear}
                    placeholder="Select Book Year"
                    isClearable
                    className="w-60 text-sm"
                    menuPortalTarget={document.body}
                    styles={selectStyles}
                  />
                </div>
              </div>
              <CustomDataTable
                columns={copyeditingColumns}
                data={filteredCopyeditingData}
                  progressPending={false}
                defaultSort={{ key: 'Working Title', direction: 'ascending' }}
              />
            </section>
            )}
          </Suspense>
      )}

      {/* Charts Section */}
        {/* Charts Section */}
      {filteredData.length > 0 && (
        <div className="grid grid-cols-1 lg:grid-cols-5 gap-6 p-4">
 
          <motion.div
            className="lg:col-span-3 bg-white p-4 rounded-xl shadow-lg border border-gray-200/70"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true, amount: 0.2 }}
            transition={{ duration: 0.6, ease: "easeOut" }}
          >
            <h3 className="text-lg font-semibold text-gray-800 mb-4 text-center">
              Chapter Submission vs Acceptance
            </h3>
            <div className="relative h-80">
              <Bar
                options={{
                  responsive: true,
                  maintainAspectRatio: false,
                  animation: filteredData.length > 100 ? false : { duration: 1000 },
                  plugins: {
                    legend: { position: "bottom" },
                    title: {
                      display: true,
                      text: "Chapter Progress per Book",
                      font: { size: 16, weight: "bold" },
                    },
                    decimation: {
                      enabled: true,
                      algorithm: "min-max",
                    },
                  },
                  scales: {
                    x: {
                      ticks: {
                        callback: function (val, index) {
                          const label = this.getLabelForValue(val);
                          return label.length > 15
                            ? label.substring(0, 15) + "..."
                            : label;
                        },
                        maxRotation: 0,
                        autoSkip: true,
                        maxTicksLimit: 10,
                      },
                    },
                    y: {
                      beginAtZero: true,
                      ticks: { stepSize: 1 },
                    },
                  },
                }}
                data={{
                  labels: filteredData.map((book) =>
                    book.book_title.length > 20
                      ? book.book_title.substring(0, 20) + "..."
                      : book.book_title
                  ),
                  datasets: [
                    {
                      label: "Chapters Submitted",
                      data: filteredData.map((book) => book.book_ChSubmitted),
                      backgroundColor: "rgba(54, 162, 235, 0.6)",
                      borderColor: "rgba(54, 162, 235, 1)",
                      borderWidth: 1,
                      borderRadius: 4,
                    },
                    {
                      label: "Chapters Accepted",
                      data: filteredData.map((book) => book.book_ChAccepted),
                      backgroundColor: "rgba(75, 192, 192, 0.6)",
                      borderColor: "rgba(75, 192, 192, 1)",
                      borderWidth: 1,
                      borderRadius: 4,
                    },
                  ],
                }}
              />
            </div>
          </motion.div>
 
 
          {/* <motion.div
            className="lg:col-span-2 bg-white p-4 rounded-xl shadow-lg border border-gray-200/70"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true, amount: 0.2 }}
            transition={{ duration: 0.6, ease: "easeOut", delay: 0.2 }}
          >
            <h3 className="text-lg font-semibold text-gray-800 mb-4 text-center">
              Copyediting Status
            </h3>
            <div className="relative h-80">
              {Object.keys(copyeditingStatusCounts).length > 0 ? (
                <motion.div
                  initial={{ scale: 0.8, opacity: 0 }}
                  animate={{ scale: 1, opacity: 1 }}
                  transition={{ delay: 0.4, duration: 0.5 }}
                >
                  <Doughnut
                    options={{
                      responsive: true,
                      maintainAspectRatio: false,
                      plugins: {
                        legend: { position: "right" },
                        title: {
                          display: true,
                          text: "Copyediting Status Distribution",
                          font: { size: 16, weight: "bold" },
                        },
                      },
                    }}
                    data={{
                      labels: Object.keys(copyeditingStatusCounts),
                      datasets: [
                        {
                          label: "Chapters",
                          data: Object.values(copyeditingStatusCounts),
                          backgroundColor: [
                            "rgba(54, 162, 235, 0.7)", // Sent
                            "rgba(75, 192, 192, 0.7)", // Accepted
                            "rgba(255, 99, 132, 0.7)", // Rejected
                            "rgba(255, 206, 86, 0.7)", // Timeout
                            "rgba(153, 102, 255, 0.7)", // Uploaded
                          ],
                          borderColor: [
                            "rgba(54, 162, 235, 1)",
                            "rgba(75, 192, 192, 1)",
                            "rgba(255, 99, 132, 1)",
                            "rgba(255, 206, 86, 1)",
                            "rgba(153, 102, 255, 1)",
                          ],
                          borderWidth: 1,
                        },
                      ],
                    }}
                  />
                </motion.div>
              ) : (
                <div className="flex items-center justify-center h-full text-gray-500">
                  No copyediting data available.
                </div>
              )}
            </div>
          </motion.div> */}
        </div>
      )}

    </div>
  </div>
);
}

function StatCard({ icon, label, value, color,href  }) {
  const colors = {
    blue: 'from-blue-100 to-blue-200 border-blue-300',
    green: 'from-green-100 to-green-200 border-green-300',
    purple: 'from-purple-100 to-purple-200 border-purple-300',
    red: 'from-red-100 to-red-200 border-red-300',
  };

  return (
    <Link href={href} className="block">
      <motion.div
        className={`bg-gradient-to-br ${colors[color]} 
            p-3 rounded-xl shadow-md border flex items-center gap-4 
            transition-all duration-300 cursor-pointer`}
        
        // ⭐ Restore hover animations
        whileHover={{ scale: 1.03 }}
        whileTap={{ scale: 0.97 }}
      >
        {/* Icon */}
        <motion.div
          className="p-2 bg-white/70 rounded-full shadow-inner flex items-center justify-center"
        >
          {icon}
        </motion.div>

        {/* Text */}
        <div>
          <motion.p className="text-2xl font-bold text-gray-800">
            {value}
          </motion.p>
          <motion.p className="text-xs text-gray-600 font-medium">
            {label}
          </motion.p>
        </div>
      </motion.div>
    </Link>
  );
}

