"use client";

import { useState, useEffect } from "react";
import Swal from "sweetalert2";
import { toast } from "react-toastify";

// --- START: HYPOTHETICAL NEW DEDICATED ACTIONS ---
import { 
    acceptCEAction, 
    rejectCEAction, 
    uploadCEAction 
} from "@/lib/actions/copyeditingAction";
// --- END: HYPOTHETICAL NEW DEDICATED ACTIONS ---


export default function CopyEditingList({ ceEditiingDetails }) {
  const [loading, setLoading] = useState(true);
  const [ceData, setCEData] = useState(ceEditiingDetails || {});
  const [acceptanceMessage, setAcceptanceMessage] = useState(null); 
  const [isProcessing, setIsProcessing] = useState(false);
  const [fileToUpload, setFileToUpload] = useState(null);
  const [ceReceivedNote, setCeReceivedNote] = useState(null);
  const [justUploaded, setJustUploaded] = useState(false);
  // *** NEW STATE ADDED ***
  const [justRejected, setJustRejected] = useState(false);

  useEffect(() => {
    setCEData(ceEditiingDetails || {});
    setLoading(false);
    // Reset temporary flags on component mount/data change
    setJustUploaded(false);
    // *** NEW STATE RESET ***
    setJustRejected(false); 
  }, [ceEditiingDetails]);
  
  // Destructure for cleaner access
  const ce_pid = ceData?.ce_pid;
  const bookName = ceData?.book?.book_title;
  const chapterName = ceData?.chapter?.chap_title;
  const authorName = ceData?.user?.user_name;
  const currentStatus = ceData.ce_status; // 1: Sent, 2: Acc, 3: Rej, 4: Timeout, 5: Uploaded

  console.log("CE Status", currentStatus);

  
  /**
   * Helper function to get current date/time string
   */
  function getCurrentDateTime() {
    const now = new Date();
    const month = String(now.getMonth() + 1).padStart(2, '0');
    const day = String(now.getDate()).padStart(2, '0');
    const year = now.getFullYear();
    const hours = String(now.getHours()).padStart(2, '0');
    const minutes = String(now.getMinutes()).padStart(2, '0');
    const seconds = String(now.getSeconds()).padStart(2, '0');

    return `${month}-${day}-${year} ${hours}:${minutes}:${seconds}`;
  }

  /**
   * Unified Confirmation Handler (Simplified)
   */
  const handleConfirmAction = async (newStatus, successMsg, confirmText, title, actionFunction) => {
    if (isProcessing) return;

    // 1. Ask for confirmation
    const result = await Swal.fire({
      title: title || "Confirm Action",
      text: `Do you want to ${confirmText.toLowerCase()} the paper "${bookName}"?`,
      icon: "question",
      showCancelButton: true,
      confirmButtonColor: newStatus === 2 ? "#3085d6" : "#d33", 
      cancelButtonColor: "#aaa",
      confirmButtonText: confirmText,
      cancelButtonText: "No, Cancel",
      
    });

    if (!result.isConfirmed) return;

    setIsProcessing(true);
    setAcceptanceMessage(null); 
    // Reset rejection flag before starting a new action
    setJustRejected(false);
    
    // 2. Execute the dedicated action function passed in
    try {
        await actionFunction();
    } catch (error) {
        toast.error(error.message || `Error during ${confirmText.toLowerCase()}`);
    } finally {
        setIsProcessing(false);
    }
  };

  /**
   * Handler for Acceptance (Status 2) - DEDICATED LOGIC (Unchanged)
   */
  const handleAcceptance = () => {
    const newStatus = 2;
    const successMsg = "Thank you for your acceptance for Editing. You will receive an Acknowledgement mail. Happy editing!";
    const confirmText = "Yes, Accept";
    const title = "Confirm Acceptance";
    
    const actionFunction = async () => {
        const payload = {
            ce_accept_date: getCurrentDateTime(),
            ce_status: newStatus,
        };

        const res = await acceptCEAction(ce_pid, payload); 

        if (res?.success) {
            setCEData(prev => ({ ...prev, ce_status: newStatus }));
            setAcceptanceMessage(successMsg);
            toast.success(`${confirmText} successful!`);
        } else {
            toast.error(res?.message || `Failed to ${confirmText.toLowerCase()} paper.`);
        }
    };
    
    handleConfirmAction(newStatus, successMsg, confirmText, title, actionFunction);
  };

  /**
   * Handler for Rejection (Status 3) - DEDICATED LOGIC (MODIFIED)
   */
  const handleReject = () => {
    const newStatus = 3;
    const successMsg = "You rejected to edit this file. The status has been updated.";
    const confirmText = "Yes, Reject";
    const title = "Confirm Rejection";

    const actionFunction = async () => {
        const payload = {
            ce_reject_date: getCurrentDateTime(),
            ce_status: newStatus,
        };

        const res = await rejectCEAction(ce_pid, payload); 

        if (res?.success) {
            setCEData(prev => ({ ...prev, ce_status: newStatus }));
            setAcceptanceMessage(successMsg);
            toast.success(`${confirmText} successful!`);
            // *** MODIFICATION 2: Set flag to hide permanent rejection status block ***
            setJustRejected(true); 
        } else {
            toast.error(res?.message || `Failed to ${confirmText.toLowerCase()} paper.`);
        }
    };

    handleConfirmAction(newStatus, successMsg, confirmText, title, actionFunction);
  };

  /**
   * Handler to close the current tab. (Unchanged)
   */
  const handleClose = () => {
    window.close();
    setTimeout(() => {
      toast.info("Please manually close this browser tab.");
    }, 50);
  };

  /**
   * Handler for File Upload (Status 5) - DEDICATED LOGIC (Unchanged, assuming previous upload fix is fine)
   */
  const handleUpload = async (e) => {
    e.preventDefault();
    if (!fileToUpload) {
      toast.error("Please select a file to upload.");
      return;
    }
    
    setIsProcessing(true);
    setAcceptanceMessage(null); 
    setJustRejected(false); // Ensure rejection message is cleared/not interfering
    
    const newStatus = 5;

    try {
        // 1. Prepare data (FormData is required for file uploads with Server Actions)
        const formData = new FormData();
        formData.append('ce_received_file', fileToUpload);
        formData.append('ce_received_date', getCurrentDateTime());
        formData.append('ce_received_note', ceReceivedNote);
        formData.append('ce_status', newStatus);

        // 2. Perform the upload and status update in one dedicated action
        const res = await uploadCEAction(ce_pid, formData); 

        if (res?.success) {
            // 3. Set success message and update state
            setCEData(prev => ({ ...prev, ce_status: newStatus }));
            
            // Set the immediate success message
            setAcceptanceMessage("Thank you for uploading the edited paper! Your submission is complete.");
            toast.success("File uploaded and status updated!");
            
            // Set flag to hide permanent status block
            setJustUploaded(true);
            
        } else {
            toast.error(res?.message || "Failed to finalize upload status.");
        }
    } catch (error) {
        toast.error("Error during file submission.");
    } finally {
        setIsProcessing(false);
    }
  };
  
  const renderMainContent = () => {
    if (loading) {
      return <p className="text-center text-sm text-gray-500 py-4">Loading details...</p>; 
    }
    
    // UI for Status 4: Timeout (Unchanged)
    if (currentStatus === 4) {
        return (
            <div className="bg-yellow-100 border-l-4 border-yellow-500 text-yellow-700 p-3 rounded-md shadow-sm m-0">
            <p className="font-bold text-sm mb-0.5 mt-0">Action Timed Out</p>
            <p className="text-sm m-0 leading-tight">
                You are not allowed to perform this action at this time.
                This assignment has <span className="font-semibold">timed out</span>. Please contact the administrator.
            </p>
            </div>
        );
    }

    // Message for Already Rejected (Status 3) (MODIFIED)
    if (currentStatus === 3) {
        // *** MODIFICATION 3: Hide Status 3 block if it was just rejected in this session ***
        if (justRejected) {
            // If just rejected, we rely ONLY on the green acceptanceMessage banner 
            // and skip the permanent status block.
            return null; 
        }

        return (
            <div className="bg-red-100 border-l-4 border-red-500 text-red-700 p-3 rounded-md shadow-sm m-0">
            <p className="font-bold text-sm mb-0.5 mt-0">Paper Rejected</p>
            <p className="text-sm m-0 leading-tight">
                You already <span className="font-semibold">rejected</span> this paper for copy editing.
                No further action is required.
            </p>
            </div>
        );
    } 
    
    // UI for Status 5: Uploaded (Protected by justUploaded flag - as per previous fix)
    if (currentStatus === 5) 
    {
        if (justUploaded) {
            return null; 
        }
        return (
            <div className="bg-blue-100 border-l-4 border-blue-500 text-blue-700 p-3 rounded-md shadow-sm m-0">
            <p className="font-bold text-sm mb-0.5 mt-0">Upload Complete</p>
            <p className="text-sm m-0 leading-tight">
                You already <span className="font-semibold">uploaded</span> the edited paper.
                Thank you for your work!
            </p>
            <button
                onClick={handleClose}
                className="mt-2 px-3 py-1.5 text-sm font-semibold text-gray-700 bg-gray-200 rounded-md hover:bg-gray-300 transition duration-150"
            >
                Close
            </button>
            </div>
        );
    }

    // UI for Acceptance/Rejection (Status 1: Sent/Pending) 
    if (currentStatus === 1) {
        // ... (Status 1 UI - Unchanged)
        return (
            <div className="bg-white shadow-md rounded-md p-3 border border-gray-200 m-0">
            {/* Paper Details */}
            <h2 className="text-md font-bold text-gray-700 mb-1 mt-0 p-0">
                Paper Details:
            </h2>

            <div className="text-sm leading-tight">
                <p className="m-0 mb-0.5">
                <span className="font-semibold text-gray-600">Book Name:</span>{" "}
                <span className="text-gray-900">{bookName || "N/A"}</span>
                </p>
                <p className="m-0 mb-0.5">
                <span className="font-semibold text-gray-600">Chapter Name:</span>{" "}
                <span className="text-gray-900">{chapterName || "N/A"}</span>
                </p>
                <p className="m-0">
                <span className="font-semibold text-gray-600">Author Name:</span>{" "}
                <span className="text-gray-900">{authorName || "N/A"}</span>
                </p>
            </div>

            {/* Divider */}
            <hr className="my-3 border-gray-200" />

            {/* Decision Prompt */}
            <p className="text-md font-bold text-gray-800 mb-2 mt-0">
                Do you want to accept or reject this paper for Copy Editing?
            </p>

            {/* Action Buttons */}
            <div className="flex gap-2">
                <button
                onClick={handleAcceptance}
                disabled={isProcessing}
                className="px-3 py-1.5 text-sm font-semibold text-white bg-green-600 rounded-md hover:bg-green-700 transition duration-150 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                {isProcessing ? "Processing..." : "Accept"}
                </button>
                <button
                onClick={handleReject}
                disabled={isProcessing}
                className="px-3 py-1.5 text-sm font-semibold text-white bg-red-600 rounded-md hover:bg-red-700 transition duration-150 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                {isProcessing ? "Processing..." : "Reject"}
                </button>
                <button
                onClick={handleClose}
                disabled={isProcessing}
                className="px-3 py-1.5 text-sm font-semibold text-gray-700 bg-gray-200 rounded-md hover:bg-gray-300 transition duration-150 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                Close
                </button>
            </div>
            </div>
        );
    }

    
    // UI for File Upload (Status 2: Accepted) 
    if (currentStatus === 2) {
        // ... (Status 2 UI - Unchanged)
        return (
            <div className="bg-white shadow-md rounded-md p-4 border border-blue-100 m-0">
            {/* Header */}
            <h2 className="text-lg font-bold text-blue-700 mb-2 mt-0 p-0">
                CE Upload
            </h2>

            {/* Book & Chapter Info */}
            <div className="pb-2 mb-2 border-b border-gray-100">
                <p className="text-sm m-0 mb-0.5 leading-tight">
                    <span className="font-semibold text-gray-600">Book Name:</span>{" "}
                    <span className="text-gray-800">{bookName || "N/A"}</span>
                </p>
                <p className="text-sm m-0 leading-tight">
                    <span className="font-semibold text-gray-600">Chapter Title:</span>{" "}
                    <span className="text-gray-800">{chapterName || "N/A"}</span>
                </p>
            </div>

            {/* Upload Form */}
            <form onSubmit={handleUpload} className="space-y-4 mt-3">
                {/* CE Received Note */}
                <div>
                    <label
                        htmlFor="ce_received_note"
                        className="block text-sm font-medium text-gray-700 mb-1"
                    >
                        CE Received Note
                    </label>
                    <textarea
                        id="ce_received_note"
                        name="ce_received_note"
                        rows="3"
                        placeholder="Enter any notes or comments here..."
                        value={ceReceivedNote ?? ''}
                        onChange={(e) => setCeReceivedNote(e.target.value)}
                        className="w-full border border-gray-300 rounded-md p-2 text-sm text-gray-800 bg-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500"

                        disabled={isProcessing}
                    />
                </div>

                {/* Drop Zone */}
                <div className="border-2 border-dashed border-gray-300 rounded-md p-5 text-center hover:border-blue-500 transition duration-200 cursor-pointer relative">
                    <input
                        id="upload-file"
                        type="file"
                          accept=".pdf,.docx,application/pdf,application/vnd.openxmlformats-officedocument.wordprocessingml.document"
                        required
                        onChange={(e) => setFileToUpload(e.target.files[0])}
                        className="absolute inset-0 w-full h-full opacity-0 cursor-pointer"
                        disabled={isProcessing}
                    />

                    {fileToUpload ? (
                        <div>
                            <p className="text-green-600 font-semibold text-sm mb-1">
                                File Selected: {fileToUpload.name}
                            </p>
                            <div className="flex justify-center gap-2">
                                <p className="text-xs text-gray-500 mt-0.5">
                                    Click 'Submit' to finalize the upload.
                                </p>
                                <button
                                    type="button"
                                    onClick={() => setFileToUpload(null)}
                                    className="text-xs text-red-500 font-medium hover:underline"
                                    disabled={isProcessing}
                                >
                                    REMOVE
                                </button>
                            </div>
                        </div>
                    ) : (
                        <div>
                            <p className="text-sm text-gray-700 font-medium mb-0">
                                Choose File
                            </p>
                            <p className="text-xs text-gray-500 mt-0.5">
                                Drag and drop or click to select the copy-edited file (DOCX, PDF).
                            </p>
                        </div>
                    )}
                </div>

                {/* Hidden Label for Accessibility */}
                <label htmlFor="upload-file" className="sr-only">
                    Upload File
                </label>

                {/* Submit Button */}
                <button
                    type="submit"
                    disabled={isProcessing || !fileToUpload}
                    className="w-full px-4 py-2 text-sm font-semibold text-white bg-blue-600 rounded-md hover:bg-blue-700 transition duration-150 disabled:opacity-50 disabled:cursor-not-allowed shadow-sm"
                >
                    {isProcessing ? "Submitting..." : "Submit Edited Files"}
                </button>
            </form>
            </div>
        );
    }


        // Default or unhandled status fallback
        return (
        <div className="bg-gray-100 border-l-4 border-gray-400 text-gray-700 p-3 rounded-md shadow-sm m-0">
            <p className="font-bold text-sm mb-0.5 mt-0">Assignment Status</p>
            <p className="text-sm m-0 leading-tight">
            The status of this copy editing assignment is currently unhandled. Please
            check with the administrator.
            </p>
        </div>
        );
    };


  
  return (
    <div className="max-w-4xl px-4 py-0 mx-auto"> 
      <header className="border-b m-0 p-0">
        <h1 className="text-xl font-extrabold text-gray-900 m-0 p-0">
            Copy Editing Action Panel
        </h1>
    </header>

      {/* Conditional Display of Acceptance/Rejection/Upload Success Message */}
      {acceptanceMessage && (
        <div className="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md shadow-sm animate-fadeIn" role="alert"> 
          <p className="font-bold text-sm mb-1">Success! </p> 
          <p className="text-sm">
            **{acceptanceMessage}**
          </p>
        </div>
      )}
      
      {/* *** MODIFICATION 4: Conditional Rendering of Main Content ***
          Hides main content if:
          1. It was just uploaded AND the success message is showing. (Previous fix)
          2. It was just rejected AND the success message is showing. (NEW fix)
      */}
      {!( (justUploaded || justRejected) && acceptanceMessage) && renderMainContent()}
      
      {/* Conditional Close Button (Only for immediate post-action success) */}
      {( (currentStatus === 5 && justUploaded) || (currentStatus === 3 && justRejected) ) && acceptanceMessage && (
          <div className="mt-4 p-0">
              <button
                  onClick={handleClose}
                  className="px-3 py-1.5 text-sm font-semibold text-gray-700 bg-gray-200 rounded-md hover:bg-gray-300 transition duration-150"
              >
                  Close Window
              </button>
          </div>
      )}
    </div>
  );
}